# 🌞 ENDPOINT DE DATOS SOLARES - Guía de Integración

**Fecha:** 29 de octubre de 2025  
**Estado:** ✅ Funcional y probado

---

## 📍 URL del Endpoint

```
GET https://escalar.app/scadmin/backend/public/api/solar/current
```

---

## 🔧 Configuración Requerida

### Headers:
```http
Accept: application/json
```

**Nota:** No requiere autenticación (endpoint público)

---

## 📊 Respuesta del Endpoint

### Estructura completa:

```json
{
  "granja": {
    "id": 1,
    "nombre": "Granja Solar La Gloria",
    "capacidad_mw": 1,
    "capacidad_kw": 1000,
    "produccion_anual_estimada_kwh": 12000000,
    "estado": "operativa",
    "scawatts_activos": 5
  },
  "solar": {
    "irradiance": 279.04,
    "uv_index": 0.01,
    "uv_level": {
      "level": "Muy Bajo",
      "color": "#289500",
      "description": "Sin riesgo"
    },
    "solar_level": {
      "level": "Muy Baja",
      "description": "Radiación mínima"
    }
  },
  "generation": {
    "current_power_kw": 237.18,
    "current_power_mw": 0.237,
    "max_power_kw": 1000,
    "efficiency_percent": 85,
    "capacity_factor": 27.9,
    "estimated_daily_production_kwh": 4250
  },
  "timestamp": "2025-10-29T17:00:02.289942",
  "location": {
    "name": "La Gloria, Cesar, Colombia",
    "latitude": 8.629667,
    "longitude": -73.782861,
    "elevation": 45,
    "timezone": "America/Bogota",
    "municipality": "La Gloria",
    "department": "Cesar",
    "country": "Colombia",
    "plus_code": "J6H8+VV6"
  }
}
```

---

## 💻 IMPLEMENTACIÓN EN FRONTEND

### ❌ NO USAR (endpoints antiguos):
```typescript
// Estos NO existen o están obsoletos:
// https://escalar.app/api-stromglass/...
// https://escalar.app/develop/solar/...
// https://escalar.app/scalar-api/api-stromglass/...
```

### ✅ USAR (endpoint correcto):

#### JavaScript / TypeScript:
```typescript
async function getSolarData() {
  const url = 'https://escalar.app/scadmin/backend/public/api/solar/current';
  
  try {
    const response = await fetch(url, {
      method: 'GET',
      headers: {
        'Accept': 'application/json'
      }
    });

    if (!response.ok) {
      throw new Error(`HTTP error! status: ${response.status}`);
    }

    const data = await response.json();
    
    // Acceder a los datos
    console.log('Irradiancia:', data.solar.irradiance, 'W/m²');
    console.log('Potencia actual:', data.generation.current_power_kw, 'kW');
    console.log('ScaWatts activos:', data.granja.scawatts_activos);
    
    return data;
    
  } catch (error) {
    console.error('❌ Error al obtener datos solares:', error);
    throw error;
  }
}
```

#### Angular Service (solar.service.ts):
```typescript
import { Injectable } from '@angular/core';
import { HttpClient, HttpHeaders } from '@angular/common/http';
import { Observable } from 'rxjs';
import { catchError } from 'rxjs/operators';

interface SolarData {
  granja: {
    id: number;
    nombre: string;
    capacidad_mw: number;
    capacidad_kw: number;
    produccion_anual_estimada_kwh: number;
    estado: string;
    scawatts_activos: number;
  };
  solar: {
    irradiance: number;
    uv_index: number;
    uv_level: {
      level: string;
      color: string;
      description: string;
    };
    solar_level: {
      level: string;
      description: string;
    };
  };
  generation: {
    current_power_kw: number;
    current_power_mw: number;
    max_power_kw: number;
    efficiency_percent: number;
    capacity_factor: number;
    estimated_daily_production_kwh: number;
  };
  timestamp: string;
  location: any;
}

@Injectable({
  providedIn: 'root'
})
export class SolarService {
  private apiUrl = 'https://escalar.app/scadmin/backend/public/api/solar/current';

  constructor(private http: HttpClient) {}

  getCurrentSolarData(): Observable<SolarData> {
    const headers = new HttpHeaders({
      'Accept': 'application/json'
    });

    return this.http.get<SolarData>(this.apiUrl, { headers })
      .pipe(
        catchError(error => {
          console.error('❌ Error al obtener datos solares:', error);
          throw new Error('Error al obtener datos solares');
        })
      );
  }
}
```

#### Uso en componente (home.page.ts):
```typescript
import { Component, OnInit } from '@angular/core';
import { SolarService } from './services/solar.service';

@Component({
  selector: 'app-home',
  templateUrl: './home.page.html',
  styleUrls: ['./home.page.scss']
})
export class HomePage implements OnInit {
  solarData: any;
  loading: boolean = false;

  constructor(private solarService: SolarService) {}

  ngOnInit() {
    this.loadSolarData();
  }

  loadSolarData() {
    this.loading = true;
    
    this.solarService.getCurrentSolarData().subscribe({
      next: (data) => {
        this.solarData = data;
        console.log('✅ Datos solares cargados:', data);
        this.loading = false;
      },
      error: (error) => {
        console.error('❌ Error al cargar datos solares:', error);
        this.loading = false;
      }
    });
  }
}
```

---

## 🧪 PRUEBAS CON CURL

### Prueba básica:
```bash
curl -X GET "https://escalar.app/scadmin/backend/public/api/solar/current" \
  -H "Accept: application/json"
```

### Con formato (jq):
```bash
curl -s "https://escalar.app/scadmin/backend/public/api/solar/current" \
  -H "Accept: application/json" | jq '.'
```

### Extraer solo irradiancia:
```bash
curl -s "https://escalar.app/scadmin/backend/public/api/solar/current" \
  -H "Accept: application/json" | jq '.solar.irradiance'
```

### Extraer solo potencia actual:
```bash
curl -s "https://escalar.app/scadmin/backend/public/api/solar/current" \
  -H "Accept: application/json" | jq '.generation.current_power_kw'
```

---

## 🔍 CAMPOS IMPORTANTES

### Datos de la Granja:
- `granja.scawatts_activos` → Cantidad de ScaWatts activos
- `granja.capacidad_kw` → Capacidad máxima en kW
- `granja.estado` → Estado operativo

### Datos Solares:
- `solar.irradiance` → Radiación solar actual (W/m²)
- `solar.uv_index` → Índice UV actual
- `solar.uv_level.level` → Nivel de UV (Muy Bajo, Bajo, Moderado, Alto, Muy Alto)
- `solar.solar_level.level` → Nivel de radiación solar

### Generación Eléctrica:
- `generation.current_power_kw` → Potencia actual en kW
- `generation.current_power_mw` → Potencia actual en MW
- `generation.capacity_factor` → Factor de capacidad (%)
- `generation.estimated_daily_production_kwh` → Producción diaria estimada

### Metadata:
- `timestamp` → Hora de la última actualización
- `location` → Ubicación de la granja

---

## ⚠️ MANEJO DE ERRORES

### Errores comunes:

#### 1. Error 500 - No hay datos disponibles:
```json
{
  "success": false,
  "error": "Error al obtener datos solares: No se encontraron datos solares"
}
```
**Solución:** El sistema de datos meteorológicos no ha actualizado los archivos. Contactar al equipo de backend.

#### 2. Error 500 - No hay granjas operativas:
```json
{
  "success": false,
  "error": "Error al obtener datos solares: No hay granjas operativas"
}
```
**Solución:** Verificar que existe al menos una granja con estado 'operativa' en la base de datos.

#### 3. Error de red (CORS):
```
Access to fetch at '...' from origin '...' has been blocked by CORS policy
```
**Solución:** Verificar configuración de CORS en el backend Laravel (ya está configurado para permitir todos los orígenes).

---

## 🔄 FRECUENCIA DE ACTUALIZACIÓN

- Los datos se actualizan cada **5 segundos** en tiempo real
- El endpoint lee siempre el archivo más reciente de datos meteorológicos
- Se recomienda hacer polling cada **10-30 segundos** desde el frontend
- Para gráficos históricos, usar intervalos mayores (1-5 minutos)

---

## 📝 NOTAS ADICIONALES

1. **No requiere autenticación:** El endpoint es público
2. **Formato de fecha:** ISO 8601 con timezone de Colombia (UTC-5)
3. **Valores numéricos:** Todos los valores de potencia y radiación son números flotantes
4. **Irradiancia estándar:** 1000 W/m² es la irradiancia de referencia al mediodía
5. **Eficiencia del sistema:** 85% (ya aplicado en los cálculos de potencia)

---

## 🆘 SOPORTE

Si encuentras problemas con el endpoint, reportar:

1. **URL exacta** que estás llamando
2. **Headers** que estás enviando
3. **Respuesta completa** del servidor (incluyendo código HTTP)
4. **Logs del navegador** (consola del desarrollador)
5. **Hora exacta** del error

**Contacto backend:** Equipo de desarrollo Laravel  
**Archivo del controlador:** `/home/scalarapp/public_html/scadmin/backend/app/Http/Controllers/SolarController.php`

---

## ✅ CHECKLIST DE MIGRACIÓN

- [ ] Actualizar URL del endpoint en `solar.service.ts`
- [ ] Cambiar de endpoints antiguos a `/api/solar/current`
- [ ] Verificar que los headers incluyen `Accept: application/json`
- [ ] Actualizar interfaces TypeScript con la estructura correcta
- [ ] Probar endpoint con CURL antes de integrar
- [ ] Implementar manejo de errores apropiado
- [ ] Configurar polling/refresh cada 10-30 segundos
- [ ] Actualizar tests unitarios si existen
- [ ] Probar en desarrollo antes de producción
- [ ] Documentar cambios en el changelog del frontend

---

**Última actualización:** 29/10/2025 17:40 COT
