# 🔄 MIGRACIÓN DE ENDPOINTS - VALORES SCAWATT

**Fecha:** 29 de Octubre de 2025  
**Tipo:** Cambio de endpoints para valores en tiempo real e historial de ScaWatt  
**Afecta a:** Frontend (Ionic/Angular) y App Móvil

---

## 📋 RESUMEN

Los endpoints antiguos de PHP que devolvían valores del ScaWatt han sido **migrados a Laravel** para mejorar la seguridad, mantenibilidad y centralización del código.

### ⚠️ ACCIÓN REQUERIDA

**El equipo de Frontend y App Móvil debe actualizar las URLs de los siguientes endpoints:**

---

## 🔴 ENDPOINTS ANTIGUOS (YA NO FUNCIONAN)

### ❌ 1. Valor actual del ScaWatt
```
https://escalar.app/develop/code-test/api/scawatt/scawatt_actual.php
```
**Estado:** ❌ 404 Not Found (archivo eliminado)

---

#### 2. Endpoint de Historial (últimos N días)

**Antigua URL (ELIMINADA):**
```
https://escalar.app/develop/code-test/api/scawatt/scawatt.php?days=1
```

**Nueva URL (Laravel):**
```
https://escalar.app/scadmin/backend/public/api/scawatt/historial?days=1&limit=10
```

**Parámetros:**
- `days` (opcional): Número de días hacia atrás (1-365). Default: 1
- `limit` (opcional): Limitar cantidad de resultados. Sin límite por defecto

---

## ✅ NUEVOS ENDPOINTS (LARAVEL)

### ✅ 1. Valor actual del ScaWatt (Tiempo Real)

**Nueva URL:**
```
GET https://escalar.app/scadmin/backend/public/api/scawatt/valor-actual
```

**Headers:**
```http
Accept: application/json
```

**Respuesta exitosa (200):**
```json
{
  "registro": {
    "valor": 1.9053,
    "fecha": "29/10/2025 16:21:25"
  }
}
```

**Ejemplo de uso (JavaScript/TypeScript):**
```typescript
async getValorActualScaWatt() {
  const url = 'https://escalar.app/scadmin/backend/public/api/scawatt/valor-actual';
  
  try {
    const response = await fetch(url, {
      method: 'GET',
      headers: {
        'Accept': 'application/json'
      }
    });
    
    const data = await response.json();
    
    // Acceder a los datos
    const valor = data.registro.valor;  // 1.9053
    const fecha = data.registro.fecha;  // "29/10/2025 16:21:25"
    
    return data;
  } catch (error) {
    console.error('Error al obtener valor actual:', error);
    throw error;
  }
}
```

**Ejemplo con Angular HttpClient:**
```typescript
import { HttpClient } from '@angular/common/http';

export class ScawattService {
  private apiUrl = 'https://escalar.app/scadmin/backend/public/api/scawatt';

  constructor(private http: HttpClient) {}

  getValorActual() {
    return this.http.get<{registro: {valor: number, fecha: string}}>
      (`${this.apiUrl}/valor-actual`);
  }
}
```

---

### ✅ 2. Historial de valores del ScaWatt

**Nueva URL:**
```
GET https://escalar.app/scadmin/backend/public/api/scawatt/historial?days={N}
```

**Parámetros:**
- `days` (opcional) - Número de días hacia atrás desde ahora. Default: 1
  - Valores válidos: 1, 3, 5, 10, 20, 50, etc.
  - Rango: 1-365
- `limit` (opcional) - Limitar cantidad de resultados retornados
  - Valores válidos: cualquier número positivo
  - Ejemplo: `limit=10` retorna solo los 10 registros más recientes

**Headers:**
```http
Accept: application/json
```

**Respuesta exitosa (200):**
```json
[
  {
    "valor": "1.7903",
    "fecha": "29/10/2025 16:21:35"
  },
  {
    "valor": "1.6763",
    "fecha": "29/10/2025 16:21:30"
  },
  {
    "valor": "1.9053",
    "fecha": "29/10/2025 16:21:25"
  }
]
```

**Ejemplo de uso (JavaScript/TypeScript):**
```typescript
async getHistorialScaWatt(days: number = 1, limit?: number) {
  let url = `https://escalar.app/scadmin/backend/public/api/scawatt/historial?days=${days}`;
  
  // Agregar limit si se especifica
  if (limit) {
    url += `&limit=${limit}`;
  }
  
  try {
    const response = await fetch(url, {
      method: 'GET',
      headers: {
        'Accept': 'application/json'
      }
    });
    
    const data = await response.json();
    
    // data es un array de registros
    // [{ valor: "1.7903", fecha: "29/10/2025 16:21:35" }, ...]
    
    return data;
  } catch (error) {
    console.error('Error al obtener historial:', error);
    throw error;
  }
}
```

**Ejemplo con Angular HttpClient:**
```typescript
import { HttpClient, HttpParams } from '@angular/common/http';

export class ScawattService {
  private apiUrl = 'https://escalar.app/scadmin/backend/public/api/scawatt';

  constructor(private http: HttpClient) {}

  getHistorial(days: number = 1, limit?: number) {
    let params = new HttpParams().set('days', days.toString());
    
    if (limit) {
      params = params.set('limit', limit.toString());
    }
    
    return this.http.get<Array<{valor: string, fecha: string}>>
      (`${this.apiUrl}/historial`, { params });
  }
}
```

**Ejemplos de uso con limit:**
```typescript
// Obtener últimos 10 registros del día de hoy
getHistorialScaWatt(1, 10);

// Obtener últimos 50 registros de los últimos 7 días
getHistorialScaWatt(7, 50);

// Obtener TODOS los registros del último mes (sin limit)
getHistorialScaWatt(30);
```

---

## 📝 EJEMPLOS COMPLETOS DE MIGRACIÓN

### Antes (código antiguo):
```typescript
// ❌ ENDPOINTS ANTIGUOS - NO USAR
const valorActualUrl = 'https://escalar.app/develop/code-test/api/scawatt/scawatt_actual.php';
const historialUrl = 'https://escalar.app/develop/code-test/api/scawatt/scawatt.php?days=1';

// Estos ya no funcionan (404)
```

### Después (código nuevo):
```typescript
// ✅ NUEVOS ENDPOINTS LARAVEL
const valorActualUrl = 'https://escalar.app/scadmin/backend/public/api/scawatt/valor-actual';
const historialUrl = 'https://escalar.app/scadmin/backend/public/api/scawatt/historial?days=1';

// Estos funcionan correctamente
```

---

## 🔧 CAMBIOS EN EL CÓDIGO

### Frontend Ionic/Angular

**Archivo:** `src/app/services/scawatt.service.ts` (o similar)

**Buscar y reemplazar:**

1. **Valor actual:**
```typescript
// ANTES:
const url = 'https://escalar.app/develop/code-test/api/scawatt/scawatt_actual.php';

// DESPUÉS:
const url = 'https://escalar.app/scadmin/backend/public/api/scawatt/valor-actual';
```

2. **Historial:**
```typescript
// ANTES:
const url = `https://escalar.app/develop/code-test/api/scawatt/scawatt.php?days=${days}`;

// DESPUÉS:
const url = `https://escalar.app/scadmin/backend/public/api/scawatt/historial?days=${days}`;
```

---

## 🧪 PRUEBAS CON CURL

### Probar valor actual:
```bash
curl -X GET "https://escalar.app/scadmin/backend/public/api/scawatt/valor-actual" \
  -H "Accept: application/json"
```

**Respuesta esperada:**
```json
{
  "registro": {
    "valor": 1.9053,
    "fecha": "29/10/2025 16:21:25"
  }
}
```

---

### Probar historial (1 día):
```bash
curl -X GET "https://escalar.app/scadmin/backend/public/api/scawatt/historial?days=1" \
  -H "Accept: application/json"
```

**Respuesta esperada:**
```json
[
  {"valor": "1.7903", "fecha": "29/10/2025 16:21:35"},
  {"valor": "1.6763", "fecha": "29/10/2025 16:21:30"},
  {"valor": "1.9053", "fecha": "29/10/2025 16:21:25"}
]
```

---

### Probar historial con limit (solo 3 registros):
```bash
curl -X GET "https://escalar.app/scadmin/backend/public/api/scawatt/historial?limit=3" \
  -H "Accept: application/json"
```

**Respuesta esperada:**
```json
[
  {"valor": "2.3813", "fecha": "29/10/2025 17:32:10"},
  {"valor": "1.1913", "fecha": "29/10/2025 17:32:05"},
  {"valor": "1.7863", "fecha": "29/10/2025 17:32:00"}
]
```

---

### Probar historial (7 días, limitado a 50):
```bash
curl -X GET "https://escalar.app/scadmin/backend/public/api/scawatt/historial?days=7&limit=50" \
  -H "Accept: application/json"
```

---

## 📊 COMPARACIÓN DE RESPUESTAS

### Formato de respuesta (sin cambios):

| Campo | Tipo | Descripción | Ejemplo |
|-------|------|-------------|---------|
| `valor` | number/string | Valor del ScaWatt en ese momento | `1.9053` o `"1.9053"` |
| `fecha` | string | Fecha y hora en formato dd/mm/yyyy HH:mm:ss | `"29/10/2025 16:21:25"` |

**Nota:** El formato de los datos **NO cambia**, solo la URL del endpoint.

---

## ⚙️ DETALLES TÉCNICOS

### Características de los nuevos endpoints:

✅ **Sin autenticación requerida** (endpoints públicos)  
✅ **Formato de respuesta JSON** idéntico al anterior  
✅ **Zona horaria:** Colombia (America/Bogota, UTC-5)  
✅ **Fuente de datos:** Mismo archivo JSON (`/home/scalarapp/public_html/develop/code-test/ScaWatt.json`)  
✅ **Ordenamiento:** Historial ordenado descendente (más reciente primero)  
✅ **Rango válido:** `days` de 1 a 365  

### Manejo de errores:

**400 Bad Request:**
```json
{
  "success": false,
  "message": "El parámetro days debe ser un número entre 1 y 365"
}
```

**404 Not Found:**
```json
{
  "success": false,
  "message": "Archivo de datos no encontrado"
}
```

**500 Internal Server Error:**
```json
{
  "error": "Error al procesar solicitud: [detalle del error]"
}
```

---

## 📅 CRONOGRAMA DE MIGRACIÓN

### Timeline sugerido:

| Fecha | Acción |
|-------|--------|
| **29 Oct 2025** | ✅ Endpoints nuevos desplegados en producción |
| **30 Oct - 1 Nov** | 🔄 Frontend y App actualizan URLs |
| **2-3 Nov** | 🧪 Pruebas exhaustivas en desarrollo |
| **4-5 Nov** | 🚀 Despliegue a producción |
| **6 Nov** | ✅ Verificación final y cierre |

---

## ✅ CHECKLIST DE MIGRACIÓN

### Frontend (Ionic/Angular):
- [ ] Actualizar `scawatt.service.ts` (o archivo equivalente)
- [ ] Cambiar URL de valor actual
- [ ] Cambiar URL de historial
- [ ] Probar en local con datos reales
- [ ] Verificar que gráficos se renderizan correctamente
- [ ] Commit y push a repositorio
- [ ] Desplegar a desarrollo
- [ ] Desplegar a producción

### App Móvil:
- [ ] Actualizar servicio de ScaWatt
- [ ] Cambiar URL de valor actual
- [ ] Cambiar URL de historial
- [ ] Probar en dispositivo real
- [ ] Verificar que portafolio carga correctamente
- [ ] Commit y push a repositorio
- [ ] Generar build de desarrollo
- [ ] Generar build de producción
- [ ] Publicar en stores (si aplica)

---

## 🆘 SOPORTE

### Contacto para dudas:
- **Backend:** Daniel Ortiz - daniel.ortiz.penagos@me.com
- **Documentación completa:** `/home/scalarapp/public_html/scadmin/backend/AUDITORIA_BACKEND_COMPLETA.md`

### Logs del servidor:
```bash
# Ver logs de Laravel
tail -f /home/scalarapp/public_html/scadmin/backend/storage/logs/laravel.log

# Ver errores de Apache
tail -f /home/scalarapp/public_html/scadmin/backend/error_log
```

---

## 📖 RECURSOS ADICIONALES

- **Código fuente del controlador:** `/home/scalarapp/public_html/scadmin/backend/app/Http/Controllers/ScawattValorController.php`
- **Rutas API:** `/home/scalarapp/public_html/scadmin/backend/routes/api.php` (líneas 73-77)
- **Archivo de datos:** `/home/scalarapp/public_html/develop/code-test/ScaWatt.json`

---

## ⚡ VENTAJAS DE LA MIGRACIÓN

1. ✅ **Centralización:** Todo el código en Laravel, fácil de mantener
2. ✅ **Seguridad:** Mejor manejo de errores y validaciones
3. ✅ **Escalabilidad:** Más fácil agregar autenticación en el futuro si se requiere
4. ✅ **Logs:** Sistema de logging integrado de Laravel
5. ✅ **Versionado:** Control de versiones en Git
6. ✅ **Testing:** Más fácil escribir tests automatizados
7. ✅ **Documentación:** Código autodocumentado con PHPDoc

---

**Última actualización:** 29 de Octubre de 2025  
**Versión del documento:** 1.0  
**Estado:** ✅ Endpoints en producción y funcionando
