<?php

namespace App\Http\Controllers;

use App\Models\SolicitudContactoScawatt;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class AdminCorreosController extends Controller
{
    public function index(Request $request)
    {
        $estado = $request->query('estado', 'todos');
        $busqueda = trim((string) $request->query('q', ''));
        $orden = $request->query('orden', 'recientes');
        $porPagina = (int) $request->query('por_pagina', 15);
        $porPagina = $porPagina > 0 ? min($porPagina, 80) : 15;

        $solicitudesQuery = SolicitudContactoScawatt::query()
            ->with(['usuario:id,nombre,email,telefono', 'adminAtendio:id,nombre']);

        if ($estado !== 'todos') {
            $solicitudesQuery->where('estado', $estado);
        }

        if ($busqueda !== '') {
            $solicitudesQuery->where(function (Builder $query) use ($busqueda) {
                $query->where('nombre_completo', 'like', "%{$busqueda}%")
                      ->orWhere('email', 'like', "%{$busqueda}%")
                      ->orWhere('telefono', 'like', "%{$busqueda}%")
                      ->orWhere('mensaje', 'like', "%{$busqueda}%")
                      ->orWhereHas('usuario', function (Builder $usuarioQuery) use ($busqueda) {
                          $usuarioQuery->where('nombre', 'like', "%{$busqueda}%")
                                       ->orWhere('email', 'like', "%{$busqueda}%");
                      });
            });
        }

        $solicitudesQuery->when($orden, function (Builder $query) use ($orden) {
            return match ($orden) {
                'presupuesto' => $query->orderByDesc('presupuesto_aproximado'),
                'cantidad' => $query->orderByDesc('cantidad_scawatts'),
                'antiguedad' => $query->orderBy('created_at'),
                default => $query->orderByDesc('created_at'),
            };
        });

        $solicitudes = $solicitudesQuery->paginate($porPagina)->withQueryString();

        $stats = [
            'total' => SolicitudContactoScawatt::count(),
            'pendientes' => SolicitudContactoScawatt::where('estado', 'pendiente')->count(),
            'contactadas' => SolicitudContactoScawatt::where('estado', 'contactado')->count(),
            'completadas' => SolicitudContactoScawatt::where('estado', 'completado')->count(),
            'presupuesto_total' => $this->formatCurrency(SolicitudContactoScawatt::sum('presupuesto_aproximado')),
        ];

        $resumenEstados = SolicitudContactoScawatt::select('estado', DB::raw('COUNT(*) as total'))
            ->groupBy('estado')
            ->orderByDesc('total')
            ->get();

        $ultimas = SolicitudContactoScawatt::with('usuario:id,nombre')
            ->orderByDesc('created_at')
            ->limit(5)
            ->get();

        return view('admin.correos.index', [
            'solicitudes' => $solicitudes,
            'stats' => $stats,
            'resumenEstados' => $resumenEstados,
            'ultimas' => $ultimas,
            'filtros' => [
                'estado' => $estado,
                'orden' => $orden,
                'q' => $busqueda,
                'por_pagina' => $porPagina,
            ],
        ]);
    }

    private function formatCurrency($value): string
    {
        $numeric = $value !== null ? (float) $value : 0.0;
        return '$' . number_format($numeric, 0, ',', '.');
    }
}
