<?php

namespace App\Http\Controllers;

use App\Models\Deposito;
use App\Models\Marketplace;
use App\Models\Retiro;
use App\Models\Scawatt;
use App\Models\Usuario;
use Illuminate\Support\Facades\DB;

class AdminDashboardController extends Controller
{
    public function index()
    {
        $totalScawatts = Scawatt::count();
        $valorTotalActualRaw = (float) Scawatt::sum('valor_actual');
        $valorTotalInicialRaw = (float) Scawatt::sum('valor_inicial');
        $valorizacionTotalRaw = $valorTotalActualRaw - $valorTotalInicialRaw;
        $kwhTotalRaw = (float) Scawatt::sum('kwh_asignados');

        $retirosPendientesMontoRaw = (float) Retiro::where('estado', 'pendiente')->sum('monto');
        $depositosPendientesMontoRaw = (float) Deposito::where('estado', 'pendiente')->sum('monto');

        $metrics = [
            'total_scawatts' => $totalScawatts,
            'scawatts_marketplace' => Marketplace::where('estado', 'en_venta')->count(),
            'valor_total_actual' => $this->formatCurrency($valorTotalActualRaw),
            'valor_total_actual_raw' => $valorTotalActualRaw,
            'valor_total_inicial' => $this->formatCurrency($valorTotalInicialRaw),
            'valor_total_inicial_raw' => $valorTotalInicialRaw,
            'valorizacion_total' => $this->formatCurrency($valorizacionTotalRaw),
            'valorizacion_total_raw' => $valorizacionTotalRaw,
            'kwh_total' => number_format($kwhTotalRaw, 0, ',', '.'),
            'total_usuarios' => Usuario::count(),
            'usuarios_ultimos_30' => Usuario::where('created_at', '>=', now()->subDays(30))->count(),
            'retiros_pendientes' => Retiro::where('estado', 'pendiente')->count(),
            'retiros_pendientes_monto_raw' => $retirosPendientesMontoRaw,
            'retiros_pendientes_monto' => $this->formatCurrency($retirosPendientesMontoRaw),
            'depositos_pendientes' => Deposito::where('estado', 'pendiente')->count(),
            'depositos_pendientes_monto_raw' => $depositosPendientesMontoRaw,
            'depositos_pendientes_monto' => $this->formatCurrency($depositosPendientesMontoRaw),
        ];

        $topGranjas = DB::table('granjas')
            ->leftJoin('scawatts', 'scawatts.granja_id', '=', 'granjas.id')
            ->select(
                'granjas.id',
                'granjas.nombre',
                DB::raw('COUNT(scawatts.id) as total_scawatts'),
                DB::raw('COALESCE(SUM(scawatts.valor_actual), 0) as valor_total')
            )
            ->groupBy('granjas.id', 'granjas.nombre')
            ->orderByDesc('total_scawatts')
            ->limit(5)
            ->get();

        $charts = [
            'granjas' => [
                'labels' => $topGranjas->pluck('nombre')->toArray(),
                'scawatts' => $topGranjas->pluck('total_scawatts')->map(fn ($value) => (int) $value)->toArray(),
                'valor_total' => $topGranjas->pluck('valor_total')->map(fn ($value) => round((float) $value))->toArray(),
            ],
        ];

        $recentRetiros = Retiro::with('usuario')
            ->latest()
            ->limit(5)
            ->get();

        $recentDepositos = Deposito::with('usuario')
            ->latest()
            ->limit(5)
            ->get();

        $topUsuarios = Usuario::orderByDesc('saldo_total')
            ->limit(5)
            ->get(['id', 'nombre', 'saldo_total', 'saldo_disponible']);

        $marketplaceReciente = Marketplace::with(['scawatt', 'vendedor'])
            ->latest()
            ->limit(5)
            ->get();

        return view('admin.dashboard', [
            'metrics' => $metrics,
            'charts' => $charts,
            'recentRetiros' => $recentRetiros,
            'recentDepositos' => $recentDepositos,
            'topUsuarios' => $topUsuarios,
            'marketplaceReciente' => $marketplaceReciente,
        ]);
    }

    private function formatCurrency(float $value): string
    {
        return '$' . number_format($value, 0, ',', '.');
    }
}
