<?php

namespace App\Http\Controllers;

use App\Models\Marketplace;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;

class AdminMarketplaceController extends Controller
{
    public function index(Request $request)
    {
        $estado = $request->query('estado', 'todos');
        $mostrarSoloDestacados = $request->boolean('destacados');
        $orden = $request->query('orden', 'recientes');
        $busqueda = trim((string) $request->query('q', ''));

        $publicacionesQuery = Marketplace::query()
            ->with([
                'scawatt:id,scawatt_id,granja_id,valor_actual,valor_inicial,estado',
                'scawatt.granja:id,nombre',
                'vendedor:id,nombre,email',
                'comprador:id,nombre,email',
            ]);

        if ($estado !== 'todos') {
            $publicacionesQuery->where('estado', $estado);
        }

        if ($mostrarSoloDestacados) {
            $publicacionesQuery->where('destacado', true);
        }

        if ($busqueda !== '') {
            $publicacionesQuery->where(function ($query) use ($busqueda) {
                $query->where('id', $busqueda)
                      ->orWhere('scawatt_id', 'like', "%{$busqueda}%")
                      ->orWhereHas('scawatt', function ($scawattQuery) use ($busqueda) {
                          $scawattQuery->where('scawatt_id', 'like', "%{$busqueda}%");
                      })
                      ->orWhereHas('vendedor', function ($vendedorQuery) use ($busqueda) {
                          $vendedorQuery->where('nombre', 'like', "%{$busqueda}%")
                                        ->orWhere('email', 'like', "%{$busqueda}%");
                      })
                      ->orWhereHas('comprador', function ($compradorQuery) use ($busqueda) {
                          $compradorQuery->where('nombre', 'like', "%{$busqueda}%")
                                         ->orWhere('email', 'like', "%{$busqueda}%");
                      });
            });
        }

        switch ($orden) {
            case 'precio_desc':
                $publicacionesQuery->orderByDesc('precio_venta');
                break;
            case 'precio_asc':
                $publicacionesQuery->orderBy('precio_venta');
                break;
            case 'antiguedad':
                $publicacionesQuery->orderBy('fecha_publicacion');
                break;
            default:
                $publicacionesQuery->orderByDesc(DB::raw('COALESCE(fecha_publicacion, created_at)'));
                break;
        }

        $publicaciones = $publicacionesQuery->paginate(12)->withQueryString();

        $inicioMes = now()->startOfMonth();
        $finMes = now();

        $disponiblesQuery = Marketplace::disponibles();
        $vendidosMesQuery = Marketplace::vendidos()->whereBetween('fecha_venta', [$inicioMes, $finMes]);

        $stats = [
            'activos' => (clone $disponiblesQuery)->count(),
            'valor_activos' => $this->formatCurrency((clone $disponiblesQuery)->sum('precio_venta')),
            'promedio_precio' => $this->formatCurrency((clone $disponiblesQuery)->avg('precio_venta')),
            'vendidos_mes' => (clone $vendidosMesQuery)->count(),
            'comision_mes' => $this->formatCurrency((clone $vendidosMesQuery)->sum('comision_plataforma')),
            'tiempo_promedio_venta' => $this->formatTiempoPromedio($this->obtenerPromedioDiasVenta()),
        ];

        $resumenEstados = $this->obtenerResumenEstados();
        $totales = [
            'publicaciones' => $resumenEstados->sum('total'),
            'valor_estimado' => $this->formatCurrency(Marketplace::sum('precio_venta')),
        ];

        $destacados = Marketplace::destacados()
            ->with(['scawatt:id,scawatt_id,valor_actual', 'vendedor:id,nombre'])
            ->orderByDesc(DB::raw('COALESCE(fecha_publicacion, created_at)'))
            ->take(5)
            ->get();

        $ventasRecientes = Marketplace::vendidos()
            ->with(['scawatt:id,scawatt_id', 'vendedor:id,nombre', 'comprador:id,nombre'])
            ->orderByDesc('fecha_venta')
            ->take(6)
            ->get();

        $filtros = [
            'estado' => $estado,
            'destacados' => $mostrarSoloDestacados,
            'orden' => $orden,
            'q' => $busqueda,
        ];

        return view('admin.marketplace.index', [
            'publicaciones' => $publicaciones,
            'stats' => $stats,
            'resumenEstados' => $resumenEstados,
            'totales' => $totales,
            'destacados' => $destacados,
            'ventasRecientes' => $ventasRecientes,
            'filtros' => $filtros,
        ]);
    }

    private function formatCurrency($value): string
    {
        $numeric = $value !== null ? (float) $value : 0.0;
        return '$' . number_format($numeric, 0, ',', '.');
    }

    private function formatTiempoPromedio(?float $dias): string
    {
        if ($dias === null) {
            return 'N/D';
        }

        if ($dias < 1) {
            return round($dias * 24) . ' h';
        }

        return number_format($dias, 1, ',', '.') . ' días';
    }

    private function obtenerPromedioDiasVenta(): ?float
    {
        $resultado = Marketplace::vendidos()
            ->whereNotNull('fecha_publicacion')
            ->whereNotNull('fecha_venta')
            ->selectRaw('AVG(TIMESTAMPDIFF(HOUR, fecha_publicacion, fecha_venta)) / 24 as promedio_dias')
            ->value('promedio_dias');

        return $resultado !== null ? (float) $resultado : null;
    }

    private function obtenerResumenEstados(): Collection
    {
        return Marketplace::select('estado',
                DB::raw('COUNT(*) as total'),
                DB::raw('SUM(precio_venta) as valor_total')
            )
            ->groupBy('estado')
            ->orderByDesc('total')
            ->get()
            ->map(function ($registro) {
                return [
                    'estado' => $registro->estado ?? 'sin_estado',
                    'total' => (int) $registro->total,
                    'valor' => $this->formatCurrency($registro->valor_total),
                ];
            });
    }
}
