<?php

namespace App\Http\Controllers;

use App\Models\Notificacion;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;

class AdminNotificacionesController extends Controller
{
    public function index(Request $request)
    {
        $estado = $request->query('estado', 'todas');
        $tipo = $request->query('tipo', 'todos');
        $prioridad = $request->query('prioridad', 'todas');
        $busqueda = trim((string) $request->query('q', ''));
        $orden = $request->query('orden', 'recientes');
        $porPagina = (int) $request->query('por_pagina', 25);
        $porPagina = $porPagina > 0 ? min($porPagina, 100) : 25;

        $notificacionesQuery = Notificacion::query()
            ->with('usuario:id,nombre,email');

        if ($estado !== 'todas') {
            $notificacionesQuery->where('leido', $estado === 'leidas');
        }

        if ($tipo !== 'todos') {
            $notificacionesQuery->where('tipo', $tipo);
        }

        if ($prioridad !== 'todas') {
            $notificacionesQuery->where('prioridad', $prioridad);
        }

        if ($busqueda !== '') {
            $notificacionesQuery->where(function (Builder $query) use ($busqueda) {
                $query->where('titulo', 'like', "%{$busqueda}%")
                      ->orWhere('mensaje', 'like', "%{$busqueda}%")
                      ->orWhereHas('usuario', function (Builder $subQuery) use ($busqueda) {
                          $subQuery->where('nombre', 'like', "%{$busqueda}%")
                                   ->orWhere('email', 'like', "%{$busqueda}%");
                      });
            });
        }

        $notificacionesQuery->when($orden, function (Builder $query) use ($orden) {
            return match ($orden) {
                'prioridad' => $query->orderByDesc(DB::raw("CASE prioridad WHEN 'alta' THEN 3 WHEN 'media' THEN 2 WHEN 'baja' THEN 1 ELSE 0 END"))
                                      ->orderByDesc('created_at'),
                'tipo' => $query->orderBy('tipo')->orderByDesc('created_at'),
                default => $query->orderByDesc('created_at'),
            };
        });

        $notificaciones = $notificacionesQuery->paginate($porPagina)->withQueryString();

        $stats = [
            'total' => Notificacion::count(),
            'sin_leer' => Notificacion::where('leido', false)->count(),
            'leidas' => Notificacion::where('leido', true)->count(),
            'ultimas_24h' => Notificacion::where('created_at', '>=', Carbon::now()->subDay())->count(),
        ];

        $resumenTipos = Notificacion::select('tipo', DB::raw('COUNT(*) as total'))
            ->groupBy('tipo')
            ->orderByDesc('total')
            ->get();

        $resumenPrioridades = Notificacion::select('prioridad', DB::raw('COUNT(*) as total'))
            ->groupBy('prioridad')
            ->orderByDesc('total')
            ->get();

        $recientesSinLeer = Notificacion::with('usuario:id,nombre')
            ->where('leido', false)
            ->orderByDesc('created_at')
            ->limit(6)
            ->get();

        return view('admin.notificaciones.index', [
            'notificaciones' => $notificaciones,
            'stats' => $stats,
            'resumenTipos' => $resumenTipos,
            'resumenPrioridades' => $resumenPrioridades,
            'recientesSinLeer' => $recientesSinLeer,
            'filtros' => [
                'estado' => $estado,
                'tipo' => $tipo,
                'prioridad' => $prioridad,
                'orden' => $orden,
                'q' => $busqueda,
                'por_pagina' => $porPagina,
            ],
        ]);
    }
}
