<?php

namespace App\Http\Controllers;

use App\Models\Usuario;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class AdminUsuariosController extends Controller
{
    public function index(Request $request)
    {
        $estado = $request->query('estado', 'todos');
        $verificacion = $request->query('verificacion', 'todos');
        $orden = $request->query('orden', 'recientes');
        $busqueda = trim((string) $request->query('q', ''));
        $porPagina = (int) $request->query('por_pagina', 20);
        $porPagina = $porPagina > 0 ? min($porPagina, 100) : 20;

        $usuariosQuery = Usuario::query()
            ->withCount('scawatts')
            ->withSum(['scawatts as valor_portafolio' => function (Builder $query) {
                $query->whereNotNull('valor_actual');
            }], 'valor_actual')
            ->withSum(['depositos as total_depositos' => function (Builder $query) {
                $query->where('estado', 'aprobado');
            }], 'monto')
            ->withSum(['retiros as total_retiros' => function (Builder $query) {
                $query->where('estado', 'completado');
            }], 'monto');

        if ($estado !== 'todos') {
            $usuariosQuery->where('estado', $estado);
        }

        if ($verificacion !== 'todos') {
            $usuariosQuery->where(function (Builder $query) use ($verificacion) {
                return match ($verificacion) {
                    'completos' => $query->whereNotNull('cedula_verificada_at')
                                            ->whereNotNull('datos_bancarios_verificados_at'),
                    'cedula' => $query->whereNotNull('cedula_verificada_at')
                                       ->whereNull('datos_bancarios_verificados_at'),
                    'bancarios' => $query->whereNotNull('datos_bancarios_verificados_at')
                                         ->whereNull('cedula_verificada_at'),
                    'pendientes' => $query->whereNull('cedula_verificada_at')
                                           ->orWhereNull('datos_bancarios_verificados_at'),
                    default => $query,
                };
            });
        }

        if ($busqueda !== '') {
            $usuariosQuery->where(function (Builder $query) use ($busqueda) {
                $query->where('nombre', 'like', "%{$busqueda}%")
                      ->orWhere('email', 'like', "%{$busqueda}%")
                      ->orWhere('cedula', 'like', "%{$busqueda}%")
                      ->orWhere('telefono', 'like', "%{$busqueda}%");
            });
        }

        $usuariosQuery->when($orden, function (Builder $query) use ($orden) {
            return match ($orden) {
                'saldo_desc' => $query->orderByDesc('saldo_total'),
                'saldo_asc' => $query->orderBy('saldo_total'),
                'nombre' => $query->orderBy('nombre'),
                'actividad' => $query->orderByDesc('updated_at'),
                default => $query->orderByDesc('created_at'),
            };
        });

        $usuarios = $usuariosQuery->paginate($porPagina)->withQueryString();

        $hoy = Carbon::now();
        $hace30Dias = $hoy->copy()->subDays(30);

        $stats = [
            'total' => Usuario::count(),
            'activos' => Usuario::where('estado', 'activo')->count(),
            'nuevos_mes' => Usuario::where('created_at', '>=', $hace30Dias)->count(),
            'verificados' => Usuario::whereNotNull('cedula_verificada_at')
                                     ->whereNotNull('datos_bancarios_verificados_at')
                                     ->count(),
            'saldo_total' => $this->formatCurrency(Usuario::sum('saldo_total')),
            'saldo_disponible' => $this->formatCurrency(Usuario::sum('saldo_disponible')),
            'saldo_pendiente' => $this->formatCurrency(Usuario::sum('saldo_pendiente')),
        ];

        $verificaciones = [
            'cedula' => Usuario::whereNotNull('cedula_verificada_at')->count(),
            'bancarios' => Usuario::whereNotNull('datos_bancarios_verificados_at')->count(),
            'completos' => Usuario::whereNotNull('cedula_verificada_at')
                                  ->whereNotNull('datos_bancarios_verificados_at')
                                  ->count(),
            'pendientes' => Usuario::where(function (Builder $query) {
                $query->whereNull('cedula_verificada_at')
                      ->orWhereNull('datos_bancarios_verificados_at');
            })->count(),
        ];

        $segmentacionEstados = Usuario::select('estado', DB::raw('COUNT(*) as total'))
            ->groupBy('estado')
            ->orderByDesc('total')
            ->get()
            ->map(function ($row) {
                return [
                    'estado' => $row->estado ?? 'sin_estado',
                    'total' => (int) $row->total,
                ];
            });

        $topClientes = Usuario::select('id', 'nombre', 'email', 'saldo_total', 'saldo_disponible', 'saldo_pendiente')
            ->orderByDesc('saldo_total')
            ->limit(6)
            ->get();

        return view('admin.usuarios.index', [
            'usuarios' => $usuarios,
            'stats' => $stats,
            'verificaciones' => $verificaciones,
            'segmentacionEstados' => $segmentacionEstados,
            'topClientes' => $topClientes,
            'filtros' => [
                'estado' => $estado,
                'verificacion' => $verificacion,
                'orden' => $orden,
                'q' => $busqueda,
                'por_pagina' => $porPagina,
            ],
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'nombre' => ['required', 'string', 'max:255'],
            'email' => ['required', 'email', 'max:255', 'unique:usuarios,email'],
            'telefono' => ['nullable', 'string', 'max:45'],
            'estado' => ['required', 'string', Rule::in(['activo', 'inactivo', 'bloqueado', 'admin'])],
            'password' => ['nullable', 'string', 'min:8', 'max:255'],
        ]);

        $password = $validated['password'] ?? Str::random(12);

        $usuario = Usuario::create([
            'nombre' => $validated['nombre'],
            'email' => $validated['email'],
            'telefono' => $validated['telefono'] ?? null,
            'estado' => $validated['estado'],
            'password' => $password,
        ]);

        $redirect = redirect()
            ->route('admin.usuarios')
            ->with('status', "Usuario {$usuario->nombre} creado exitosamente.");

        if (empty($validated['password'])) {
            $redirect->with('generated_password', $password);
        }

        return $redirect;
    }

    public function update(Request $request, Usuario $usuario)
    {
        $validated = $request->validate([
            'nombre' => ['required', 'string', 'max:255'],
            'email' => ['required', 'email', 'max:255', Rule::unique('usuarios', 'email')->ignore($usuario->id)],
            'telefono' => ['nullable', 'string', 'max:45'],
            'estado' => ['required', 'string', Rule::in(['activo', 'inactivo', 'bloqueado', 'admin'])],
            'password' => ['nullable', 'string', 'min:8', 'max:255'],
        ]);

        $usuario->nombre = $validated['nombre'];
        $usuario->email = $validated['email'];
        $usuario->telefono = $validated['telefono'] ?? null;
        $usuario->estado = $validated['estado'];

        if (!empty($validated['password'])) {
            $usuario->password = $validated['password'];
        }

        $usuario->save();

        return redirect()
            ->back()
            ->with('status', "Usuario {$usuario->nombre} actualizado exitosamente.");
    }

    public function actualizarEstado(Request $request, Usuario $usuario)
    {
        $validated = $request->validate([
            'estado' => ['required', Rule::in(['activo', 'inactivo', 'bloqueado', 'admin'])],
        ]);

        $usuario->estado = $validated['estado'];
        $usuario->save();

        return redirect()
            ->back()
            ->with('status', "Estado de {$usuario->nombre} actualizado a {$usuario->estado}.");
    }

    private function formatCurrency($value): string
    {
        $numeric = $value !== null ? (float) $value : 0.0;
        return '$' . number_format($numeric, 0, ',', '.');
    }
}
