<?php

namespace App\Http\Controllers;

use App\Models\Granja;
use Illuminate\Http\Request;

class GranjaController extends Controller
{
    /**
     * Listar todas las granjas
     * GET /api/granjas
     */
    public function index(Request $request)
    {
        $query = Granja::withCount('scawatts');

        // Filtros opcionales
        if ($request->has('categoria')) {
            $query->where('categoria', $request->categoria);
        }

        if ($request->has('departamento')) {
            $query->where('departamento', $request->departamento);
        }

        if ($request->has('estado')) {
            $query->where('estado', $request->estado);
        }

        $granjas = $query->orderBy('nombre')->get();

        return response()->json([
            'success' => true,
            'granjas' => $granjas
        ]);
    }

    /**
     * Ver detalle de una granja
     * GET /api/granjas/{id}
     */
    public function show($id)
    {
        $granja = Granja::withCount('scawatts')
            ->with(['scawatts' => function($query) {
                $query->select('id', 'scawatt_id', 'granja_id', 'usuario_id', 'valor_actual', 'estado')
                      ->limit(10);
            }])
            ->findOrFail($id);

        return response()->json([
            'success' => true,
            'granja' => $granja
        ]);
    }

    /**
     * Crear nueva granja (admin)
     * POST /api/admin/granjas
     */
    public function store(Request $request)
    {
        $request->validate([
            'nombre' => 'required|string|max:200',
            'categoria' => 'required|in:granja_solar,parque_solar,industrial,domestico,vehicular',
            'departamento' => 'required|string|max:100',
            'municipio' => 'required|string|max:100',
            'capacidad_mw' => 'required|numeric|min:0',
            'estado' => 'required|in:operativa,construccion,planificada,mantenimiento'
        ]);

        $granja = Granja::create($request->all());

        return response()->json([
            'success' => true,
            'message' => 'Granja creada exitosamente',
            'granja' => $granja
        ], 201);
    }

    /**
     * Actualizar granja (admin)
     * PUT /api/admin/granjas/{id}
     */
    public function update(Request $request, $id)
    {
        $granja = Granja::findOrFail($id);

        $request->validate([
            'nombre' => 'sometimes|string|max:200',
            'categoria' => 'sometimes|in:granja_solar,parque_solar,industrial,domestico,vehicular',
            'departamento' => 'sometimes|string|max:100',
            'municipio' => 'sometimes|string|max:100',
            'capacidad_mw' => 'sometimes|numeric|min:0',
            'estado' => 'sometimes|in:operativa,construccion,planificada,mantenimiento'
        ]);

        $granja->update($request->all());

        return response()->json([
            'success' => true,
            'message' => 'Granja actualizada exitosamente',
            'granja' => $granja
        ]);
    }

    /**
     * Obtener estadísticas de granjas (admin)
     * GET /api/admin/granjas/estadisticas
     */
    public function estadisticas()
    {
        $stats = [
            'total_granjas' => Granja::count(),
            'operativas' => Granja::where('estado', 'operativa')->count(),
            'en_construccion' => Granja::where('estado', 'construccion')->count(),
            'planificadas' => Granja::where('estado', 'planificada')->count(),
            'capacidad_total_mw' => Granja::sum('capacidad_mw'),
            'por_departamento' => Granja::selectRaw('departamento, COUNT(*) as total')
                ->groupBy('departamento')
                ->get(),
            'por_categoria' => Granja::selectRaw('categoria, COUNT(*) as total')
                ->groupBy('categoria')
                ->get()
        ];

        return response()->json($stats);
    }

    /**
     * Listar departamentos con granjas
     * GET /api/granjas/departamentos
     */
    public function departamentos()
    {
        $departamentos = Granja::select('departamento')
            ->distinct()
            ->orderBy('departamento')
            ->pluck('departamento');

        return response()->json([
            'success' => true,
            'departamentos' => $departamentos
        ]);
    }

    /**
     * Listar categorías disponibles
     * GET /api/granjas/categorias
     */
    public function categorias()
    {
        return response()->json([
            'success' => true,
            'categorias' => [
                ['value' => 'granja_solar', 'label' => 'Granja Solar'],
                ['value' => 'parque_solar', 'label' => 'Parque Solar'],
                ['value' => 'industrial', 'label' => 'Industrial'],
                ['value' => 'domestico', 'label' => 'Doméstico'],
                ['value' => 'vehicular', 'label' => 'Vehicular']
            ]
        ]);
    }
}

