<?php

namespace App\Http\Controllers;

use App\Models\Usuario;
use App\Models\AuditoriaAdmin;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class UsuarioController extends Controller
{
    /**
     * Listar todos los usuarios
     * GET /api/admin/usuarios
     */
    public function index(Request $request)
    {
        $query = Usuario::query();

        // Filtros opcionales
        if ($request->has('estado')) {
            $query->where('estado', $request->estado);
        }

        if ($request->has('buscar')) {
            $buscar = $request->buscar;
            $query->where(function($q) use ($buscar) {
                $q->where('nombre', 'like', "%{$buscar}%")
                  ->orWhere('email', 'like', "%{$buscar}%");
            });
        }

        $usuarios = $query->with(['scawatts', 'notificaciones', 'mensajesRecibidos'])
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return response()->json($usuarios);
    }

    /**
     * Ver detalle de usuario
     * GET /api/admin/usuarios/{id}
     */
    public function show($id)
    {
        try {
            $usuario = Usuario::with([
                'scawatts',
                'depositos',
                'retiros',
                'notificaciones',
                'mensajesRecibidos',
                'mensajesEnviados',
                'documentos',
                'trazabilidad'
            ])->findOrFail($id);

            // Estadísticas del usuario
            $estadisticas = [
                'total_scawatts' => $usuario->scawatts()->count(),
                'total_depositos' => $usuario->depositos()->sum('monto'),
                'total_retiros' => $usuario->retiros()->where('estado', 'completado')->sum('monto'),
                'notificaciones_no_leidas' => $usuario->notificacionesNoLeidas()->count(),
                'mensajes_no_leidos' => $usuario->mensajesNoLeidos()->count(),
                'valor_total_scawatts' => $usuario->scawatts()->sum('valor_actual')
            ];

            return response()->json([
                'usuario' => $usuario,
                'estadisticas' => $estadisticas
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Usuario no encontrado',
                'error' => $e->getMessage()
            ], 404);
        }
    }

    /**
     * Crear nuevo usuario
     * POST /api/admin/usuarios
     * 
     * Lógica de contraseña:
     * - Si se proporciona password, se usa ese
     * - Si no hay password pero sí cédula, usa la cédula como password
     * - Si no hay ni password ni cédula, genera password aleatoria
     */
    public function store(Request $request)
    {
        $request->validate([
            'nombre' => 'required|string|max:255',
            'email' => 'required|email|unique:usuarios,email',
            'password' => 'nullable|string|min:6',
            'telefono' => 'nullable|string|max:20',
            'cedula' => 'nullable|string|max:20',
            'id_wordpress' => 'nullable|integer|unique:usuarios,id_wordpress',
            'estado' => 'nullable|in:activo,inactivo,bloqueado'
        ]);

        try {
            DB::beginTransaction();

            // Determinar la contraseña a usar
            $passwordGenerada = null;
            $passwordTextoPlano = null;

            if ($request->filled('password')) {
                // Si el admin proporciona una contraseña, usarla
                $passwordTextoPlano = $request->password;
            } elseif ($request->filled('cedula')) {
                // Si no hay password pero sí cédula, usar la cédula como password
                $passwordTextoPlano = $request->cedula;
                $passwordGenerada = $request->cedula;
            } else {
                // Si no hay ni password ni cédula, generar una aleatoria
                $passwordTextoPlano = $this->generarPasswordAleatoria();
                $passwordGenerada = $passwordTextoPlano;
            }

            $usuario = Usuario::create([
                'nombre' => $request->nombre,
                'email' => $request->email,
                'password' => \Hash::make($passwordTextoPlano),
                'telefono' => $request->telefono,
                'cedula' => $request->cedula,
                'id_wordpress' => $request->id_wordpress,
                'saldo_disponible' => 0,
                'saldo_pendiente' => 0,
                'saldo_total' => 0,
                'estado' => $request->estado ?? 'activo'
            ]);

            // Si se generó una contraseña, enviar notificación
            if ($passwordGenerada) {
                $this->enviarPasswordPorEmail($usuario, $passwordGenerada);
            }

            // Registrar auditoría
            if ($request->has('admin_id')) {
                AuditoriaAdmin::registrar(
                    $request->admin_id,
                    'crear',
                    'usuarios',
                    "Usuario creado: {$usuario->nombre}" . ($passwordGenerada ? " - Password generada" : ""),
                    'Usuario',
                    $usuario->id,
                    null,
                    $usuario->toArray()
                );
            }

            DB::commit();

            $responseData = [
                'success' => true,
                'message' => 'Usuario creado exitosamente',
                'usuario' => $usuario
            ];

            // Incluir password generada en la respuesta para que el admin la vea
            if ($passwordGenerada) {
                $responseData['password_generada'] = $passwordGenerada;
                $responseData['mensaje_password'] = 'Se ha enviado una notificación al usuario con su contraseña';
            }

            return response()->json($responseData, 201);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error al crear usuario',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Actualizar usuario
     * PUT /api/admin/usuarios/{id}
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'nombre' => 'sometimes|string|max:255',
            'email' => 'sometimes|email|unique:usuarios,email,' . $id,
            'telefono' => 'nullable|string|max:20',
            'estado' => 'sometimes|in:activo,inactivo,bloqueado'
        ]);

        try {
            $usuario = Usuario::findOrFail($id);
            $datosAnteriores = $usuario->toArray();

            DB::beginTransaction();

            $usuario->update($request->all());

            // Registrar auditoría
            if ($request->has('admin_id')) {
                AuditoriaAdmin::registrar(
                    $request->admin_id,
                    'actualizar',
                    'usuarios',
                    "Usuario actualizado: {$usuario->nombre}",
                    'Usuario',
                    $usuario->id,
                    $datosAnteriores,
                    $usuario->toArray()
                );
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Usuario actualizado exitosamente',
                'usuario' => $usuario
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error al actualizar usuario',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Cambiar estado de usuario
     * PUT /api/admin/usuarios/{id}/estado
     */
    public function cambiarEstado(Request $request, $id)
    {
        $request->validate([
            'estado' => 'required|in:activo,inactivo,bloqueado',
            'admin_id' => 'required|exists:usuarios,id',
            'motivo' => 'nullable|string'
        ]);

        try {
            $usuario = Usuario::findOrFail($id);
            $estadoAnterior = $usuario->estado;

            DB::beginTransaction();

            $usuario->update(['estado' => $request->estado]);

            // Registrar auditoría
            AuditoriaAdmin::registrar(
                $request->admin_id,
                'cambiar_estado_usuario',
                'usuarios',
                "Estado cambiado de '{$estadoAnterior}' a '{$request->estado}'. Motivo: " . ($request->motivo ?? 'No especificado'),
                'Usuario',
                $usuario->id
            );

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Estado de usuario actualizado',
                'usuario' => $usuario
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error al cambiar estado',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Dashboard del usuario (vista completa)
     * GET /api/usuarios/{id}/dashboard
     */
    public function dashboard($id)
    {
        try {
            $usuario = Usuario::with([
                'scawatts' => function($q) {
                    $q->where('estado', 'activo');
                },
                'depositos' => function($q) {
                    $q->orderBy('created_at', 'desc')->limit(10);
                },
                'retiros' => function($q) {
                    $q->orderBy('created_at', 'desc')->limit(10);
                }
            ])->findOrFail($id);

            // DEVOLVER DATOS DIRECTOS SIN WRAPPER (formato original)
            $dashboard = [
                'usuario' => [
                    'id' => $usuario->id,
                    'nombre' => $usuario->nombre,
                    'email' => $usuario->email,
                    'estado' => $usuario->estado,
                    'saldo_disponible' => $usuario->saldo_disponible,
                    'saldo_total' => $usuario->saldo_total
                ],
                'scawatts' => [
                    'total' => $usuario->scawatts->count(),
                    'valor_total' => $usuario->scawatts->sum('valor_actual'),
                    'listado' => $usuario->scawatts
                ],
                'transacciones' => [
                    'depositos_total' => $usuario->depositos()->where('estado', 'aprobado')->sum('monto'),
                    'retiros_total' => $usuario->retiros()->where('estado', 'completado')->sum('monto'),
                    'ultimos_depositos' => $usuario->depositos,
                    'ultimos_retiros' => $usuario->retiros
                ],
                'notificaciones_no_leidas' => $usuario->notificacionesNoLeidas()->count(),
                'mensajes_no_leidos' => 0  // Tabla mensajes no tiene estado de lectura por usuario
            ];

            return response()->json($dashboard);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al cargar dashboard',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Obtener perfil general del usuario
     * GET /api/usuarios/{id}/perfil
     */
    public function obtenerPerfil($id)
    {
        try {
            $usuario = Usuario::findOrFail($id);

            return response()->json([
                'success' => true,
                'perfil' => $this->formatearPerfilUsuario($usuario)
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Usuario no encontrado',
                'error' => $e->getMessage()
            ], 404);
        }
    }

    /**
     * Actualizar perfil general del usuario
     * PUT /api/usuarios/{id}/perfil
     */
    public function actualizarPerfil(Request $request, $id)
    {
        $validated = $request->validate([
            'nombre' => 'sometimes|string|max:255',
            'email' => 'sometimes|email|unique:usuarios,email,' . $id,
            'telefono' => 'sometimes|nullable|string|max:20',
            'picture' => 'sometimes|nullable|string|max:500'
        ]);

        try {
            $usuario = Usuario::findOrFail($id);

            $usuario->fill($validated);
            $usuario->save();

            return response()->json([
                'success' => true,
                'message' => 'Perfil actualizado correctamente',
                'perfil' => $this->formatearPerfilUsuario($usuario->fresh())
            ]);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Usuario no encontrado'
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al actualizar el perfil',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Subir avatar del usuario (almacena archivo y devuelve URL pública)
     * POST /api/usuarios/{id}/avatar/upload
     */
    public function subirAvatar(Request $request, $id)
    {
        $request->validate([
            'avatar' => 'required|image|mimes:jpg,jpeg,png,heic,webp|max:10240'
        ]);

        try {
            $usuario = Usuario::findOrFail($id);

            $archivo = $request->file('avatar');
            $directorio = "avatars/user-{$usuario->id}";

            // Guardamos la ruta anterior para intentar limpiarla si pertenece a nuestro storage
            $rutaAnterior = $usuario->picture;

            // Aseguramos la existencia del directorio destino
            Storage::disk('public')->makeDirectory($directorio);

            $path = $archivo->store($directorio, 'public');
            $url = Storage::disk('public')->url($path);

            // Persistimos la nueva URL del avatar
            $usuario->picture = $url;
            $usuario->save();

            // Intentamos eliminar el archivo anterior si estaba dentro del storage público
            if ($rutaAnterior && str_contains($rutaAnterior, '/storage/')) {
                $relativePath = ltrim(str_replace(url('/storage'), '', $rutaAnterior), '/');
                if ($relativePath && Storage::disk('public')->exists($relativePath)) {
                    Storage::disk('public')->delete($relativePath);
                }
            }

            return response()->json([
                'success' => true,
                'message' => 'Avatar subido correctamente',
                'data' => [
                    'url' => $url,
                    'path' => $path,
                    'user' => $this->formatearPerfilUsuario($usuario->fresh()),
                ]
            ], 201);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Usuario no encontrado'
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al subir el avatar',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Registrar token FCM para notificaciones push
     * POST /api/usuarios/{id}/register-token
     */
    public function registrarTokenFCM(Request $request, $id)
    {
        $request->validate([
            'fcm_token' => 'required|string|max:255'
        ]);

        try {
            $usuario = Usuario::findOrFail($id);
            
            $usuario->fcm_token = $request->fcm_token;
            $usuario->fcm_token_updated_at = now();
            $usuario->save();

            return response()->json([
                'success' => true,
                'message' => 'Token FCM registrado exitosamente'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al registrar token',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Eliminar token FCM (logout)
     * DELETE /api/usuarios/{id}/unregister-token
     */
    public function eliminarTokenFCM($id)
    {
        try {
            $usuario = Usuario::findOrFail($id);
            
            $usuario->fcm_token = null;
            $usuario->fcm_token_updated_at = null;
            $usuario->save();

            return response()->json([
                'success' => true,
                'message' => 'Token FCM eliminado'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al eliminar token',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Obtener todos los movimientos (depósitos, retiros, compras y ventas) de un usuario
     * GET /api/usuarios/{id}/movimientos
     * 
     * ESPECIFICACIÓN:
     * - tipo: "ingreso" (compras ScaWatt, depósitos, ventas) | "egreso" (retiros)
     * - monto: SIEMPRE POSITIVO
     * - Ordenado por fecha descendente
     */
    public function movimientos($id)
    {
        try {
            $usuario = Usuario::findOrFail($id);
            
            // INGRESOS: Depósitos/Abonos
            $depositos = $usuario->depositos()
                ->where('estado', 'aprobado')
                ->get()
                ->map(function($deposito) {
                    return [
                        'id' => $deposito->id,
                        'tipo' => 'ingreso',
                        'concepto' => 'Abono',
                        'descripcion' => $deposito->metodo_pago ?? 'Abono a billetera',
                        'monto' => abs((float) $deposito->monto),  // SIEMPRE POSITIVO
                        'fecha' => $deposito->created_at->format('Y-m-d H:i:s'),
                        'scawatt_id' => null,
                        'detalle' => $deposito->metodo_pago,
                        'estado' => 'completado'
                    ];
                });
            
            // EGRESOS: Retiros
            $retiros = $usuario->retiros()
                ->where('estado', 'completado')
                ->get()
                ->map(function($retiro) {
                    return [
                        'id' => $retiro->id,
                        'tipo' => 'egreso',
                        'concepto' => 'Retiro',
                        'descripcion' => 'Retiro a cuenta bancaria',
                        'monto' => abs((float) $retiro->monto),  // SIEMPRE POSITIVO
                        'fecha' => $retiro->created_at->format('Y-m-d H:i:s'),
                        'scawatt_id' => null,
                        'detalle' => 'Retiro a cuenta bancaria',
                        'estado' => 'completado'
                    ];
                });
            
            // Compras de ScaWatts NO afectan el saldo de billetera
            // Se muestran solo como registro informativo
            $compras_list = \DB::table('marketplace')
                ->join('scawatts', 'marketplace.scawatt_id', '=', 'scawatts.id')
                ->where('marketplace.comprador_id', $id)
                ->where('marketplace.estado', 'vendido')
                ->select('marketplace.*', 'scawatts.scawatt_id as scawatt_serial')
                ->orderBy('marketplace.fecha_venta', 'ASC')
                ->get();
            
            $primera_compra_id = $compras_list->first()->id ?? null;
            
            $compras = $compras_list->map(function($compra) use ($primera_compra_id) {
                $es_primera = ($compra->id === $primera_compra_id);
                return [
                    'id' => $compra->id,
                    'tipo' => 'egreso',  // EGRESO: el dinero sale de la billetera al comprar
                    'concepto' => 'Compra ScaWatt',
                    'descripcion' => $es_primera ? 'Compra de tu primer ScaWatt' : 'Compra de ScaWatt',
                    'monto' => abs((float) $compra->precio_venta),  // SIEMPRE POSITIVO
                    'fecha' => $compra->fecha_venta,
                    'scawatt_id' => $compra->scawatt_serial,
                    'detalle' => 'ScaWatt ' . $compra->scawatt_serial,
                    'estado' => 'completado',
                    'es_primera_compra' => $es_primera,
                    'afecta_saldo' => false  // NO afecta saldo en el cálculo (ya está descontado)
                ];
            });
            
            // Ventas de ScaWatts en el marketplace (NO afectan saldo inmediatamente)
            // El dinero se acredita en el "Cierre Mensual" (depósito automático)
            $ventas = \DB::table('marketplace')
                ->join('scawatts', 'marketplace.scawatt_id', '=', 'scawatts.id')
                ->where('marketplace.vendedor_id', $id)
                ->where('marketplace.estado', 'vendido')
                ->select('marketplace.*', 'scawatts.scawatt_id as scawatt_serial')
                ->get()
                ->map(function($venta) {
                    $monto_neto = (float) $venta->precio_venta - (float) $venta->comision_plataforma;
                    return [
                        'id' => $venta->id,
                        'tipo' => 'ingreso',
                        'concepto' => 'Ganancia',
                        'descripcion' => 'Venta de ScaWatt (Se acreditará en Cierre Mensual)',
                        'monto' => abs($monto_neto),  // SIEMPRE POSITIVO
                        'fecha' => $venta->fecha_venta,
                        'scawatt_id' => $venta->scawatt_serial,
                        'detalle' => 'Venta de ScaWatt ' . $venta->scawatt_serial,
                        'estado' => 'pendiente_liquidacion',
                        'es_primera_compra' => false,
                        'afecta_saldo' => false  // NO afecta saldo (se paga en Cierre Mensual)
                    ];
                });
            
            // Combinar todos los movimientos
            $movimientos = collect()
                ->merge($depositos)
                ->merge($retiros)
                ->merge($compras)
                ->merge($ventas)
                ->sortByDesc('fecha')
                ->values();
            
            // Calcular resumen financiero
            // INCLUIR TODOS LOS MOVIMIENTOS en el historial (incluyendo compras y ventas)
            $ingresos_total = $movimientos->where('tipo', 'ingreso')->sum('monto');
            $egresos_total = $movimientos->where('tipo', 'egreso')->sum('monto');
            
            // El saldo_actual es el saldo_disponible real del usuario (fuente de verdad)
            $saldo_actual = $usuario->saldo_disponible;            return response()->json([
                'resumen' => [
                    'ingresos_total' => $ingresos_total,
                    'egresos_total' => $egresos_total,
                    'saldo_actual' => $saldo_actual
                ],
                'movimientos' => $movimientos
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al obtener movimientos',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Resumen de retiros del usuario
     * GET /api/usuarios/{id}/retiros/resumen
     */
    public function resumenRetiros($id)
    {
        try {
            $usuario = Usuario::findOrFail($id);
            
            $retiros = $usuario->retiros();
            
            $resumen = [
                'totales' => [
                    'total_retirado' => $retiros->where('estado', 'completado')->sum('monto'),
                    'total_pendiente' => $retiros->where('estado', 'pendiente')->sum('monto'),
                    'total_rechazado' => $retiros->where('estado', 'rechazado')->sum('monto')
                ],
                'cantidad' => [
                    'completados' => $retiros->where('estado', 'completado')->count(),
                    'pendientes' => $retiros->where('estado', 'pendiente')->count(),
                    'rechazados' => $retiros->where('estado', 'rechazado')->count()
                ],
                'ultimo_retiro' => $retiros->where('estado', 'completado')
                    ->orderBy('created_at', 'desc')
                    ->first()
            ];
            
            return response()->json($resumen);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al obtener resumen de retiros',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Exportar historial de movimientos a PDF
     * GET /api/usuarios/{id}/movimientos/pdf
     */
    public function movimientosPDF($id)
    {
        try {
            $usuario = Usuario::findOrFail($id);
            
            // Obtener los mismos datos que el endpoint movimientos()
            $response = $this->movimientos($id);
            $data = json_decode($response->getContent(), true);
            
            // Generar HTML para el PDF
            $html = $this->generarHTMLMovimientos($usuario, $data);
            
            // Crear PDF con HTML y CSS
            return response($html)
                ->header('Content-Type', 'application/pdf')
                ->header('Content-Disposition', 'attachment; filename="movimientos_' . $usuario->id . '_' . date('Y-m-d') . '.pdf"');
                
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al generar PDF',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Generar HTML formateado para el PDF de movimientos
     */
    private function generarHTMLMovimientos($usuario, $data)
    {
        $resumen = $data['resumen'];
        $movimientos = $data['movimientos'];
        
        $html = '
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Historial de Movimientos - ScaWatts</title>
    <style>
        body { font-family: Arial, sans-serif; font-size: 12px; margin: 20px; }
        .header { text-align: center; margin-bottom: 30px; border-bottom: 2px solid #333; padding-bottom: 10px; }
        .header h1 { margin: 0; color: #2c3e50; }
        .header p { margin: 5px 0; color: #7f8c8d; }
        .resumen { background: #f8f9fa; padding: 15px; margin-bottom: 20px; border-radius: 5px; }
        .resumen table { width: 100%; }
        .resumen td { padding: 8px; }
        .resumen .label { font-weight: bold; width: 150px; }
        .movimientos { margin-top: 20px; }
        .movimientos table { width: 100%; border-collapse: collapse; }
        .movimientos th { background: #34495e; color: white; padding: 10px; text-align: left; }
        .movimientos td { padding: 8px; border-bottom: 1px solid #ddd; }
        .movimientos tr:hover { background: #f5f5f5; }
        .ingreso { color: #27ae60; font-weight: bold; }
        .egreso { color: #e74c3c; font-weight: bold; }
        .footer { margin-top: 30px; text-align: center; font-size: 10px; color: #95a5a6; }
        .total-row { font-weight: bold; background: #ecf0f1; }
    </style>
</head>
<body>
    <div class="header">
        <h1>Historial de Movimientos</h1>
        <p><strong>' . htmlspecialchars($usuario->nombre) . '</strong></p>
        <p>Generado: ' . date('d/m/Y H:i:s') . '</p>
    </div>
    
    <div class="resumen">
        <h2>Resumen Financiero</h2>
        <table>
            <tr>
                <td class="label">Total Ingresos:</td>
                <td class="ingreso">$' . number_format($resumen['ingresos_total'], 0, ',', '.') . '</td>
            </tr>
            <tr>
                <td class="label">Total Egresos:</td>
                <td class="egreso">$' . number_format($resumen['egresos_total'], 0, ',', '.') . '</td>
            </tr>
            <tr class="total-row">
                <td class="label">Saldo Actual:</td>
                <td>$' . number_format($resumen['saldo_actual'], 0, ',', '.') . '</td>
            </tr>
        </table>
    </div>
    
    <div class="movimientos">
        <h2>Detalle de Movimientos (' . count($movimientos) . ' registros)</h2>
        <table>
            <thead>
                <tr>
                    <th>Fecha</th>
                    <th>Concepto</th>
                    <th>Descripción</th>
                    <th>ScaWatt</th>
                    <th style="text-align: right;">Monto</th>
                </tr>
            </thead>
            <tbody>';
        
        foreach ($movimientos as $mov) {
            $clase = $mov['tipo'] === 'ingreso' ? 'ingreso' : 'egreso';
            $signo = $mov['tipo'] === 'ingreso' ? '+' : '-';
            $fecha = date('d/m/Y H:i', strtotime($mov['fecha']));
            $scawatt = $mov['scawatt_id'] ?? '-';
            
            $html .= '
                <tr>
                    <td>' . $fecha . '</td>
                    <td>' . htmlspecialchars($mov['concepto']) . '</td>
                    <td>' . htmlspecialchars($mov['descripcion']) . '</td>
                    <td>' . htmlspecialchars($scawatt) . '</td>
                    <td style="text-align: right;" class="' . $clase . '">' . $signo . '$' . number_format($mov['monto'], 0, ',', '.') . '</td>
                </tr>';
        }
        
        $html .= '
            </tbody>
        </table>
    </div>
    
    <div class="footer">
        <p>ScaWatts - Sistema de Gestión de Inversiones en Energía Solar</p>
        <p>Este documento es un registro oficial de movimientos financieros</p>
    </div>
</body>
</html>';
        
        return $html;
    }

    /**
     * Estadísticas generales de usuarios
     * GET /api/admin/usuarios/estadisticas
     */
    public function estadisticas()
    {
        $stats = [
            'total_usuarios' => Usuario::count(),
            'usuarios_activos' => Usuario::where('estado', 'activo')->count(),
            'usuarios_inactivos' => Usuario::where('estado', 'inactivo')->count(),
            'usuarios_bloqueados' => Usuario::where('estado', 'bloqueado')->count(),
            'usuarios_con_scawatts' => Usuario::has('scawatts')->count(),
            'total_saldo_sistema' => Usuario::sum('saldo_disponible')
        ];

        return response()->json($stats);
    }

    /**
     * Generar contraseña aleatoria segura
     */
    private function generarPasswordAleatoria(): string
    {
        // Generar password de 10 caracteres: letras mayúsculas, minúsculas y números
        $caracteres = 'ABCDEFGHJKLMNPQRSTUVWXYZabcdefghijkmnpqrstuvwxyz23456789';
        $password = '';
        $longitudCaracteres = strlen($caracteres);
        
        for ($i = 0; $i < 10; $i++) {
            $password .= $caracteres[random_int(0, $longitudCaracteres - 1)];
        }
        
        return $password;
    }

    /**
     * Enviar contraseña por correo electrónico
     */
    private function enviarPasswordPorEmail($usuario, $password)
    {
        // TODO: Implementar envío de email real
        // Por ahora, registrar en los logs
        \Log::info('Password generada para usuario', [
            'usuario_id' => $usuario->id,
            'nombre' => $usuario->nombre,
            'email' => $usuario->email,
            'password' => $password
        ]);

        // Crear una notificación en el sistema
        try {
            \App\Models\Notificacion::create([
                'usuario_id' => $usuario->id,
                'tipo' => 'sistema',
                'titulo' => 'Bienvenido a ScaWatts',
                'contenido' => "Tu cuenta ha sido creada exitosamente. Tu contraseña es: {$password}. Por seguridad, te recomendamos cambiarla después de tu primer inicio de sesión.",
                'leido' => false,
                'icono' => 'key'
            ]);
        } catch (\Exception $e) {
            \Log::warning('No se pudo crear notificación de bienvenida: ' . $e->getMessage());
        }
    }

    /**
     * Estructurar la respuesta del perfil del usuario para consumo del frontend
     */
    private function formatearPerfilUsuario(Usuario $usuario): array
    {
        $numeroCuenta = $usuario->numero_cuenta ? (string) $usuario->numero_cuenta : null;
        $cuenta = null;

        if ($numeroCuenta) {
            $mascara = strlen($numeroCuenta) >= 4
                ? '****' . substr($numeroCuenta, -4)
                : $numeroCuenta;

            $cuenta = [
                'banco' => $usuario->banco,
                'tipo_cuenta' => $usuario->tipo_cuenta,
                'numero_cuenta' => $numeroCuenta,
                'numero_cuenta_parcial' => $mascara,
                'estado' => $usuario->datos_bancarios_verificados_at ? 'verificado' : 'pendiente',
                'verificado' => (bool) $usuario->datos_bancarios_verificados_at,
                'verificado_fecha' => $usuario->datos_bancarios_verificados_at,
            ];
        }

        return [
            'id' => $usuario->id,
            'nombre' => $usuario->nombre,
            'email' => $usuario->email,
            'telefono' => $usuario->telefono,
            'picture' => $usuario->picture,
            'estado' => $usuario->estado,
            'fecha_registro' => $usuario->created_at,
            'actualizado_at' => $usuario->updated_at,
            'saldo' => [
                'disponible' => $usuario->saldo_disponible,
                'pendiente' => $usuario->saldo_pendiente,
                'total' => $usuario->saldo_total,
            ],
            'verificaciones' => [
                'cedula' => (bool) $usuario->cedula_verificada_at,
                'datos_bancarios' => (bool) $usuario->datos_bancarios_verificados_at,
            ],
            'cuentas_bancarias' => $cuenta ? [$cuenta] : [],
        ];
    }

    /**
     * Obtener valorización de ScaWatts del usuario
     * GET /api/usuarios/{id}/valorizacion
     */
    public function valorizacion($id)
    {
        try {
            $usuario = Usuario::findOrFail($id);
            
            // Obtener todos los scawatts activos del usuario
            $scawatts = $usuario->scawatts()->where('estado', 'activo')->get();
            
            // Calcular totales reales
            $totalScawatts = $scawatts->count();
            $valorInicialTotal = $scawatts->sum('valor_inicial');
            $valorActualTotal = $scawatts->sum('valor_actual');
            $valorizacionTotal = $scawatts->sum('valorizacion_total');
            
            // DEVOLVER DATOS DIRECTOS SIN WRAPPER (formato original)
            return response()->json([
                'total_scawatts' => (int) $totalScawatts,
                'valor_inicial_total' => round((float) $valorInicialTotal, 2),
                'valor_actual_total' => round((float) $valorActualTotal, 2),
                'valorizacion_total' => round((float) $valorizacionTotal, 2),
                'valorizacion_porcentaje' => $valorInicialTotal > 0 ? round(($valorizacionTotal / $valorInicialTotal) * 100, 2) : 0,
                'scawatts' => $scawatts->map(function($s) {
                    return [
                        'id' => $s->id,
                        'scawatt_id' => $s->scawatt_id,
                        'valor_inicial' => round((float) $s->valor_inicial, 2),
                        'valor_actual' => round((float) $s->valor_actual, 2),
                        'valorizacion' => round((float) $s->valorizacion_total, 2),
                        'kwh_asignados' => (float) $s->kwh_asignados,
                        'fecha_inicio' => $s->fecha_inicio,
                        'fecha_final' => $s->fecha_final
                    ];
                })
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al obtener valorización',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}

