<?php

namespace App\Http\Controllers;

use App\Models\Usuario;
use App\Models\Scawatt;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;

class WebController extends Controller
{
    /**
     * Mostrar formulario de login
     */
    public function showLogin()
    {
        if (Auth::check()) {
            return redirect()->route('dashboard');
        }
        
        return view('auth.login');
    }
    
    /**
     * Procesar login
     */
    public function login(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required|min:6'
        ]);
        
        $usuario = Usuario::where('email', $request->email)->first();
        
        if (!$usuario || !Hash::check($request->password, $usuario->password)) {
            return back()->withErrors([
                'email' => 'Las credenciales no coinciden con nuestros registros.'
            ])->withInput();
        }
        
        if ($usuario->estado !== 'activo') {
            return back()->withErrors([
                'email' => 'Tu cuenta está ' . $usuario->estado . '. Contacta al administrador.'
            ])->withInput();
        }
        
        Auth::login($usuario, $request->filled('remember'));
        
        $request->session()->regenerate();
        
        return redirect()->intended(route('dashboard'));
    }
    
    /**
     * Cerrar sesión
     */
    public function logout(Request $request)
    {
        Auth::logout();
        
        $request->session()->invalidate();
        $request->session()->regenerateToken();
        
        return redirect()->route('login');
    }
    
    /**
     * Dashboard principal
     */
    public function dashboard()
    {
        // Estadísticas generales
        $stats = [
            'total_usuarios' => Usuario::count(),
            'usuarios_activos' => Usuario::where('estado', 'activo')->count(),
            'total_scawatts' => Scawatt::count(),
            'scawatts_activos' => Scawatt::where('estado', 'activo')->count(),
            'inversion_total' => Scawatt::sum('valor_inicial'),
            'valor_actual' => Scawatt::sum('valor_actual'),
            'valorizacion_total' => Scawatt::sum('valor_actual') - Scawatt::sum('valor_inicial')
        ];
        
        // Usuarios recientes
        $recent_usuarios = Usuario::latest()->limit(5)->get();
        
        // ScaWatts recientes
        $recent_scawatts = Scawatt::with('usuario')->latest()->limit(5)->get();
        
        return view('dashboard', compact('stats', 'recent_usuarios', 'recent_scawatts'));
    }
}
