<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Configuracion extends Model
{
    use HasFactory;

    protected $table = 'configuracion';

    protected $fillable = [
        'clave',
        'valor',
        'tipo',
        'grupo',
        'descripcion',
        'editable'
    ];

    protected $casts = [
        'editable' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    /**
     * Obtener valor de configuración
     */
    public static function obtener($clave, $default = null)
    {
        $config = self::where('clave', $clave)->first();
        
        if (!$config) {
            return $default;
        }

        // Convertir según tipo
        return self::convertirValor($config->valor, $config->tipo);
    }

    /**
     * Establecer valor de configuración
     */
    public static function establecer($clave, $valor, $tipo = 'string')
    {
        return self::updateOrCreate(
            ['clave' => $clave],
            [
                'valor' => $valor,
                'tipo' => $tipo
            ]
        );
    }

    /**
     * Obtener todas las configuraciones de un grupo
     */
    public static function obtenerGrupo($grupo)
    {
        return self::where('grupo', $grupo)->get()->mapWithKeys(function ($config) {
            return [
                $config->clave => self::convertirValor($config->valor, $config->tipo)
            ];
        });
    }

    /**
     * Convertir valor según tipo
     */
    private static function convertirValor($valor, $tipo)
    {
        switch ($tipo) {
            case 'integer':
                return (int) $valor;
            case 'float':
            case 'decimal':
                return (float) $valor;
            case 'boolean':
                return filter_var($valor, FILTER_VALIDATE_BOOLEAN);
            case 'json':
            case 'array':
                return json_decode($valor, true);
            default:
                return $valor;
        }
    }

    /**
     * Scope por grupo
     */
    public function scopePorGrupo($query, $grupo)
    {
        return $query->where('grupo', $grupo);
    }

    /**
     * Scope editables
     */
    public function scopeEditables($query)
    {
        return $query->where('editable', true);
    }
}
