<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Deposito extends Model
{
    use HasFactory;

    protected $table = 'depositos';

    protected $fillable = [
        'usuario_id',
        'monto',
        'metodo_pago',
        'referencia_pago',
        'comprobante_ruta',
        'estado',
        'fecha_aprobacion',
        'aprobado_por',
        'notas_admin',
        'datos_pago'
    ];

    protected $casts = [
        'monto' => 'decimal:2',
        'datos_pago' => 'array',
        'fecha_aprobacion' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    /**
     * Relación con Usuario
     */
    public function usuario()
    {
        return $this->belongsTo(Usuario::class, 'usuario_id');
    }

    /**
     * Relación con admin que aprobó
     */
    public function aprobador()
    {
        return $this->belongsTo(Usuario::class, 'aprobado_por');
    }

    /**
     * Scope por estado
     */
    public function scopePorEstado($query, $estado)
    {
        return $query->where('estado', $estado);
    }

    /**
     * Scope pendientes
     */
    public function scopePendientes($query)
    {
        return $query->where('estado', 'pendiente');
    }

    /**
     * Scope aprobados
     */
    public function scopeAprobados($query)
    {
        return $query->where('estado', 'aprobado');
    }

    /**
     * Scope rechazados
     */
    public function scopeRechazados($query)
    {
        return $query->where('estado', 'rechazado');
    }

    /**
     * Aprobar depósito
     */
    public function aprobar($adminId, $notas = null)
    {
        $this->update([
            'estado' => 'aprobado',
            'fecha_aprobacion' => now(),
            'aprobado_por' => $adminId,
            'notas_admin' => $notas
        ]);

        // Actualizar saldo del usuario
        $this->usuario->increment('saldo_disponible', $this->monto);
        $this->usuario->increment('saldo_total', $this->monto);
    }

    /**
     * Rechazar depósito
     */
    public function rechazar($adminId, $motivo)
    {
        $this->update([
            'estado' => 'rechazado',
            'fecha_aprobacion' => now(),
            'aprobado_por' => $adminId,
            'notas_admin' => $motivo
        ]);
    }
}
