<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Granja extends Model
{
    use HasFactory;

    protected $table = 'granjas';

    protected $fillable = [
        'nombre',
        'categoria',
        'departamento',
        'municipio',
        'direccion_completa',
        'latitud',
        'longitud',
        'capacidad_mw',
        'produccion_estimada_anual',
        'fecha_inicio_operacion',
        'vida_util_años',
        'fecha_final_operacion',
        'tasa_valorizacion_anual',
        'ganancia_mensual_base',
        'estado',
        'descripcion',
        'imagen_url'
    ];

    protected $casts = [
        'capacidad_mw' => 'decimal:2',
        'produccion_estimada_anual' => 'decimal:2',
        'tasa_valorizacion_anual' => 'decimal:2',
        'ganancia_mensual_base' => 'decimal:2',
        'vida_util_años' => 'integer',
        'latitud' => 'decimal:7',
        'longitud' => 'decimal:7',
        'fecha_inicio_operacion' => 'date',
        'fecha_final_operacion' => 'date',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    /**
     * Relación con ScaWatts
     */
    public function scawatts()
    {
        return $this->hasMany(Scawatt::class, 'granja_id');
    }

    /**
     * Scope para granjas operativas
     */
    public function scopeOperativas($query)
    {
        return $query->where('estado', 'operativa');
    }

    /**
     * Scope por categoría
     */
    public function scopeCategoria($query, $categoria)
    {
        return $query->where('categoria', $categoria);
    }

    /**
     * Scope por departamento
     */
    public function scopeDepartamento($query, $departamento)
    {
        return $query->where('departamento', $departamento);
    }

    /**
     * Obtener categoría formateada
     */
    public function getCategoriaFormateadaAttribute()
    {
        return match($this->categoria) {
            'granja_solar' => 'Granja Solar',
            'parque_solar' => 'Parque Solar',
            'industrial' => 'Industrial',
            'domestico' => 'Doméstico',
            'vehicular' => 'Vehicular',
            default => ucfirst($this->categoria)
        };
    }

    /**
     * Obtener estado formateado
     */
    public function getEstadoFormateadoAttribute()
    {
        return match($this->estado) {
            'operativa' => 'Operativa',
            'construccion' => 'En Construcción',
            'planificada' => 'Planificada',
            'mantenimiento' => 'En Mantenimiento',
            default => ucfirst($this->estado)
        };
    }
}