<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

class Marketplace extends Model
{
    use HasFactory;

    protected $table = 'marketplace';

    protected $fillable = [
        'scawatt_id',
        'vendedor_id',
        'comprador_id',
        'precio_venta',
        'estado',
        'fecha_publicacion',
        'fecha_venta',
        'comision_plataforma',
        'descripcion',
        'destacado'
    ];

    protected $casts = [
        'precio_venta' => 'decimal:2',
        'comision_plataforma' => 'decimal:2',
        'fecha_publicacion' => 'datetime',
        'fecha_venta' => 'datetime',
        'destacado' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    /**
     * Relación con ScaWatt
     */
    public function scawatt()
    {
        return $this->belongsTo(Scawatt::class, 'scawatt_id');
    }

    /**
     * Relación con Usuario vendedor
     */
    public function vendedor()
    {
        return $this->belongsTo(Usuario::class, 'vendedor_id');
    }

    /**
     * Relación con Usuario comprador
     */
    public function comprador()
    {
        return $this->belongsTo(Usuario::class, 'comprador_id');
    }

    /**
     * Scope disponibles
     */
    public function scopeDisponibles($query)
    {
        return $query->where('estado', 'en_venta')
                     ->whereNull('comprador_id');
    }

    /**
     * Scope vendidos
     */
    public function scopeVendidos($query)
    {
        return $query->where('estado', 'vendido')
                     ->whereNotNull('comprador_id');
    }

    /**
     * Scope destacados
     */
    public function scopeDestacados($query)
    {
        return $query->where('destacado', true)
                     ->where('estado', 'en_venta');
    }

    /**
     * Publicar en marketplace
     */
    public function publicar()
    {
        $this->update([
            'estado' => 'en_venta',
            'fecha_publicacion' => now()
        ]);

        // Actualizar estado del ScaWatt
        $this->scawatt->update(['estado' => 'marketplace']);
    }

    /**
     * Retirar de marketplace
     */
    public function retirar()
    {
        $this->update([
            'estado' => 'cancelado'
        ]);

        // Actualizar estado del ScaWatt
        $this->scawatt->update(['estado' => 'activo']);
    }

    /**
     * Procesar venta
     */
    public function vender($compradorId)
    {
        // Calcular comisión (ejemplo: 5%)
        $comision = $this->precio_venta * 0.05;
        $montoVendedor = $this->precio_venta - $comision;

        $this->update([
            'estado' => 'vendido',
            'comprador_id' => $compradorId,
            'fecha_venta' => now(),
            'comision_plataforma' => $comision
        ]);

        // Transferir saldo al vendedor
        if ($vendedor = $this->vendedor()->lockForUpdate()->first()) {
            $vendedor->increment('saldo_disponible', $montoVendedor);
        }

        // Descontar saldo del comprador
        if ($comprador = $this->comprador()->lockForUpdate()->first()) {
            $comprador->decrement('saldo_disponible', $this->precio_venta);
        }

        // Transferir propiedad del ScaWatt y registrar la trazabilidad básica
        if ($scawatt = $this->scawatt()->lockForUpdate()->first()) {
            $scawatt->forceFill([
                'usuario_id' => $compradorId,
                'estado' => 'activo'
            ])->save();

            DB::table('propietarios')
                ->where('scawatt_id', $scawatt->id)
                ->where('es_propietario_actual', true)
                ->update([
                    'es_propietario_actual' => false,
                    'updated_at' => now()
                ]);

            DB::table('propietarios')->insert([
                'scawatt_id' => $scawatt->id,
                'usuario_id' => $compradorId,
                'fecha_inicio' => now(),
                'tipo_adquisicion' => 'marketplace',
                'precio_adquisicion' => $this->precio_venta,
                'es_propietario_actual' => true,
                'created_at' => now(),
                'updated_at' => now()
            ]);
        }

        return $comision;
    }
}
