<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Mensaje extends Model
{
    use HasFactory;

    protected $table = 'mensajes';

    // Campos reales de la tabla mensajes
    protected $fillable = [
        'titulo',
        'contenido',
        'tipo',
        'destinatario',
        'usuario_id',
        'fecha_programada',
        'fecha_enviado',
        'enviado',
        'creado_por'
    ];

    protected $casts = [
        'fecha_programada' => 'datetime',
        'fecha_enviado' => 'datetime',
        'enviado' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    /**
     * Relación con Usuario destinatario (para mensajes individuales)
     */
    public function usuario()
    {
        return $this->belongsTo(Usuario::class, 'usuario_id');
    }

    /**
     * Relación con Usuario creador
     */
    public function creador()
    {
        return $this->belongsTo(Usuario::class, 'creado_por');
    }

    /**
     * Scope para mensajes de un usuario específico
     */
    public function scopeParaUsuario($query, $usuarioId)
    {
        return $query->where(function($q) use ($usuarioId) {
            $q->where('destinatario', 'todos')
              ->orWhere(function($q2) use ($usuarioId) {
                  $q2->where('destinatario', 'individual')
                     ->where('usuario_id', $usuarioId);
              });
        });
    }

    /**
     * Scope para mensajes enviados
     */
    public function scopeEnviados($query)
    {
        return $query->where('enviado', true);
    }

    /**
     * Scope para mensajes pendientes
     */
    public function scopePendientes($query)
    {
        return $query->where('enviado', false);
    }

    /**
     * Scope por tipo
     */
    public function scopeTipo($query, $tipo)
    {
        return $query->where('tipo', $tipo);
    }

    /**
     * Marcar como enviado
     */
    public function marcarEnviado()
    {
        $this->update([
            'enviado' => true,
            'fecha_enviado' => now()
        ]);
    }
}
