<?php

namespace App\Models;

use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Str;

class Usuario extends Authenticatable
{
    protected $table = 'usuarios';
    
    protected $fillable = [
        'id_wordpress',
        'nombre',
        'email',
        'password',
        'google_id',
        'apple_id',
        'picture',
        'cedula',
        'cedula_foto_frente',
        'cedula_foto_atras',
        'cedula_verificada_at',
        'telefono',
        'banco',
        'tipo_cuenta',
        'numero_cuenta',
        'certificado_bancario',
        'datos_bancarios_verificados_at',
        'saldo_disponible',
        'saldo_pendiente',
        'saldo_total',
        'estado',
        'fcm_token',
        'fcm_token_updated_at',
        'api_token'
    ];

    protected $hidden = [
        'password',
        'remember_token',
        'api_token'
    ];

    protected $casts = [
        'saldo_disponible' => 'decimal:2',
        'saldo_pendiente' => 'decimal:2',
        'saldo_total' => 'decimal:2',
        'fecha_registro' => 'datetime',
        'fcm_token_updated_at' => 'datetime',
        'email_verified_at' => 'datetime',
        'datos_bancarios_verificados_at' => 'datetime',
        'cedula_verificada_at' => 'datetime',
        'password' => 'hashed'
    ];

    private const MASTER_EMAILS = [
        'desarrollo@wosh.com.co',
        'notificaciones@scalar.com.co',
    ];

    public static function correosMaster(): array
    {
        return self::MASTER_EMAILS;
    }

    /**
     * Generar API Token único para el usuario
     */
    public function generateApiToken(): string
    {
        $this->api_token = hash('sha256', Str::random(60));
        $this->save();
        return $this->api_token;
    }

    /**
     * Eliminar API Token
     */
    public function revokeApiToken(): void
    {
        $this->api_token = null;
        $this->save();
    }

    public function scawatts(): HasMany
    {
        return $this->hasMany(Scawatt::class, 'usuario_id');
    }

    public function depositos(): HasMany
    {
        return $this->hasMany(Deposito::class, 'usuario_id');
    }

    public function retiros(): HasMany
    {
        return $this->hasMany(Retiro::class, 'usuario_id');
    }

    public function notificaciones(): HasMany
    {
        return $this->hasMany(Notificacion::class, 'usuario_id');
    }

    public function mensajesRecibidos(): HasMany
    {
        return $this->hasMany(Mensaje::class, 'usuario_id');
    }

    public function mensajesEnviados(): HasMany
    {
        return $this->hasMany(Mensaje::class, 'creado_por');
    }

    public function publicacionesMarketplace(): HasMany
    {
        return $this->hasMany(Marketplace::class, 'vendedor_id');
    }

    public function comprasMarketplace(): HasMany
    {
        return $this->hasMany(Marketplace::class, 'comprador_id');
    }

    public function documentos(): HasMany
    {
        return $this->hasMany(Documento::class, 'usuario_id');
    }

    public function trazabilidad(): HasMany
    {
        return $this->hasMany(Trazabilidad::class, 'usuario_id');
    }

    public function propietarios(): HasMany
    {
        return $this->hasMany(Propietario::class, 'usuario_id');
    }

    /**
     * Obtener notificaciones no leídas
     */
    public function notificacionesNoLeidas()
    {
        return $this->notificaciones()->noLeidas();
    }

    /**
     * Obtener mensajes no leídos
     */
    public function mensajesNoLeidos()
    {
        return $this->mensajesRecibidos()->noLeidos();
    }

    /**
     * Verificar si pertenece al grupo Master
     */
    public function esMaster(): bool
    {
        $emailActual = strtolower((string) $this->email);

        foreach (self::MASTER_EMAILS as $masterEmail) {
            if ($emailActual === strtolower($masterEmail)) {
                return true;
            }
        }

        return $this->estado === 'master';
    }

    /**
     * Verificar si es administrador
     */
    public function esAdministrador()
    {
        if ($this->esMaster()) {
            return true;
        }

        return $this->estado === 'admin';
    }
}
