@extends('layouts.app')

@section('title', 'Transacciones')
@section('page-title', 'Transacciones y tesorería')
@section('page-subtitle', 'Control total de depósitos, retiros, conciliaciones y saldos operativos')

@section('page-actions')
<div class="d-flex gap-2">
    <a href="{{ route('admin.usuarios') }}" class="btn btn-sm btn-outline-primary">
        <i class="fas fa-users me-1"></i> Usuarios
    </a>
    <a href="{{ route('admin.marketplace') }}" class="btn btn-sm btn-secondary">
        <i class="fas fa-store me-1"></i> Marketplace
    </a>
</div>
@endsection

@php
    $defaultAdminId = auth()->id();
@endphp

<div class="modal fade" id="modalAprobarDeposito" tabindex="-1" aria-labelledby="modalAprobarDepositoLabel" aria-hidden="true" data-default-admin-id="{{ $defaultAdminId }}">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <form method="post" action="#" id="formAprobarDeposito">
                @csrf
                @method('PATCH')
                <div class="modal-header">
                    <h5 class="modal-title" id="modalAprobarDepositoLabel">
                        <i class="fas fa-circle-check text-success me-2"></i>Aprobar depósito
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Cerrar"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <div class="small text-muted text-uppercase mb-1">Resumen</div>
                        <p class="mb-0">
                            <strong id="depositoAprobarUsuario">Usuario seleccionado</strong><br>
                            <span id="depositoAprobarMonto" class="text-success">$0</span>
                            · <span id="depositoAprobarMetodo">Método</span><br>
                            <small class="text-muted">Referencia: <span id="depositoAprobarReferencia">N/A</span></small>
                        </p>
                    </div>
                    <div class="mb-3">
                        <label for="depositoAprobarAdmin" class="form-label">Administrador responsable</label>
                        <select id="depositoAprobarAdmin" name="admin_id" class="form-select" required @disabled($admins->isEmpty())>
                            <option value="">Selecciona un administrador</option>
                            @foreach($admins as $admin)
                                <option value="{{ $admin->id }}">{{ $admin->nombre }} — {{ $admin->email }}@if($defaultAdminId === $admin->id) (Tú)@endif</option>
                            @endforeach
                        </select>
                        @if($admins->isEmpty())
                            <div class="form-text text-danger">Debes registrar al menos un usuario con estado administrador para asignar la acción.</div>
                        @endif
                    </div>
                    <div class="mb-0">
                        <label for="depositoAprobarNotas" class="form-label">Notas internas (opcional)</label>
                        <textarea id="depositoAprobarNotas" name="notas_admin" class="form-control" rows="3" placeholder="Detalle de conciliación o referencia bancaria"></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-success" @disabled($admins->isEmpty())>Aprobar depósito</button>
                </div>
            </form>
        </div>
    </div>
</div>

<div class="modal fade" id="modalRechazarDeposito" tabindex="-1" aria-labelledby="modalRechazarDepositoLabel" aria-hidden="true" data-default-admin-id="{{ $defaultAdminId }}">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <form method="post" action="#" id="formRechazarDeposito">
                @csrf
                @method('PATCH')
                <div class="modal-header">
                    <h5 class="modal-title" id="modalRechazarDepositoLabel">
                        <i class="fas fa-circle-xmark text-danger me-2"></i>Rechazar depósito
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Cerrar"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <div class="small text-muted text-uppercase mb-1">Resumen</div>
                        <p class="mb-0">
                            <strong id="depositoRechazarUsuario">Usuario seleccionado</strong><br>
                            <span id="depositoRechazarMonto" class="text-danger">$0</span>
                            · <span id="depositoRechazarMetodo">Método</span><br>
                            <small class="text-muted">Referencia: <span id="depositoRechazarReferencia">N/A</span></small>
                        </p>
                    </div>
                    <div class="mb-3">
                        <label for="depositoRechazarAdmin" class="form-label">Administrador responsable</label>
                        <select id="depositoRechazarAdmin" name="admin_id" class="form-select" required @disabled($admins->isEmpty())>
                            <option value="">Selecciona un administrador</option>
                            @foreach($admins as $admin)
                                <option value="{{ $admin->id }}">{{ $admin->nombre }} — {{ $admin->email }}@if($defaultAdminId === $admin->id) (Tú)@endif</option>
                            @endforeach
                        </select>
                        @if($admins->isEmpty())
                            <div class="form-text text-danger">Debes registrar un administrador para continuar.</div>
                        @endif
                    </div>
                    <div class="mb-0">
                        <label for="depositoRechazarMotivo" class="form-label">Motivo del rechazo</label>
                        <textarea id="depositoRechazarMotivo" name="motivo" class="form-control" rows="3" required placeholder="Describe brevemente la razón del rechazo"></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-danger" @disabled($admins->isEmpty())>Rechazar depósito</button>
                </div>
            </form>
        </div>
    </div>
</div>

<div class="modal fade" id="modalAprobarRetiro" tabindex="-1" aria-labelledby="modalAprobarRetiroLabel" aria-hidden="true" data-default-admin-id="{{ $defaultAdminId }}">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <form method="post" action="#" id="formAprobarRetiro">
                @csrf
                @method('PATCH')
                <div class="modal-header">
                    <h5 class="modal-title" id="modalAprobarRetiroLabel">
                        <i class="fas fa-circle-check text-success me-2"></i>Procesar retiro
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Cerrar"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <div class="small text-muted text-uppercase mb-1">Resumen</div>
                        <p class="mb-0">
                            <strong id="retiroAprobarUsuario">Usuario seleccionado</strong><br>
                            <span id="retiroAprobarMonto" class="text-success">$0</span><br>
                            <small class="text-muted">Cuenta destino: <span id="retiroAprobarDestino">N/A</span></small>
                        </p>
                    </div>
                    <div class="mb-3">
                        <label for="retiroAprobarAdmin" class="form-label">Administrador responsable</label>
                        <select id="retiroAprobarAdmin" name="admin_id" class="form-select" required @disabled($admins->isEmpty())>
                            <option value="">Selecciona un administrador</option>
                            @foreach($admins as $admin)
                                <option value="{{ $admin->id }}">{{ $admin->nombre }} — {{ $admin->email }}@if($defaultAdminId === $admin->id) (Tú)@endif</option>
                            @endforeach
                        </select>
                        @if($admins->isEmpty())
                            <div class="form-text text-danger">Registra un administrador para poder asignarlo al retiro.</div>
                        @endif
                    </div>
                    <div class="mb-3">
                        <label for="retiroAprobarReferencia" class="form-label">Referencia de la transacción</label>
                        <input type="text" id="retiroAprobarReferencia" name="referencia_transaccion" class="form-control" required placeholder="Ej: TRANS-123456">
                    </div>
                    <div class="mb-0">
                        <label for="retiroAprobarNotas" class="form-label">Notas internas (opcional)</label>
                        <textarea id="retiroAprobarNotas" name="notas_admin" class="form-control" rows="3" placeholder="Detalle del movimiento bancario"></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-success" @disabled($admins->isEmpty())>Confirmar pago</button>
                </div>
            </form>
        </div>
    </div>
</div>

<div class="modal fade" id="modalRetiroProceso" tabindex="-1" aria-labelledby="modalRetiroProcesoLabel" aria-hidden="true" data-default-admin-id="{{ $defaultAdminId }}">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <form method="post" action="#" id="formRetiroProceso">
                @csrf
                @method('PATCH')
                <div class="modal-header">
                    <h5 class="modal-title" id="modalRetiroProcesoLabel">
                        <i class="fas fa-clock text-info me-2"></i>Marcar retiro en proceso
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Cerrar"></button>
                </div>
                <div class="modal-body">
                    <p class="mb-3">
                        Estás marcando el retiro de <strong id="retiroProcesoUsuario">Usuario seleccionado</strong> por
                        <span id="retiroProcesoMonto" class="text-primary">$0</span> como "en proceso".
                        Usa este estado cuando ya estás conciliando el pago pero aún no lo completas.
                    </p>
                    <div class="mb-3">
                        <label for="retiroProcesoAdmin" class="form-label">Administrador responsable</label>
                        <select id="retiroProcesoAdmin" name="admin_id" class="form-select" required @disabled($admins->isEmpty())>
                            <option value="">Selecciona un administrador</option>
                            @foreach($admins as $admin)
                                <option value="{{ $admin->id }}">{{ $admin->nombre }} — {{ $admin->email }}@if($defaultAdminId === $admin->id) (Tú)@endif</option>
                            @endforeach
                        </select>
                        @if($admins->isEmpty())
                            <div class="form-text text-danger">Sin administradores registrados no es posible continuar.</div>
                        @endif
                    </div>
                    <div class="mb-0">
                        <label for="retiroProcesoNotas" class="form-label">Notas internas (opcional)</label>
                        <textarea id="retiroProcesoNotas" name="notas_admin" class="form-control" rows="3" placeholder="Detalle el estado del pago o la entidad bancaria"></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-info text-white" @disabled($admins->isEmpty())>Marcar en proceso</button>
                </div>
            </form>
        </div>
    </div>
</div>

<div class="modal fade" id="modalRechazarRetiro" tabindex="-1" aria-labelledby="modalRechazarRetiroLabel" aria-hidden="true" data-default-admin-id="{{ $defaultAdminId }}">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <form method="post" action="#" id="formRechazarRetiro">
                @csrf
                @method('PATCH')
                <div class="modal-header">
                    <h5 class="modal-title" id="modalRechazarRetiroLabel">
                        <i class="fas fa-circle-xmark text-danger me-2"></i>Rechazar retiro
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Cerrar"></button>
                </div>
                <div class="modal-body">
                    <p class="mb-3">
                        Este retiro pertenece a <strong id="retiroRechazarUsuario">Usuario seleccionado</strong> por
                        <span id="retiroRechazarMonto" class="text-danger">$0</span>. El saldo se reintegrará automáticamente al marcarlo como rechazado.
                    </p>
                    <div class="mb-3">
                        <label for="retiroRechazarAdmin" class="form-label">Administrador responsable</label>
                        <select id="retiroRechazarAdmin" name="admin_id" class="form-select" required @disabled($admins->isEmpty())>
                            <option value="">Selecciona un administrador</option>
                            @foreach($admins as $admin)
                                <option value="{{ $admin->id }}">{{ $admin->nombre }} — {{ $admin->email }}@if($defaultAdminId === $admin->id) (Tú)@endif</option>
                            @endforeach
                        </select>
                        @if($admins->isEmpty())
                            <div class="form-text text-danger">Debes asignar un administrador para registrar la acción.</div>
                        @endif
                    </div>
                    <div class="mb-0">
                        <label for="retiroRechazarMotivo" class="form-label">Motivo del rechazo</label>
                        <textarea id="retiroRechazarMotivo" name="motivo" class="form-control" rows="3" required placeholder="Ej: Datos bancarios incompletos"></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-danger" @disabled($admins->isEmpty())>Rechazar retiro</button>
                </div>
            </form>
        </div>
    </div>
</div>

@section('content')
@php
    $estadoDepositos = [
        'todos' => 'Todos los depósitos',
        'pendiente' => 'Pendientes',
        'aprobado' => 'Aprobados',
        'rechazado' => 'Rechazados',
    ];

    $estadoRetiros = [
        'todos' => 'Todos los retiros',
        'pendiente' => 'Pendientes',
        'en_proceso' => 'En proceso',
        'completado' => 'Completados',
        'rechazado' => 'Rechazados',
    ];

    $ordenOptions = [
        'recientes' => 'Más recientes',
        'antiguedad' => 'Más antiguos',
        'monto_desc' => 'Monto alto a bajo',
        'monto_asc' => 'Monto bajo a alto',
    ];

    $estadoBadges = [
        'pendiente' => 'bg-warning text-dark',
        'aprobado' => 'bg-success',
        'rechazado' => 'bg-danger',
        'en_proceso' => 'bg-info text-dark',
        'completado' => 'bg-primary',
    ];
@endphp

@if($admins->isEmpty())
    <div class="alert alert-warning d-flex align-items-center gap-2 mb-3">
        <i class="fas fa-triangle-exclamation fa-lg"></i>
        <div>
            <strong>Sin administradores asignables.</strong> Registra al menos un usuario con estado <em>admin</em> para poder aprobar, rechazar o marcar transacciones desde este módulo.
        </div>
    </div>
@endif

<div class="row g-3 mb-1">
    <div class="col-xl-2 col-lg-3 col-md-4 col-sm-6">
        <div class="card shadow-sm border-0 h-100">
            <div class="card-body">
                <div class="d-flex justify-content-between align-items-start">
                    <div>
                        <p class="text-muted text-uppercase small mb-1">Depósitos pendientes</p>
                        <h4 class="fw-semibold mb-0">{{ number_format($stats['depositos_pendientes']) }}</h4>
                    </div>
                    <span class="icon-circle bg-warning-subtle text-warning"><i class="fas fa-inbox"></i></span>
                </div>
            </div>
        </div>
    </div>
    <div class="col-xl-2 col-lg-3 col-md-4 col-sm-6">
        <div class="card shadow-sm border-0 h-100">
            <div class="card-body">
                <div class="d-flex justify-content-between align-items-start">
                    <div>
                        <p class="text-muted text-uppercase small mb-1">Valor pendiente</p>
                        <h4 class="fw-semibold mb-0">{{ $stats['depositos_pendientes_monto'] }}</h4>
                    </div>
                    <span class="icon-circle bg-primary-subtle text-primary"><i class="fas fa-sack-dollar"></i></span>
                </div>
            </div>
        </div>
    </div>
    <div class="col-xl-2 col-lg-3 col-md-4 col-sm-6">
        <div class="card shadow-sm border-0 h-100">
            <div class="card-body">
                <div class="d-flex justify-content-between align-items-start">
                    <div>
                        <p class="text-muted text-uppercase small mb-1">Depósitos mes</p>
                        <h4 class="fw-semibold mb-0">{{ $stats['depositos_mes'] }}</h4>
                    </div>
                    <span class="icon-circle bg-success-subtle text-success"><i class="fas fa-arrow-trend-up"></i></span>
                </div>
            </div>
        </div>
    </div>
    <div class="col-xl-2 col-lg-3 col-md-4 col-sm-6">
        <div class="card shadow-sm border-0 h-100">
            <div class="card-body">
                <div class="d-flex justify-content-between align-items-start">
                    <div>
                        <p class="text-muted text-uppercase small mb-1">Retiros pendientes</p>
                        <h4 class="fw-semibold mb-0">{{ number_format($stats['retiros_pendientes']) }}</h4>
                    </div>
                    <span class="icon-circle bg-danger-subtle text-danger"><i class="fas fa-hourglass-half"></i></span>
                </div>
            </div>
        </div>
    </div>
    <div class="col-xl-2 col-lg-3 col-md-4 col-sm-6">
        <div class="card shadow-sm border-0 h-100">
            <div class="card-body">
                <div class="d-flex justify-content-between align-items-start">
                    <div>
                        <p class="text-muted text-uppercase small mb-1">Valor por liberar</p>
                        <h4 class="fw-semibold mb-0">{{ $stats['retiros_pendientes_monto'] }}</h4>
                    </div>
                    <span class="icon-circle bg-secondary-subtle text-secondary"><i class="fas fa-scale-balanced"></i></span>
                </div>
            </div>
        </div>
    </div>
    <div class="col-xl-2 col-lg-3 col-md-4 col-sm-6">
        <div class="card shadow-sm border-0 h-100">
            <div class="card-body">
                <div class="d-flex justify-content-between align-items-start">
                    <div>
                        <p class="text-muted text-uppercase small mb-1">Retiros mes</p>
                        <h4 class="fw-semibold mb-0">{{ $stats['retiros_mes'] }}</h4>
                    </div>
                    <span class="icon-circle bg-info-subtle text-info"><i class="fas fa-arrow-trend-down"></i></span>
                </div>
            </div>
        </div>
    </div>
</div>

<div class="card shadow-sm border-0 mb-3">
    <div class="card-body">
        <form method="get" class="row g-3 align-items-end">
            <div class="col-xl-3 col-md-4">
                <label for="deposito_estado" class="form-label small text-uppercase text-muted mb-1">Depósitos</label>
                <select id="deposito_estado" name="deposito_estado" class="form-select form-select-sm">
                    @foreach($estadoDepositos as $value => $label)
                        <option value="{{ $value }}" @selected($filtros['deposito_estado'] === $value)>{{ $label }}</option>
                    @endforeach
                </select>
            </div>
            <div class="col-xl-3 col-md-4">
                <label for="retiro_estado" class="form-label small text-uppercase text-muted mb-1">Retiros</label>
                <select id="retiro_estado" name="retiro_estado" class="form-select form-select-sm">
                    @foreach($estadoRetiros as $value => $label)
                        <option value="{{ $value }}" @selected($filtros['retiro_estado'] === $value)>{{ $label }}</option>
                    @endforeach
                </select>
            </div>
            <div class="col-xl-2 col-md-4">
                <label for="metodo" class="form-label small text-uppercase text-muted mb-1">Método de pago</label>
                <select id="metodo" name="metodo" class="form-select form-select-sm">
                    <option value="todos" @selected($filtros['metodo'] === 'todos')>Todos</option>
                    @foreach($metodos as $metodo)
                        <option value="{{ $metodo }}" @selected($filtros['metodo'] === $metodo)>{{ ucfirst(str_replace('_', ' ', $metodo)) }}</option>
                    @endforeach
                </select>
            </div>
            <div class="col-xl-2 col-md-4">
                <label for="orden" class="form-label small text-uppercase text-muted mb-1">Ordenar por</label>
                <select id="orden" name="orden" class="form-select form-select-sm">
                    @foreach($ordenOptions as $value => $label)
                        <option value="{{ $value }}" @selected($filtros['orden'] === $value)>{{ $label }}</option>
                    @endforeach
                </select>
            </div>
            <div class="col-xl-2 col-md-4">
                <label for="q" class="form-label small text-uppercase text-muted mb-1">Buscar</label>
                <input type="search" id="q" name="q" value="{{ $filtros['q'] }}" class="form-control form-control-sm" placeholder="Referencia, usuario, correo...">
            </div>
            <div class="col-xl-2 col-md-4">
                <label for="desde" class="form-label small text-uppercase text-muted mb-1">Desde</label>
                <input type="date" id="desde" name="desde" value="{{ $filtros['desde'] }}" class="form-control form-control-sm">
            </div>
            <div class="col-xl-2 col-md-4">
                <label for="hasta" class="form-label small text-uppercase text-muted mb-1">Hasta</label>
                <input type="date" id="hasta" name="hasta" value="{{ $filtros['hasta'] }}" class="form-control form-control-sm">
            </div>
            <div class="col-xl-2 col-md-4 d-flex gap-2">
                <button type="submit" class="btn btn-sm btn-primary flex-fill">
                    <i class="fas fa-filter me-1"></i> Aplicar
                </button>
                <a href="{{ route('admin.transacciones') }}" class="btn btn-sm btn-outline-secondary" title="Limpiar filtros">
                    <i class="fas fa-rotate-right"></i>
                </a>
            </div>
        </form>
    </div>
</div>

<div class="row g-3">
    <div class="col-xxl-8">
        <div class="card shadow-sm border-0 mb-3">
            <div class="card-header bg-white border-0 d-flex justify-content-between align-items-center">
                <div>
                    <h6 class="mb-0">Depósitos</h6>
                    <small class="text-muted">{{ $depositos->total() }} registros · Mostrando {{ $depositos->firstItem() ?? 0 }}-{{ $depositos->lastItem() ?? 0 }}</small>
                </div>
                <span class="badge bg-primary-subtle text-primary">{{ strtoupper($filtros['deposito_estado']) }}</span>
            </div>
            <div class="card-body p-0">
                <div class="table-responsive">
                    <table class="table table-hover align-middle mb-0">
                        <thead class="table-light">
                            <tr>
                                <th>Usuario</th>
                                <th>Detalles del pago</th>
                                <th>Monto</th>
                                <th>Estado</th>
                                <th>Fechas</th>
                                <th class="text-end">Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                        @forelse($depositos as $deposito)
                            @php
                                $usuario = $deposito->usuario;
                                $aprobador = $deposito->aprobador;
                            @endphp
                            <tr>
                                <td>
                                    <div class="fw-semibold">{{ $usuario->nombre ?? 'Usuario N/D' }}</div>
                                    <small class="text-muted">{{ $usuario->email ?? 'Sin correo' }}</small>
                                </td>
                                <td>
                                    <div class="fw-semibold">{{ ucfirst($deposito->metodo_pago ?? 'N/D') }}</div>
                                    <small class="text-muted">Ref: {{ $deposito->referencia_pago ?? 'N/A' }}</small>
                                </td>
                                <td>
                                    <div class="fw-semibold">${{ number_format((float) $deposito->monto, 0, ',', '.') }}</div>
                                    <small class="text-muted">Última revisión: {{ $deposito->updated_at?->diffForHumans() ?? 'N/D' }}</small>
                                </td>
                                <td>
                                    <span class="badge {{ $estadoBadges[$deposito->estado] ?? 'bg-secondary' }}">{{ ucfirst(str_replace('_', ' ', $deposito->estado ?? 'N/D')) }}</span>
                                    @if($aprobador)
                                        <div class="small text-muted mt-1">Por: {{ $aprobador->nombre }}</div>
                                    @endif
                                </td>
                                <td>
                                    <div class="fw-semibold">Creado: {{ $deposito->created_at?->format('d M Y H:i') ?? 'N/D' }}</div>
                                    <small class="text-muted">Aprobado: {{ $deposito->fecha_aprobacion?->format('d M Y H:i') ?? 'Pendiente' }}</small>
                                </td>
                                <td class="text-end">
                                    @php
                                        $puedeAprobarDeposito = in_array($deposito->estado, ['pendiente', 'rechazado'], true);
                                        $puedeRechazarDeposito = $deposito->estado === 'pendiente';
                                    @endphp
                                    <div class="btn-toolbar justify-content-end gap-1">
                                        @if($deposito->comprobante_ruta)
                                            <div class="btn-group btn-group-sm">
                                                <a href="{{ asset('storage/' . ltrim($deposito->comprobante_ruta, '/')) }}" target="_blank" class="btn btn-sm btn-outline-secondary" title="Ver comprobante">
                                                    <i class="fas fa-file-invoice"></i>
                                                </a>
                                            </div>
                                        @endif
                                        <div class="btn-group btn-group-sm">
                                            <button type="button" class="btn btn-sm btn-outline-secondary" data-bs-toggle="popover"
                                                data-bs-title="Notas administrativas" data-bs-content="{{ $deposito->notas_admin ?? 'Sin notas' }}">
                                                <i class="fas fa-note-sticky"></i>
                                            </button>
                                        </div>
                                        <div class="btn-group btn-group-sm">
                                            <button type="button" class="btn btn-sm btn-outline-secondary copy-ref" data-referencia="{{ $deposito->referencia_pago ?? '' }}" title="Copiar referencia">
                                                <i class="fas fa-copy"></i>
                                            </button>
                                        </div>
                                        <div class="btn-group btn-group-sm">
                                            <button type="button" class="btn btn-sm btn-outline-primary dropdown-toggle" data-bs-toggle="dropdown" aria-expanded="false">
                                                <i class="fas fa-ellipsis-vertical"></i>
                                            </button>
                                            <ul class="dropdown-menu dropdown-menu-end shadow-sm">
                                                <li class="dropdown-header text-uppercase small text-muted">Acciones</li>
                                                @if($puedeAprobarDeposito)
                                                    <li>
                                                        <button type="button" class="dropdown-item accion-aprobar-deposito"
                                                            data-bs-toggle="modal"
                                                            data-bs-target="#modalAprobarDeposito"
                                                            data-action="{{ route('admin.transacciones.depositos.aprobar', $deposito) }}"
                                                            data-usuario="{{ $usuario->nombre ?? 'Usuario N/D' }}"
                                                            data-monto="{{ number_format((float) $deposito->monto, 0, ',', '.') }}"
                                                            data-metodo="{{ ucfirst($deposito->metodo_pago ?? 'N/D') }}"
                                                            data-referencia="{{ $deposito->referencia_pago ?? 'N/A' }}">
                                                            <i class="fas fa-check text-success me-2"></i>Aprobar depósito
                                                        </button>
                                                    </li>
                                                @endif
                                                @if($puedeRechazarDeposito)
                                                    <li>
                                                        <button type="button" class="dropdown-item accion-rechazar-deposito"
                                                            data-bs-toggle="modal"
                                                            data-bs-target="#modalRechazarDeposito"
                                                            data-action="{{ route('admin.transacciones.depositos.rechazar', $deposito) }}"
                                                            data-usuario="{{ $usuario->nombre ?? 'Usuario N/D' }}"
                                                            data-monto="{{ number_format((float) $deposito->monto, 0, ',', '.') }}"
                                                            data-metodo="{{ ucfirst($deposito->metodo_pago ?? 'N/D') }}"
                                                            data-referencia="{{ $deposito->referencia_pago ?? 'N/A' }}">
                                                            <i class="fas fa-ban text-danger me-2"></i>Rechazar depósito
                                                        </button>
                                                    </li>
                                                @endif
                                                @if(!$puedeAprobarDeposito && !$puedeRechazarDeposito)
                                                    <li>
                                                        <span class="dropdown-item text-muted small">No hay acciones disponibles.</span>
                                                    </li>
                                                @endif
                                            </ul>
                                        </div>
                                    </div>
                                </td>
                            </tr>
                        @empty
                            <tr>
                                <td colspan="6" class="text-center text-muted py-5">No se encontraron depósitos con los filtros aplicados.</td>
                            </tr>
                        @endforelse
                        </tbody>
                    </table>
                </div>
            </div>
            <div class="card-footer bg-white border-0">
                <div class="d-flex flex-column flex-lg-row justify-content-between align-items-lg-center gap-2">
                    <small class="text-muted">Totales por estado actualizados en tiempo real.</small>
                    {{ $depositos->onEachSide(1)->links('pagination::bootstrap-5') }}
                </div>
            </div>
        </div>

        <div class="card shadow-sm border-0">
            <div class="card-header bg-white border-0 d-flex justify-content-between align-items-center">
                <div>
                    <h6 class="mb-0">Retiros</h6>
                    <small class="text-muted">{{ $retiros->total() }} registros · Mostrando {{ $retiros->firstItem() ?? 0 }}-{{ $retiros->lastItem() ?? 0 }}</small>
                </div>
                <span class="badge bg-danger-subtle text-danger">{{ strtoupper($filtros['retiro_estado']) }}</span>
            </div>
            <div class="card-body p-0">
                <div class="table-responsive">
                    <table class="table table-hover align-middle mb-0">
                        <thead class="table-light">
                            <tr>
                                <th>Usuario</th>
                                <th>Destino</th>
                                <th>Monto</th>
                                <th>Estado</th>
                                <th>Fechas</th>
                                <th class="text-end">Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                        @forelse($retiros as $retiro)
                            @php
                                $usuario = $retiro->usuario;
                                $procesador = $retiro->procesador;
                            @endphp
                            <tr>
                                <td>
                                    <div class="fw-semibold">{{ $usuario->nombre ?? 'Usuario N/D' }}</div>
                                    <small class="text-muted">{{ $usuario->email ?? 'Sin correo' }}</small>
                                </td>
                                <td>
                                    <div class="fw-semibold">{{ strtoupper($usuario->banco ?? 'Banco N/D') }}</div>
                                    <small class="text-muted">Cuenta {{ strtoupper($usuario->tipo_cuenta ?? 'N/D') }} · {{ $usuario->numero_cuenta ?? 'No registrada' }}</small>
                                </td>
                                <td>
                                    <div class="fw-semibold">${{ number_format((float) $retiro->monto, 0, ',', '.') }}</div>
                                    <small class="text-muted">Solicitado: {{ $retiro->fecha_solicitud?->format('d M Y H:i') ?? 'N/D' }}</small>
                                </td>
                                <td>
                                    <span class="badge {{ $estadoBadges[$retiro->estado] ?? 'bg-secondary' }}">{{ ucfirst(str_replace('_', ' ', $retiro->estado ?? 'N/D')) }}</span>
                                    @if($procesador)
                                        <div class="small text-muted mt-1">Procesó: {{ $procesador->nombre }}</div>
                                    @endif
                                </td>
                                <td>
                                    <div class="fw-semibold">Procesado: {{ $retiro->fecha_procesado?->format('d M Y H:i') ?? 'Pendiente' }}</div>
                                    <small class="text-muted">Ref: {{ $retiro->referencia_transaccion ?? 'Sin asignar' }}</small>
                                </td>
                                <td class="text-end">
                                    @php
                                        $puedeMarcarProceso = $retiro->estado === 'pendiente';
                                        $puedeAprobarRetiro = in_array($retiro->estado, ['pendiente', 'en_proceso'], true);
                                        $puedeRechazarRetiro = in_array($retiro->estado, ['pendiente', 'en_proceso'], true);
                                        $destinoRetiro = trim(implode(' · ', array_filter([
                                            strtoupper($usuario->banco ?? 'Banco N/D'),
                                            strtoupper($usuario->tipo_cuenta ?? 'Tipo N/D'),
                                            $usuario->numero_cuenta ?? 'Sin cuenta',
                                        ])));
                                    @endphp
                                    <div class="btn-toolbar justify-content-end gap-1">
                                        @if($retiro->comprobante_ruta)
                                            <div class="btn-group btn-group-sm">
                                                <a href="{{ asset('storage/' . ltrim($retiro->comprobante_ruta, '/')) }}" target="_blank" class="btn btn-sm btn-outline-secondary" title="Ver comprobante">
                                                    <i class="fas fa-file-invoice-dollar"></i>
                                                </a>
                                            </div>
                                        @endif
                                        <div class="btn-group btn-group-sm">
                                            <button type="button" class="btn btn-sm btn-outline-secondary" data-bs-toggle="popover"
                                                data-bs-title="Notas administrativas" data-bs-content="{{ $retiro->notas_admin ?? 'Sin notas' }}">
                                                <i class="fas fa-note-sticky"></i>
                                            </button>
                                        </div>
                                        <div class="btn-group btn-group-sm">
                                            <button type="button" class="btn btn-sm btn-outline-secondary" data-bs-toggle="popover"
                                                data-bs-title="Mensaje del usuario" data-bs-content="{{ $retiro->notas_usuario ?? 'Sin comentarios' }}">
                                                <i class="fas fa-comment-dots"></i>
                                            </button>
                                        </div>
                                        @if($retiro->referencia_transaccion)
                                            <div class="btn-group btn-group-sm">
                                                <button type="button" class="btn btn-sm btn-outline-secondary copy-ref" data-referencia="{{ $retiro->referencia_transaccion }}" title="Copiar referencia">
                                                    <i class="fas fa-copy"></i>
                                                </button>
                                            </div>
                                        @endif
                                        <div class="btn-group btn-group-sm">
                                            <button type="button" class="btn btn-sm btn-outline-primary dropdown-toggle" data-bs-toggle="dropdown" aria-expanded="false">
                                                <i class="fas fa-ellipsis-vertical"></i>
                                            </button>
                                            <ul class="dropdown-menu dropdown-menu-end shadow-sm">
                                                <li class="dropdown-header text-uppercase small text-muted">Acciones</li>
                                                @if($puedeMarcarProceso)
                                                    <li>
                                                        <button type="button" class="dropdown-item accion-retiro-proceso"
                                                            data-bs-toggle="modal"
                                                            data-bs-target="#modalRetiroProceso"
                                                            data-action="{{ route('admin.transacciones.retiros.proceso', $retiro) }}"
                                                            data-usuario="{{ $usuario->nombre ?? 'Usuario N/D' }}"
                                                            data-monto="{{ number_format((float) $retiro->monto, 0, ',', '.') }}">
                                                            <i class="fas fa-clock text-info me-2"></i>Marcar en proceso
                                                        </button>
                                                    </li>
                                                @endif
                                                @if($puedeAprobarRetiro)
                                                    <li>
                                                        <button type="button" class="dropdown-item accion-aprobar-retiro"
                                                            data-bs-toggle="modal"
                                                            data-bs-target="#modalAprobarRetiro"
                                                            data-action="{{ route('admin.transacciones.retiros.aprobar', $retiro) }}"
                                                            data-usuario="{{ $usuario->nombre ?? 'Usuario N/D' }}"
                                                            data-monto="{{ number_format((float) $retiro->monto, 0, ',', '.') }}"
                                                            data-destino="{{ $destinoRetiro }}"
                                                            data-referencia="{{ $retiro->referencia_transaccion ?? '' }}">
                                                            <i class="fas fa-circle-check text-success me-2"></i>Confirmar pago
                                                        </button>
                                                    </li>
                                                @endif
                                                @if($puedeRechazarRetiro)
                                                    <li>
                                                        <button type="button" class="dropdown-item accion-rechazar-retiro"
                                                            data-bs-toggle="modal"
                                                            data-bs-target="#modalRechazarRetiro"
                                                            data-action="{{ route('admin.transacciones.retiros.rechazar', $retiro) }}"
                                                            data-usuario="{{ $usuario->nombre ?? 'Usuario N/D' }}"
                                                            data-monto="{{ number_format((float) $retiro->monto, 0, ',', '.') }}">
                                                            <i class="fas fa-circle-xmark text-danger me-2"></i>Rechazar retiro
                                                        </button>
                                                    </li>
                                                @endif
                                                @if(!$puedeMarcarProceso && !$puedeAprobarRetiro && !$puedeRechazarRetiro)
                                                    <li>
                                                        <span class="dropdown-item text-muted small">No hay acciones disponibles.</span>
                                                    </li>
                                                @endif
                                            </ul>
                                        </div>
                                    </div>
                                </td>
                            </tr>
                        @empty
                            <tr>
                                <td colspan="6" class="text-center text-muted py-5">No se encontraron retiros con los filtros aplicados.</td>
                            </tr>
                        @endforelse
                        </tbody>
                    </table>
                </div>
            </div>
            <div class="card-footer bg-white border-0">
                <div class="d-flex flex-column flex-lg-row justify-content-between align-items-lg-center gap-2">
                    <small class="text-muted">Recuerda liberar el saldo pendiente luego de cada retiro completado.</small>
                    {{ $retiros->onEachSide(1)->links('pagination::bootstrap-5') }}
                </div>
            </div>
        </div>
    </div>
    <div class="col-xxl-4">
        <div class="card shadow-sm border-0 mb-3">
            <div class="card-header bg-white border-0 d-flex justify-content-between align-items-center">
                <h6 class="mb-0">Resumen depósitos</h6>
                <span class="badge bg-primary-subtle text-primary">{{ $resumenEstadosDepositos->count() }}</span>
            </div>
            <div class="card-body">
                @forelse($resumenEstadosDepositos as $resumen)
                    <div class="d-flex justify-content-between align-items-center py-2 border-bottom">
                        <div>
                            <div class="fw-semibold">{{ ucfirst(str_replace('_', ' ', $resumen['estado'])) }}</div>
                            <small class="text-muted">{{ $resumen['monto'] }}</small>
                        </div>
                        <span class="badge bg-light text-dark">{{ number_format($resumen['total']) }}</span>
                    </div>
                @empty
                    <p class="text-muted mb-0">Sin información para mostrar.</p>
                @endforelse
            </div>
        </div>

        <div class="card shadow-sm border-0 mb-3">
            <div class="card-header bg-white border-0 d-flex justify-content-between align-items-center">
                <h6 class="mb-0">Resumen retiros</h6>
                <span class="badge bg-danger-subtle text-danger">{{ $resumenEstadosRetiros->count() }}</span>
            </div>
            <div class="card-body">
                @forelse($resumenEstadosRetiros as $resumen)
                    <div class="d-flex justify-content-between align-items-center py-2 border-bottom">
                        <div>
                            <div class="fw-semibold">{{ ucfirst(str_replace('_', ' ', $resumen['estado'])) }}</div>
                            <small class="text-muted">{{ $resumen['monto'] }}</small>
                        </div>
                        <span class="badge bg-light text-dark">{{ number_format($resumen['total']) }}</span>
                    </div>
                @empty
                    <p class="text-muted mb-0">Sin información para mostrar.</p>
                @endforelse
            </div>
        </div>

        <div class="card shadow-sm border-0">
            <div class="card-header bg-white border-0">
                <h6 class="mb-0">Buenas prácticas</h6>
            </div>
            <div class="card-body">
                <p class="text-muted small mb-2"><i class="fas fa-circle-info me-1"></i>Antes de aprobar un depósito valida el comprobante y la conciliación bancaria.</p>
                <p class="text-muted small mb-2"><i class="fas fa-circle-info me-1"></i>Los retiros pendientes afectan el saldo pendiente de cada usuario; ciérralos para liberar saldo.</p>
                <p class="text-muted small mb-0"><i class="fas fa-circle-info me-1"></i>Puedes copiar referencias con el botón <i class="fas fa-copy"></i> para trazabilidad en tu ERP.</p>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', () => {
    const popoverTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="popover"]'));
    popoverTriggerList.forEach(popoverTriggerEl => {
        new bootstrap.Popover(popoverTriggerEl, { trigger: 'focus' });
    });

    document.querySelectorAll('.copy-ref').forEach(button => {
        button.addEventListener('click', () => {
            const referencia = button.getAttribute('data-referencia') || '';
            if (!referencia) {
                return;
            }

            navigator.clipboard.writeText(referencia).then(() => {
                button.classList.remove('btn-outline-secondary');
                button.classList.add('btn-success');
                button.innerHTML = '<i class="fas fa-check"></i>';
                setTimeout(() => {
                    button.classList.add('btn-outline-secondary');
                    button.classList.remove('btn-success');
                    button.innerHTML = '<i class="fas fa-copy"></i>';
                }, 2000);
            });
        });
    });

    const modalAprobarDeposito = document.getElementById('modalAprobarDeposito');
    if (modalAprobarDeposito) {
        modalAprobarDeposito.addEventListener('show.bs.modal', event => {
            const button = event.relatedTarget;
            if (!button) {
                return;
            }

            const form = modalAprobarDeposito.querySelector('form');
            form.action = button.getAttribute('data-action') || '#';
            const select = modalAprobarDeposito.querySelector('#depositoAprobarAdmin');
            const defaultAdminId = modalAprobarDeposito.dataset.defaultAdminId || '';
            if (select) {
                select.value = defaultAdminId;
            }
            modalAprobarDeposito.querySelector('#depositoAprobarUsuario').textContent = button.getAttribute('data-usuario') || 'Usuario seleccionado';
            modalAprobarDeposito.querySelector('#depositoAprobarMonto').textContent = `$${button.getAttribute('data-monto') || '0'}`;
            modalAprobarDeposito.querySelector('#depositoAprobarMetodo').textContent = button.getAttribute('data-metodo') || 'Método';
            modalAprobarDeposito.querySelector('#depositoAprobarReferencia').textContent = button.getAttribute('data-referencia') || 'N/A';
        });

        modalAprobarDeposito.addEventListener('hidden.bs.modal', () => {
            const form = modalAprobarDeposito.querySelector('form');
            form.action = '#';
            form.reset();
            const select = modalAprobarDeposito.querySelector('#depositoAprobarAdmin');
            const defaultAdminId = modalAprobarDeposito.dataset.defaultAdminId || '';
            if (select) {
                select.value = defaultAdminId;
            }
            modalAprobarDeposito.querySelector('#depositoAprobarUsuario').textContent = 'Usuario seleccionado';
            modalAprobarDeposito.querySelector('#depositoAprobarMonto').textContent = '$0';
            modalAprobarDeposito.querySelector('#depositoAprobarMetodo').textContent = 'Método';
            modalAprobarDeposito.querySelector('#depositoAprobarReferencia').textContent = 'N/A';
        });
    }

    const modalRechazarDeposito = document.getElementById('modalRechazarDeposito');
    if (modalRechazarDeposito) {
        modalRechazarDeposito.addEventListener('show.bs.modal', event => {
            const button = event.relatedTarget;
            if (!button) {
                return;
            }

            const form = modalRechazarDeposito.querySelector('form');
            form.action = button.getAttribute('data-action') || '#';
            const select = modalRechazarDeposito.querySelector('#depositoRechazarAdmin');
            const defaultAdminId = modalRechazarDeposito.dataset.defaultAdminId || '';
            if (select) {
                select.value = defaultAdminId;
            }
            modalRechazarDeposito.querySelector('#depositoRechazarUsuario').textContent = button.getAttribute('data-usuario') || 'Usuario seleccionado';
            modalRechazarDeposito.querySelector('#depositoRechazarMonto').textContent = `$${button.getAttribute('data-monto') || '0'}`;
            modalRechazarDeposito.querySelector('#depositoRechazarMetodo').textContent = button.getAttribute('data-metodo') || 'Método';
            modalRechazarDeposito.querySelector('#depositoRechazarReferencia').textContent = button.getAttribute('data-referencia') || 'N/A';
        });

        modalRechazarDeposito.addEventListener('hidden.bs.modal', () => {
            const form = modalRechazarDeposito.querySelector('form');
            form.action = '#';
            form.reset();
            const select = modalRechazarDeposito.querySelector('#depositoRechazarAdmin');
            const defaultAdminId = modalRechazarDeposito.dataset.defaultAdminId || '';
            if (select) {
                select.value = defaultAdminId;
            }
            modalRechazarDeposito.querySelector('#depositoRechazarUsuario').textContent = 'Usuario seleccionado';
            modalRechazarDeposito.querySelector('#depositoRechazarMonto').textContent = '$0';
            modalRechazarDeposito.querySelector('#depositoRechazarMetodo').textContent = 'Método';
            modalRechazarDeposito.querySelector('#depositoRechazarReferencia').textContent = 'N/A';
        });
    }

    const modalAprobarRetiro = document.getElementById('modalAprobarRetiro');
    if (modalAprobarRetiro) {
        modalAprobarRetiro.addEventListener('show.bs.modal', event => {
            const button = event.relatedTarget;
            if (!button) {
                return;
            }

            const form = modalAprobarRetiro.querySelector('form');
            form.action = button.getAttribute('data-action') || '#';
            const select = modalAprobarRetiro.querySelector('#retiroAprobarAdmin');
            const defaultAdminId = modalAprobarRetiro.dataset.defaultAdminId || '';
            if (select) {
                select.value = defaultAdminId;
            }
            modalAprobarRetiro.querySelector('#retiroAprobarUsuario').textContent = button.getAttribute('data-usuario') || 'Usuario seleccionado';
            modalAprobarRetiro.querySelector('#retiroAprobarMonto').textContent = `$${button.getAttribute('data-monto') || '0'}`;
            modalAprobarRetiro.querySelector('#retiroAprobarDestino').textContent = button.getAttribute('data-destino') || 'N/A';
            modalAprobarRetiro.querySelector('#retiroAprobarReferencia').value = button.getAttribute('data-referencia') || '';
        });

        modalAprobarRetiro.addEventListener('hidden.bs.modal', () => {
            const form = modalAprobarRetiro.querySelector('form');
            form.action = '#';
            form.reset();
            const select = modalAprobarRetiro.querySelector('#retiroAprobarAdmin');
            const defaultAdminId = modalAprobarRetiro.dataset.defaultAdminId || '';
            if (select) {
                select.value = defaultAdminId;
            }
            modalAprobarRetiro.querySelector('#retiroAprobarUsuario').textContent = 'Usuario seleccionado';
            modalAprobarRetiro.querySelector('#retiroAprobarMonto').textContent = '$0';
            modalAprobarRetiro.querySelector('#retiroAprobarDestino').textContent = 'N/A';
        });
    }

    const modalRetiroProceso = document.getElementById('modalRetiroProceso');
    if (modalRetiroProceso) {
        modalRetiroProceso.addEventListener('show.bs.modal', event => {
            const button = event.relatedTarget;
            if (!button) {
                return;
            }

            const form = modalRetiroProceso.querySelector('form');
            form.action = button.getAttribute('data-action') || '#';
            const select = modalRetiroProceso.querySelector('#retiroProcesoAdmin');
            const defaultAdminId = modalRetiroProceso.dataset.defaultAdminId || '';
            if (select) {
                select.value = defaultAdminId;
            }
            modalRetiroProceso.querySelector('#retiroProcesoUsuario').textContent = button.getAttribute('data-usuario') || 'Usuario seleccionado';
            modalRetiroProceso.querySelector('#retiroProcesoMonto').textContent = `$${button.getAttribute('data-monto') || '0'}`;
        });

        modalRetiroProceso.addEventListener('hidden.bs.modal', () => {
            const form = modalRetiroProceso.querySelector('form');
            form.action = '#';
            form.reset();
            const select = modalRetiroProceso.querySelector('#retiroProcesoAdmin');
            const defaultAdminId = modalRetiroProceso.dataset.defaultAdminId || '';
            if (select) {
                select.value = defaultAdminId;
            }
            modalRetiroProceso.querySelector('#retiroProcesoUsuario').textContent = 'Usuario seleccionado';
            modalRetiroProceso.querySelector('#retiroProcesoMonto').textContent = '$0';
        });
    }

    const modalRechazarRetiro = document.getElementById('modalRechazarRetiro');
    if (modalRechazarRetiro) {
        modalRechazarRetiro.addEventListener('show.bs.modal', event => {
            const button = event.relatedTarget;
            if (!button) {
                return;
            }

            const form = modalRechazarRetiro.querySelector('form');
            form.action = button.getAttribute('data-action') || '#';
            const select = modalRechazarRetiro.querySelector('#retiroRechazarAdmin');
            const defaultAdminId = modalRechazarRetiro.dataset.defaultAdminId || '';
            if (select) {
                select.value = defaultAdminId;
            }
            modalRechazarRetiro.querySelector('#retiroRechazarUsuario').textContent = button.getAttribute('data-usuario') || 'Usuario seleccionado';
            modalRechazarRetiro.querySelector('#retiroRechazarMonto').textContent = `$${button.getAttribute('data-monto') || '0'}`;
        });

        modalRechazarRetiro.addEventListener('hidden.bs.modal', () => {
            const form = modalRechazarRetiro.querySelector('form');
            form.action = '#';
            form.reset();
            const select = modalRechazarRetiro.querySelector('#retiroRechazarAdmin');
            const defaultAdminId = modalRechazarRetiro.dataset.defaultAdminId || '';
            if (select) {
                select.value = defaultAdminId;
            }
            modalRechazarRetiro.querySelector('#retiroRechazarUsuario').textContent = 'Usuario seleccionado';
            modalRechazarRetiro.querySelector('#retiroRechazarMonto').textContent = '$0';
        });
    }
});
</script>
@endpush
@endsection
