# Sincronización de `scawatts_tokens.json` con el backend

Este documento describe cómo se mantiene sincronizado el archivo maestro `scawatts_tokens.json` con el resto de componentes del servidor (API Python, Laravel, WordPress y scripts auxiliares). También detalla la frecuencia de actualización de la valorización y el mecanismo de reconciliación con la base de datos MySQL.

## 1. Fuente de verdad y proceso principal

| Componente | Rol | Archivo/endpoint | Notas |
|------------|-----|------------------|-------|
| API Python (`API_produccion/api_scawatts.py`) | **Fuente de verdad** para todos los ScaWatts. | `scawatts_tokens.json` | Todas las operaciones (crear, transferir, marketplace) pasan por `update_scawatts_atomic`, que mantiene un `LOCK_EX` durante el ciclo read → modify → write y fuerza `flush()` + `os.fsync()` antes de liberar el candado. |
| Hilo `actualizar_valorizacion()` | Recalcula la valorización de todos los tokens cada **5 segundos**. | — | Usa la caché en memoria de `base_ScaWatt_2025.json`; escribe los nuevos valores en `scawatts_tokens.json` con el mismo mecanismo atómico. |
| Script `actualizar_usuario_scawatt.py` | Refleja cambios inmediatos por usuario. | `usuarios-app/<id>/mi-scawatt.json` (producción y WordPress) | Se ejecuta justo después de crear/transferir un token; garantiza que el portafolio del usuario tenga los datos más recientes del JSON maestro. |
| Script `sincronizar_valorizaciones.py` | Replica las valorizaciones recién calculadas a los portafolios. | `usuarios-app/<id>/mi-scawatt.json` + WordPress | Se dispara al final de cada ciclo de valorización (cada 5 s). Calcula los totales y copia los archivos a `public_html/scalar-api/...`. |
| Script `monitor_scawatt_v2.py` | Publica los dashboards/JSON de frontend. | `scawatts_disponibles.json`, etc. | Vuelve a leer `scawatts_tokens.json` cada ~9 s para mantener calientes los datos expuestos al frontend. |

## 2. Flujo detallado

1. **Creación o cambio de propietario**
   - El endpoint `/api/scawatts/crear` (u otro flujo) modifica `scawatts_tokens.json` mediante `update_scawatts_atomic`.
   - En el mismo request se actualizan `trazabilidad_scawatts.json`, `scawatts_active_user.json`, `usuarios-aliados.json` y se lanza `actualizar_usuario_scawatt.py` para refrescar los archivos del usuario (producción + WordPress).

2. **Valorización continua (cada 5 s)**
   - El hilo `actualizar_valorizacion()` despierta cada 5 segundos.
   - Recorre los 32 tokens actuales (o los que existan), calcula `calcular_valorizacion(fecha_inicio)` y persiste el nuevo valor en el JSON maestro.
   - Al concluir, ejecuta `sincronizar_valorizaciones.py`, que:
     - Lee `scawatts_tokens.json`.
     - Actualiza los campos `valorizacion_actual` de cada `mi-scawatt.json` y recalcula los totales.
     - Copia esos archivos a la ruta pública de WordPress (`public_html/scalar-api/api-scawatts/produccion/scausers/usuarios-app/<id>/`).
   - Resultado: tanto el backend Laravel como el frontend tienen valorizaciones actualizadas con un retraso máximo de **5–10 s**.

3. **Monitor y dashboards**
   - `monitor_scawatt_v2.py` (cron en background) lee el JSON maestro, genera archivos agregados (`scawatts_disponibles.json`, indicadores para dashboards) y los deja listos para consumo del frontend/monitores cada ~9 s.

## 3. Sincronización con la base de datos MySQL

La migración dejó la tabla `scawatts` en MySQL como referencia histórica. En la operación diaria **no** se escribe ahí, pero existe el script `sincronizar_bd_json.py` para reconciliar diferencias cuando sea necesario.

- **Dirección del flujo:** Base de datos → JSON.
- **Uso típico:** auditorías o migraciones cuando se detecta que el JSON perdió registros (por ejemplo, antes de implementar el file locking).
- **Qué hace:**
  1. Lee todos los registros `estado = 'activo'` desde `scawatts`.
  2. Compara con las llaves del JSON maestro.
  3. Elimina del JSON los seriales que ya no existen en BD.
  4. Inserta en el JSON los que faltan, estimando `valorizacion = valor_actual – valor_inicial` (el hilo de valorización los actualizará bien en el siguiente ciclo).
  5. Persiste los cambios utilizando también `LOCK_EX` + `fsync`.

> **Periodicidad:** No tiene cron programado; se ejecuta manualmente cuando el equipo de backend lo considera necesario (`python3 sincronizar_bd_json.py`).

## 4. Frecuencias y SLAs

| Proceso | Frecuencia | Responsable | Resultado |
|---------|------------|-------------|-----------|
| Hilo de valorización (`actualizar_valorizacion`) | Cada 5 segundos | API Python (daemon) | Campo `valorizacion` actualizado en `scawatts_tokens.json`. |
| `sincronizar_valorizaciones.py` | Cada vez que corre el hilo (≈5 s) | API Python (post-job) | `mi-scawatt.json` y totales de usuarios sincronizados en producción + WordPress. |
| `monitor_scawatt_v2.py` | Cada 9 segundos | Servicio monitor | Dashboards/JSON de frontend actualizados. |
| `sincronizar_bd_json.py` | Bajo demanda (manual) | Equipo backend | Reconcilia la tabla `scawatts` de MySQL con el JSON maestro. |

## 5. Comandos útiles

```bash
# Verificar el estado actual del hilo (desde produccion-scawatts)
tail -f API_produccion/api_scawatts.log

# Ejecutar la validación de desempeño y muestras de valorización
python3 validar_valorizacion_cache.py

# Reconciliar BD → JSON si se detectan diferencias
python3 sincronizar_bd_json.py
```

## 6. Consideraciones

- Nunca modifiques `scawatts_tokens.json` manualmente; usa los endpoints de la API o los scripts provistos.
- Si detienes el servicio Python, el hilo de valorización y la sincronización de usuarios dejan de correr; verifica siempre que el proceso en el puerto 5004 esté arriba antes de generar tokens.
- Para auditorías, conserva los logs:
  - `API_produccion/api_scawatts.log` (operación de API e hilo).
  - `sincronizacion_bd_json.log` (cada ejecución manual de reconciliación).
- Eliminar scripts antiguos (como `asignar_scawatts_jose.php`) ayuda a prevenir ejecuciones sobre endpoints viejos (5001) que ya no tienen locking ni cache.

Con este flujo, la valorización se mantiene actualizada casi en tiempo real y los distintos componentes del backend consumen datos consistentes en cuestión de segundos.
