# 🚨 SOLUCIÓN: Error "Error al obtener datos solares"

**Fecha:** 29 de octubre de 2025  
**Error reportado:** `Error fetching solar data: Error: Error al obtener datos solares at solar.service.ts:58:17`

---

## ✅ DIAGNÓSTICO

El **backend funciona perfectamente**. El problema está en el **frontend** que probablemente está llamando a una URL incorrecta o inexistente.

### Prueba realizada (backend OK ✅):
```bash
curl "https://escalar.app/scadmin/backend/public/api/solar/current"
```
**Resultado:** Devuelve datos correctamente con código 200.

---

## 🔧 SOLUCIÓN RÁPIDA

### En `solar.service.ts` (línea ~58):

#### ❌ INCORRECTO (puede estar así):
```typescript
// Cualquiera de estas URLs está MAL:
const url = 'https://escalar.app/api/solar/current';  // ❌ Falta la ruta completa
const url = 'https://escalar.app/develop/solar/...';  // ❌ No existe
const url = 'https://escalar.app/api-stromglass/...'; // ❌ Eliminado
```

#### ✅ CORRECTO (debe ser):
```typescript
const url = 'https://escalar.app/scadmin/backend/public/api/solar/current';
```

---

## 📋 CHECKLIST DE CORRECCIÓN

1. **Abrir archivo:** `solar.service.ts`
2. **Buscar línea ~58** donde se define la URL del endpoint
3. **Reemplazar URL** con: `https://escalar.app/scadmin/backend/public/api/solar/current`
4. **Verificar headers:**
   ```typescript
   headers: {
     'Accept': 'application/json'
   }
   ```
5. **Guardar y recompilar**
6. **Probar en navegador**

---

## 💻 CÓDIGO COMPLETO DE REFERENCIA

### Servicio correcto (solar.service.ts):
```typescript
import { Injectable } from '@angular/core';
import { HttpClient, HttpHeaders } from '@angular/common/http';
import { Observable } from 'rxjs';
import { catchError } from 'rxjs/operators';

@Injectable({
  providedIn: 'root'
})
export class SolarService {
  // ✅ URL CORRECTA
  private apiUrl = 'https://escalar.app/scadmin/backend/public/api/solar/current';

  constructor(private http: HttpClient) {}

  getCurrentSolarData(): Observable<any> {
    const headers = new HttpHeaders({
      'Accept': 'application/json'
    });

    return this.http.get(this.apiUrl, { headers })
      .pipe(
        catchError(error => {
          console.error('❌ Error al obtener datos solares:', error);
          console.error('URL llamada:', this.apiUrl);
          console.error('Status:', error.status);
          console.error('Response:', error.error);
          throw new Error('Error al obtener datos solares');
        })
      );
  }
}
```

---

## 🧪 VERIFICACIÓN

### Desde el navegador (consola):
```javascript
fetch('https://escalar.app/scadmin/backend/public/api/solar/current', {
  headers: { 'Accept': 'application/json' }
})
.then(r => r.json())
.then(data => console.log('✅ Datos solares:', data))
.catch(err => console.error('❌ Error:', err));
```

**Resultado esperado:** Debe mostrar objeto con datos de granja, solar, generation, etc.

---

## 📄 ESTRUCTURA DE RESPUESTA

El endpoint devuelve:
```json
{
  "granja": {
    "id": 1,
    "nombre": "Granja Solar La Gloria",
    "capacidad_mw": 1,
    "capacidad_kw": 1000,
    "scawatts_activos": 5,
    ...
  },
  "solar": {
    "irradiance": 279.04,
    "uv_index": 0.01,
    "uv_level": { "level": "Muy Bajo", ... },
    "solar_level": { "level": "Muy Baja", ... }
  },
  "generation": {
    "current_power_kw": 237.18,
    "current_power_mw": 0.237,
    "capacity_factor": 27.9,
    ...
  },
  "timestamp": "2025-10-29T17:00:02.289942",
  "location": { ... }
}
```

---

## 🆘 SI EL PROBLEMA PERSISTE

Revisar en `solar.service.ts`:

1. **Variable de entorno incorrecta:**
   ```typescript
   // ❌ NO USAR
   private apiUrl = environment.apiUrl + '/solar/current';
   
   // ✅ USAR (URL completa y hardcoded por ahora)
   private apiUrl = 'https://escalar.app/scadmin/backend/public/api/solar/current';
   ```

2. **Interceptor modificando la URL:**
   - Verificar si hay un HTTP interceptor que esté modificando las URLs
   - Revisar `app.module.ts` o archivos de interceptores

3. **Proxy configuration:**
   - Verificar `proxy.conf.json` o similar
   - Asegurarse de que no está redirigiendo `/api/` a otro lugar

4. **CORS (poco probable):**
   - El backend ya tiene CORS configurado para aceptar todas las peticiones
   - Si aún da error CORS, revisar la consola del navegador

---

## 📚 DOCUMENTACIÓN COMPLETA

Para más detalles, consultar:
- **`ENDPOINT_DATOS_SOLARES.md`** - Guía completa de uso
- **Ejemplos de código** completos en Angular, TypeScript, JavaScript
- **Pruebas con CURL**
- **Manejo de errores**

---

## ⏱️ PRÓXIMOS PASOS

1. ✅ Backend funcionando correctamente
2. ⚠️ **PENDIENTE:** Frontend debe actualizar la URL en `solar.service.ts`
3. ⚠️ **PENDIENTE:** Probar después de la corrección
4. ⚠️ **PENDIENTE:** Verificar que no haya otros servicios con URLs incorrectas

---

**Estimación de tiempo:** 5 minutos (solo cambiar la URL)  
**Prioridad:** Alta (afecta funcionalidad principal)

**Archivo afectado:** `src/app/services/solar.service.ts` (línea ~58)  
**Cambio requerido:** Actualizar URL del endpoint

---

**✅ BACKEND:** Funcionando correctamente  
**⚠️ FRONTEND:** Requiere corrección de URL
