<?php

namespace App\Http\Controllers;

use App\Models\LoginAudit;
use App\Models\Usuario;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\View\View;

class AdminAuthController extends Controller
{
    public function showLoginForm(Request $request): View|RedirectResponse
    {
        $usuario = $request->user();

        if ($usuario instanceof Usuario && $usuario->esMaster()) {
            return redirect()->route('admin.dashboard');
        }

        if ($usuario instanceof Usuario) {
            Auth::logout();
            $request->session()->invalidate();
            $request->session()->regenerateToken();
        }

        return view('auth.login');
    }

    public function login(Request $request): RedirectResponse
    {
        $credentials = $request->validate([
            'email' => ['required', 'string', 'email'],
            'password' => ['required', 'string'],
        ]);

        if (!Auth::attempt($credentials, $request->boolean('remember'))) {
            LoginAudit::create([
                'email' => $credentials['email'],
                'evento' => 'login_failed',
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
            ]);

            return back()->withInput($request->only('email', 'remember'))
                ->with('error', 'Credenciales incorrectas. Verifica tu correo y contraseña.');
        }

        $request->session()->regenerate();

        /** @var Usuario|null $usuario */
        $usuario = Auth::user();

        if (!$usuario instanceof Usuario || !$usuario->esMaster()) {
            LoginAudit::create([
                'usuario_id' => $usuario?->id,
                'email' => $credentials['email'] ?? $usuario?->email,
                'evento' => 'login_denied_no_master',
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
            ]);

            Auth::logout();
            $request->session()->invalidate();
            $request->session()->regenerateToken();

            return back()->withInput($request->only('email'))
                ->with('error', 'Este panel solo está disponible para el usuario Master.');
        }

        LoginAudit::create([
            'usuario_id' => $usuario->id,
            'email' => $usuario->email,
            'evento' => 'login_success',
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
        ]);

        return redirect()->intended(route('admin.dashboard'));
    }

    public function logout(Request $request): RedirectResponse
    {
        /** @var Usuario|null $usuario */
        $usuario = $request->user();

        if ($usuario instanceof Usuario) {
            LoginAudit::create([
                'usuario_id' => $usuario->id,
                'email' => $usuario->email,
                'evento' => 'logout',
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
            ]);
        }

        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('login')->with('status', 'Sesión cerrada correctamente.');
    }

    public function updatePassword(Request $request): RedirectResponse
    {
        $request->validate([
            'password' => ['required', 'string', 'min:8', 'confirmed'],
        ]);

        /** @var Usuario|null $usuario */
        $usuario = $request->user();

        if (!$usuario instanceof Usuario || !$usuario->esMaster()) {
            return redirect()->route('login')->with('error', 'Solo el Master puede actualizar su contraseña.');
        }

        $usuario->password = Hash::make($request->string('password'));
        $usuario->save();

        return back()->with('status', 'Contraseña actualizada correctamente.');
    }
}
