<?php

namespace App\Http\Controllers;

use App\Models\Granja;
use App\Models\Scawatt;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;

class AdminGranjasController extends Controller
{
    public function index(Request $request)
    {
        $estado = $request->query('estado', 'todas');
        $departamento = $request->query('departamento', 'todos');
        $orden = $request->query('orden', 'recientes');
        $busqueda = trim((string) $request->query('q', ''));
        $porPagina = (int) $request->query('por_pagina', 12);
        $porPagina = $porPagina > 0 ? min($porPagina, 60) : 12;

        $granjasQuery = Granja::query()
            ->withCount('scawatts')
            ->withSum('scawatts', 'valor_actual')
            ->withSum(['scawatts as kwh_total' => function ($query) {
                $query->whereNotNull('kwh_asignados');
            }], 'kwh_asignados');

        if ($estado !== 'todas') {
            $granjasQuery->where('estado', $estado);
        }

        if ($departamento !== 'todos' && $departamento !== null) {
            $granjasQuery->where('departamento', $departamento);
        }

        if ($busqueda !== '') {
            $granjasQuery->where(function (Builder $query) use ($busqueda) {
                $query->where('nombre', 'like', "%{$busqueda}%")
                      ->orWhere('municipio', 'like', "%{$busqueda}%")
                      ->orWhere('categoria', 'like', "%{$busqueda}%");
            });
        }

        $granjasQuery->when($orden, function (Builder $query) use ($orden) {
            return match ($orden) {
                'capacidad' => $query->orderByDesc('capacidad_mw'),
                'produccion' => $query->orderByDesc('produccion_estimada_anual'),
                'scawatts' => $query->orderByDesc('scawatts_count'),
                'valor' => $query->orderByDesc('scawatts_sum_valor_actual'),
                default => $query->orderByDesc('created_at'),
            };
        });

        $granjas = $granjasQuery->paginate($porPagina)->withQueryString();

        $stats = [
            'total' => Granja::count(),
            'operativas' => Granja::where('estado', 'operativa')->count(),
            'capacidad_total' => $this->formatNumber(Granja::sum('capacidad_mw')),
            'produccion_total' => $this->formatNumber(Granja::sum('produccion_estimada_anual')),
            'scawatts' => Scawatt::count(),
            'valor_scawatts' => $this->formatCurrency(Scawatt::sum('valor_actual')),
        ];

        $segmentacionEstados = Granja::select('estado', DB::raw('COUNT(*) as total'))
            ->groupBy('estado')
            ->orderByDesc('total')
            ->get();

        $departamentos = Granja::select('departamento')
            ->whereNotNull('departamento')
            ->distinct()
            ->orderBy('departamento')
            ->pluck('departamento');

        $topProduccion = Granja::select('nombre', 'produccion_estimada_anual', 'capacidad_mw')
            ->orderByDesc('produccion_estimada_anual')
            ->limit(5)
            ->get();

        return view('admin.granjas.index', [
            'granjas' => $granjas,
            'stats' => $stats,
            'segmentacionEstados' => $segmentacionEstados,
            'departamentos' => $departamentos,
            'topProduccion' => $topProduccion,
            'filtros' => [
                'estado' => $estado,
                'departamento' => $departamento,
                'orden' => $orden,
                'q' => $busqueda,
                'por_pagina' => $porPagina,
            ],
        ]);
    }

    private function formatCurrency($value): string
    {
        $numeric = $value !== null ? (float) $value : 0.0;
        return '$' . number_format($numeric, 0, ',', '.');
    }

    private function formatNumber($value): string
    {
        $numeric = $value !== null ? (float) $value : 0.0;
        return number_format($numeric, 2, ',', '.');
    }

    public function store(Request $request)
    {
        $estadosPermitidos = ['operativa', 'construccion', 'planificada', 'mantenimiento'];
        $categoriasPermitidas = ['granja_solar', 'parque_solar', 'industrial', 'domestico', 'vehicular'];

        $validated = $request->validate([
            'nombre' => ['required', 'string', 'max:150'],
            'categoria' => ['nullable', 'string', Rule::in($categoriasPermitidas)],
            'departamento' => ['nullable', 'string', 'max:120'],
            'municipio' => ['nullable', 'string', 'max:120'],
            'direccion_completa' => ['nullable', 'string', 'max:255'],
            'latitud' => ['nullable', 'numeric', 'between:-90,90'],
            'longitud' => ['nullable', 'numeric', 'between:-180,180'],
            'capacidad_mw' => ['nullable', 'numeric', 'min:0'],
            'produccion_estimada_anual' => ['nullable', 'numeric', 'min:0'],
            'fecha_inicio_operacion' => ['nullable', 'date'],
            'fecha_final_operacion' => ['nullable', 'date', 'after_or_equal:fecha_inicio_operacion'],
            'vida_util_años' => ['nullable', 'integer', 'min:0', 'max:100'],
            'tasa_valorizacion_anual' => ['nullable', 'numeric', 'between:-100,100'],
            'ganancia_mensual_base' => ['nullable', 'numeric'],
            'estado' => ['required', Rule::in($estadosPermitidos)],
            'descripcion' => ['nullable', 'string'],
            'imagen_url' => ['nullable', 'url', 'max:255'],
        ]);

        $payload = $this->preparePayload($validated);

        DB::transaction(function () use ($payload) {
            Granja::create($payload);
        });

        return back()->with('status', 'Granja registrada correctamente.');
    }

    public function update(Request $request, Granja $granja)
    {
        $estadosPermitidos = ['operativa', 'construccion', 'planificada', 'mantenimiento'];
        $categoriasPermitidas = ['granja_solar', 'parque_solar', 'industrial', 'domestico', 'vehicular'];

        $validated = $request->validate([
            'nombre' => ['required', 'string', 'max:150'],
            'categoria' => ['nullable', 'string', Rule::in($categoriasPermitidas)],
            'departamento' => ['nullable', 'string', 'max:120'],
            'municipio' => ['nullable', 'string', 'max:120'],
            'direccion_completa' => ['nullable', 'string', 'max:255'],
            'latitud' => ['nullable', 'numeric', 'between:-90,90'],
            'longitud' => ['nullable', 'numeric', 'between:-180,180'],
            'capacidad_mw' => ['nullable', 'numeric', 'min:0'],
            'produccion_estimada_anual' => ['nullable', 'numeric', 'min:0'],
            'fecha_inicio_operacion' => ['nullable', 'date'],
            'fecha_final_operacion' => ['nullable', 'date', 'after_or_equal:fecha_inicio_operacion'],
            'vida_util_años' => ['nullable', 'integer', 'min:0', 'max:100'],
            'tasa_valorizacion_anual' => ['nullable', 'numeric', 'between:-100,100'],
            'ganancia_mensual_base' => ['nullable', 'numeric'],
            'estado' => ['required', Rule::in($estadosPermitidos)],
            'descripcion' => ['nullable', 'string'],
            'imagen_url' => ['nullable', 'url', 'max:255'],
        ]);

        $payload = $this->preparePayload($validated);

        DB::transaction(function () use ($granja, $payload) {
            $granja->update($payload);
        });

        return back()->with('status', 'Granja actualizada correctamente.');
    }

    private function preparePayload(array $validated): array
    {
        $numericFields = [
            'latitud',
            'longitud',
            'capacidad_mw',
            'produccion_estimada_anual',
            'tasa_valorizacion_anual',
            'ganancia_mensual_base',
        ];

        $integerFields = ['vida_util_años'];

        $payload = [];
        foreach ($validated as $key => $value) {
            if ($value === '' || $value === null) {
                $payload[$key] = null;
                continue;
            }

            if (in_array($key, $numericFields, true)) {
                $payload[$key] = (float) $value;
                continue;
            }

            if (in_array($key, $integerFields, true)) {
                $payload[$key] = (int) $value;
                continue;
            }

            $payload[$key] = $value;
        }

        return $payload;
    }
}
