<?php

namespace App\Http\Controllers;

use App\Models\Mensaje;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;

class AdminMensajesController extends Controller
{
    public function index(Request $request)
    {
        $estado = $request->query('estado', 'todos');
        $tipo = $request->query('tipo', 'todos');
        $destinatario = $request->query('destinatario', 'todos');
        $busqueda = trim((string) $request->query('q', ''));
        $orden = $request->query('orden', 'recientes');
        $porPagina = (int) $request->query('por_pagina', 20);
        $porPagina = $porPagina > 0 ? min($porPagina, 100) : 20;

        $mensajesQuery = Mensaje::query()
            ->with(['usuario:id,nombre,email', 'creador:id,nombre']);

        if ($estado !== 'todos') {
            $mensajesQuery->where('enviado', $estado === 'enviados');
        }

        if ($tipo !== 'todos') {
            $mensajesQuery->where('tipo', $tipo);
        }

        if ($destinatario !== 'todos') {
            $mensajesQuery->where('destinatario', $destinatario);
        }

        if ($busqueda !== '') {
            $mensajesQuery->where(function (Builder $query) use ($busqueda) {
                $query->where('titulo', 'like', "%{$busqueda}%")
                      ->orWhere('contenido', 'like', "%{$busqueda}%")
                      ->orWhereHas('usuario', function (Builder $subQuery) use ($busqueda) {
                          $subQuery->where('nombre', 'like', "%{$busqueda}%")
                                   ->orWhere('email', 'like', "%{$busqueda}%");
                      });
            });
        }

        $mensajesQuery->when($orden, function (Builder $query) use ($orden) {
            return match ($orden) {
                'programados' => $query->orderByDesc('fecha_programada'),
                'enviados' => $query->orderByDesc('fecha_enviado'),
                default => $query->orderByDesc('created_at'),
            };
        });

        $mensajes = $mensajesQuery->paginate($porPagina)->withQueryString();

        $stats = [
            'total' => Mensaje::count(),
            'enviados' => Mensaje::where('enviado', true)->count(),
            'pendientes' => Mensaje::where('enviado', false)->count(),
            'programados_hoy' => Mensaje::whereDate('fecha_programada', Carbon::today())->count(),
        ];

        $resumenTipos = Mensaje::select('tipo', DB::raw('COUNT(*) as total'))
            ->groupBy('tipo')
            ->orderByDesc('total')
            ->get();

        $resumenDestinatarios = Mensaje::select('destinatario', DB::raw('COUNT(*) as total'))
            ->groupBy('destinatario')
            ->orderByDesc('total')
            ->get();

        $pendientes = Mensaje::where('enviado', false)
            ->orderBy('fecha_programada')
            ->limit(5)
            ->get();

        return view('admin.mensajes.index', [
            'mensajes' => $mensajes,
            'stats' => $stats,
            'resumenTipos' => $resumenTipos,
            'resumenDestinatarios' => $resumenDestinatarios,
            'pendientes' => $pendientes,
            'filtros' => [
                'estado' => $estado,
                'tipo' => $tipo,
                'destinatario' => $destinatario,
                'orden' => $orden,
                'q' => $busqueda,
                'por_pagina' => $porPagina,
            ],
        ]);
    }
}
