<?php

namespace App\Http\Controllers;

use App\Models\Granja;
use App\Models\Scawatt;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;

class AdminScawattsController extends Controller
{
    public function index(Request $request)
    {
        $estado = $request->query('estado', 'todos');
        $granjaId = $request->query('granja', 'todas');
        $tipo = $request->query('tipo', 'todos');
        $orden = $request->query('orden', 'recientes');
        $busqueda = trim((string) $request->query('q', ''));
        $porPagina = (int) $request->query('por_pagina', 25);
        $porPagina = $porPagina > 0 ? min($porPagina, 100) : 25;

        $scawattsQuery = Scawatt::query()
            ->with([
                'usuario:id,nombre,email',
                'granja:id,nombre,departamento,municipio,estado',
            ]);

        if ($estado !== 'todos') {
            $scawattsQuery->where('estado', $estado);
        }

        if ($granjaId !== 'todas' && $granjaId !== null) {
            $scawattsQuery->where('granja_id', $granjaId);
        }

        if ($tipo !== 'todos') {
            $scawattsQuery->where('tipo', $tipo);
        }

        if ($busqueda !== '') {
            $scawattsQuery->where(function (Builder $query) use ($busqueda) {
                $query->where('scawatt_id', 'like', "%{$busqueda}%")
                      ->orWhereHas('usuario', function (Builder $subQuery) use ($busqueda) {
                          $subQuery->where('nombre', 'like', "%{$busqueda}%")
                                   ->orWhere('email', 'like', "%{$busqueda}%");
                      });
            });
        }

        $scawattsQuery->when($orden, function (Builder $query) use ($orden) {
            return match ($orden) {
                'valor_desc' => $query->orderByDesc('valor_actual'),
                'valor_asc' => $query->orderBy('valor_actual'),
                'kwh_desc' => $query->orderByDesc('kwh_asignados'),
                'kwh_asc' => $query->orderBy('kwh_asignados'),
                'valorizacion' => $query->orderByDesc(DB::raw('valor_actual - valor_inicial')),
                default => $query->orderByDesc('created_at'),
            };
        });

        $scawatts = $scawattsQuery->paginate($porPagina)->withQueryString();

        $stats = [
            'total' => Scawatt::count(),
            'activos' => Scawatt::where('estado', 'activo')->count(),
            'marketplace' => Scawatt::where('estado', 'marketplace')->count(),
            'valor_actual' => $this->formatCurrency(Scawatt::sum('valor_actual')),
            'valor_inicial' => $this->formatCurrency(Scawatt::sum('valor_inicial')),
            'valorizacion' => $this->formatCurrency(
                Scawatt::selectRaw('SUM(valor_actual - valor_inicial) as total')->value('total')
            ),
            'kwh_total' => $this->formatNumber(Scawatt::sum('kwh_asignados')),
        ];

        $porEstado = Scawatt::select('estado',
                DB::raw('COUNT(*) as total'),
                DB::raw('SUM(valor_actual) as valor_actual')
            )
            ->groupBy('estado')
            ->orderByDesc('total')
            ->get()
            ->map(function ($row) {
                return [
                    'estado' => $row->estado ?? 'sin_estado',
                    'total' => (int) $row->total,
                    'valor' => $this->formatCurrency($row->valor_actual),
                ];
            });

        $porTipo = Scawatt::select('tipo', DB::raw('COUNT(*) as total'))
            ->groupBy('tipo')
            ->orderByDesc('total')
            ->get();

        $granjas = Granja::select('id', 'nombre')->orderBy('nombre')->get();

        $topGranjas = Granja::select('granjas.id', 'granjas.nombre')
            ->withCount('scawatts')
            ->withSum('scawatts', 'valor_actual')
            ->orderByDesc('scawatts_count')
            ->limit(5)
            ->get();

        return view('admin.scawatts.index', [
            'scawatts' => $scawatts,
            'stats' => $stats,
            'porEstado' => $porEstado,
            'porTipo' => $porTipo,
            'granjas' => $granjas,
            'topGranjas' => $topGranjas,
            'filtros' => [
                'estado' => $estado,
                'granja' => $granjaId,
                'tipo' => $tipo,
                'orden' => $orden,
                'q' => $busqueda,
                'por_pagina' => $porPagina,
            ],
        ]);
    }

    private function formatCurrency($value): string
    {
        $numeric = $value !== null ? (float) $value : 0.0;
        return '$' . number_format($numeric, 0, ',', '.');
    }

    private function formatNumber($value): string
    {
        $numeric = $value !== null ? (float) $value : 0.0;
        return number_format($numeric, 2, ',', '.');
    }

    public function store(Request $request)
    {
        $estadoPermitidos = ['activo', 'marketplace', 'reservado', 'vendido', 'bloqueado'];

        $validated = $request->validate([
            'scawatt_id' => ['required', 'string', 'max:100', 'unique:scawatts,scawatt_id'],
            'usuario_id' => ['nullable', 'integer', Rule::exists('usuarios', 'id')],
            'granja_id' => ['nullable', 'integer', Rule::exists('granjas', 'id')],
            'estado' => ['required', 'string', Rule::in($estadoPermitidos)],
            'tipo' => ['nullable', 'string', 'max:50'],
            'kwh_asignados' => ['required', 'numeric', 'min:0'],
            'valor_inicial' => ['required', 'numeric', 'min:0'],
            'valor_actual' => ['nullable', 'numeric', 'min:0'],
            'fecha_inicio' => ['nullable', 'date'],
            'fecha_final' => ['nullable', 'date', 'after_or_equal:fecha_inicio'],
        ], [], [
            'scawatt_id' => 'serial',
            'usuario_id' => 'propietario',
            'granja_id' => 'granja',
            'kwh_asignados' => 'kWh asignados',
            'valor_inicial' => 'valor inicial',
            'valor_actual' => 'valor actual',
        ]);

        $valorActual = array_key_exists('valor_actual', $validated) && $validated['valor_actual'] !== null
            ? (float) $validated['valor_actual']
            : (float) $validated['valor_inicial'];

        $diferencia = $valorActual - (float) $validated['valor_inicial'];

        $payload = [
            'scawatt_id' => $validated['scawatt_id'],
            'usuario_id' => $validated['usuario_id'] ?? null,
            'granja_id' => $validated['granja_id'] ?? null,
            'estado' => $validated['estado'],
            'tipo' => $validated['tipo'] ?? null,
            'kwh_asignados' => (float) $validated['kwh_asignados'],
            'valor_inicial' => (float) $validated['valor_inicial'],
            'valor_actual' => $valorActual,
            'fecha_inicio' => $validated['fecha_inicio'] ?? null,
            'fecha_final' => $validated['fecha_final'] ?? null,
            'valorizaciones' => $diferencia !== 0.0 ? [$diferencia] : [],
        ];

        DB::transaction(function () use ($payload) {
            Scawatt::create($payload);
        });

        return back()->with('status', 'ScaWatt creado correctamente.');
    }

    public function update(Request $request, Scawatt $scawatt)
    {
        $estadoPermitidos = ['activo', 'marketplace', 'reservado', 'vendido', 'bloqueado'];

        $validated = $request->validate([
            'granja_id' => ['nullable', 'integer', Rule::exists('granjas', 'id')],
            'estado' => ['required', 'string', Rule::in($estadoPermitidos)],
            'tipo' => ['nullable', 'string', 'max:50'],
            'kwh_asignados' => ['required', 'numeric', 'min:0'],
            'valor_inicial' => ['required', 'numeric', 'min:0'],
            'valor_actual' => ['required', 'numeric', 'min:0'],
            'fecha_inicio' => ['nullable', 'date'],
            'fecha_final' => ['nullable', 'date'],
        ], [], [
            'granja_id' => 'granja',
            'kwh_asignados' => 'kWh asignados',
        ]);

        $payload = [
            'granja_id' => $validated['granja_id'] ?? null,
            'estado' => $validated['estado'],
            'tipo' => $validated['tipo'] ?? null,
            'kwh_asignados' => $validated['kwh_asignados'],
            'valor_inicial' => $validated['valor_inicial'],
            'valor_actual' => $validated['valor_actual'],
            'fecha_inicio' => $validated['fecha_inicio'] ?? null,
            'fecha_final' => $validated['fecha_final'] ?? null,
        ];

        DB::transaction(function () use ($scawatt, $payload) {
            $scawatt->fill($payload);

            $valorizaciones = $scawatt->valorizaciones ?? [];
            $valorizaciones = is_array($valorizaciones) ? $valorizaciones : (array) $valorizaciones;
            $nuevaValorizacion = (float) $scawatt->valor_actual - (float) $scawatt->valor_inicial;

            if (empty($valorizaciones) || end($valorizaciones) !== $nuevaValorizacion) {
                $valorizaciones[] = $nuevaValorizacion;
            }

            $scawatt->valorizaciones = $valorizaciones;
            $scawatt->save();
        });

        return back()->with('status', 'ScaWatt actualizado correctamente.');
    }
}
