<?php

namespace App\Http\Controllers;

use App\Models\SolicitudContactoScawatt;
use App\Models\Usuario;
use App\Services\SolicitudScawattWorkflowService;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;

class AdminSolicitudesScawattController extends Controller
{
    public function actualizar(
        Request $request,
        Usuario $usuario,
        SolicitudContactoScawatt $solicitud,
        SolicitudScawattWorkflowService $workflow
    ): RedirectResponse {
        if ($solicitud->usuario_id !== $usuario->id) {
            abort(404);
        }

        /** @var Usuario $admin */
        $admin = $request->user();
        if (!$admin || !$admin->esAdministrador()) {
            abort(403);
        }

        $accion = $request->input('accion');

        try {
            return match ($accion) {
                'datos_pendientes' => $this->procesarDatosPendientes($request, $solicitud, $admin, $workflow),
                'liberar_pago' => $this->procesarLiberarPago($request, $solicitud, $admin, $workflow),
                'pago_revision' => $this->procesarPagoRevision($request, $solicitud, $admin, $workflow),
                'confirmar_pago' => $this->procesarConfirmarPago($request, $solicitud, $admin, $workflow),
                'cancelar' => $this->procesarCancelar($request, $solicitud, $admin, $workflow),
                default => redirect()->back()->withErrors(['accion' => 'Accion no soportada.']),
            };
        } catch (ValidationException $exception) {
            return redirect()->back()->withErrors($exception->errors())->withInput();
        } catch (\Throwable $exception) {
            report($exception);

            return redirect()->back()->withErrors([
                'accion' => 'No se pudo completar la accion solicitada. ' . $exception->getMessage(),
            ]);
        }
    }

    private function procesarDatosPendientes(
        Request $request,
        SolicitudContactoScawatt $solicitud,
        Usuario $admin,
        SolicitudScawattWorkflowService $workflow
    ): RedirectResponse {
        $data = $request->validate([
            'nota' => ['nullable', 'string', 'max:500'],
        ]);

        $workflow->marcarDatosPendientes($solicitud, $admin, $data['nota'] ?? null);

        return redirect()->back()->with('status', 'Se solicito al usuario completar la informacion.');
    }

    private function procesarLiberarPago(
        Request $request,
        SolicitudContactoScawatt $solicitud,
        Usuario $admin,
        SolicitudScawattWorkflowService $workflow
    ): RedirectResponse {
        $data = $request->validate([
            'link_pago' => ['required', 'url', 'max:500'],
            'monto_pago' => ['required', 'numeric', 'min:0'],
            'nota' => ['nullable', 'string', 'max:500'],
        ]);

        $workflow->liberarPago(
            $solicitud,
            $admin,
            $data['link_pago'],
            (float) $data['monto_pago'],
            $data['nota'] ?? null
        );

        return redirect()->back()->with('status', 'Se envio el enlace de pago al usuario.');
    }

    private function procesarPagoRevision(
        Request $request,
        SolicitudContactoScawatt $solicitud,
        Usuario $admin,
        SolicitudScawattWorkflowService $workflow
    ): RedirectResponse {
        $data = $request->validate([
            'nota' => ['nullable', 'string', 'max:500'],
        ]);

        $workflow->marcarPagoEnRevision($solicitud, $admin, $data['nota'] ?? null);

        return redirect()->back()->with('status', 'Se marco el pago como en revision.');
    }

    private function procesarConfirmarPago(
        Request $request,
        SolicitudContactoScawatt $solicitud,
        Usuario $admin,
        SolicitudScawattWorkflowService $workflow
    ): RedirectResponse {
        $data = $request->validate([
            'monto_pago' => ['nullable', 'numeric', 'min:0'],
            'nota' => ['nullable', 'string', 'max:500'],
        ]);

        $workflow->confirmarPago(
            $solicitud,
            $admin,
            $data['monto_pago'] !== null ? (float) $data['monto_pago'] : null,
            $data['nota'] ?? null
        );

        return redirect()->back()->with('status', 'Pago confirmado y ScaWatts generados.');
    }

    private function procesarCancelar(
        Request $request,
        SolicitudContactoScawatt $solicitud,
        Usuario $admin,
        SolicitudScawattWorkflowService $workflow
    ): RedirectResponse {
        $data = $request->validate([
            'motivo' => ['required', 'string', 'max:500'],
        ]);

        $workflow->cancelar($solicitud, $admin, $data['motivo']);

        return redirect()->back()->with('status', 'Solicitud cancelada.');
    }
}
