<?php

namespace App\Http\Controllers;

use App\Models\Scawatt;
use App\Models\SolicitudContactoScawatt;
use App\Models\Usuario;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class AdminUsuariosController extends Controller
{
    public function index(Request $request)
    {
        $estado = $request->query('estado', 'todos');
        $verificacion = $request->query('verificacion', 'todos');
        $orden = $request->query('orden', 'recientes');
        $busqueda = trim((string) $request->query('q', ''));
        $porPagina = (int) $request->query('por_pagina', 20);
        $porPagina = $porPagina > 0 ? min($porPagina, 100) : 20;

        $usuariosQuery = Usuario::query()
            ->withCount('scawatts')
            ->withSum(['scawatts as valor_portafolio' => function (Builder $query) {
                $query->whereNotNull('valor_actual');
            }], 'valor_actual')
            ->withSum(['depositos as total_depositos' => function (Builder $query) {
                $query->where('estado', 'aprobado');
            }], 'monto')
            ->withSum(['retiros as total_retiros' => function (Builder $query) {
                $query->where('estado', 'completado');
            }], 'monto');

        if ($estado !== 'todos') {
            $usuariosQuery->where('estado', $estado);
        }

        if ($verificacion !== 'todos') {
            $usuariosQuery->where(function (Builder $query) use ($verificacion) {
                return match ($verificacion) {
                    'completos' => $query->whereNotNull('cedula_verificada_at')
                                            ->whereNotNull('datos_bancarios_verificados_at'),
                    'cedula' => $query->whereNotNull('cedula_verificada_at')
                                       ->whereNull('datos_bancarios_verificados_at'),
                    'bancarios' => $query->whereNotNull('datos_bancarios_verificados_at')
                                         ->whereNull('cedula_verificada_at'),
                    'pendientes' => $query->whereNull('cedula_verificada_at')
                                           ->orWhereNull('datos_bancarios_verificados_at'),
                    default => $query,
                };
            });
        }

        if ($busqueda !== '') {
            $usuariosQuery->where(function (Builder $query) use ($busqueda) {
                $query->where('nombre', 'like', "%{$busqueda}%")
                      ->orWhere('email', 'like', "%{$busqueda}%")
                      ->orWhere('cedula', 'like', "%{$busqueda}%")
                      ->orWhere('telefono', 'like', "%{$busqueda}%");
            });
        }

        $usuariosQuery->when($orden, function (Builder $query) use ($orden) {
            return match ($orden) {
                'saldo_desc' => $query->orderByDesc('saldo_total'),
                'saldo_asc' => $query->orderBy('saldo_total'),
                'nombre' => $query->orderBy('nombre'),
                'actividad' => $query->orderByDesc('updated_at'),
                default => $query->orderByDesc('created_at'),
            };
        });

        $usuarios = $usuariosQuery->paginate($porPagina)->withQueryString();

        $hoy = Carbon::now();
        $hace30Dias = $hoy->copy()->subDays(30);

        $stats = [
            'total' => Usuario::count(),
            'activos' => Usuario::where('estado', 'activo')->count(),
            'nuevos_mes' => Usuario::where('created_at', '>=', $hace30Dias)->count(),
            'verificados' => Usuario::whereNotNull('cedula_verificada_at')
                                     ->whereNotNull('datos_bancarios_verificados_at')
                                     ->count(),
            'saldo_total' => $this->formatCurrency(Usuario::sum('saldo_total')),
            'saldo_disponible' => $this->formatCurrency(Usuario::sum('saldo_disponible')),
            'saldo_pendiente' => $this->formatCurrency(Usuario::sum('saldo_pendiente')),
        ];

        $verificaciones = [
            'cedula' => Usuario::whereNotNull('cedula_verificada_at')->count(),
            'bancarios' => Usuario::whereNotNull('datos_bancarios_verificados_at')->count(),
            'completos' => Usuario::whereNotNull('cedula_verificada_at')
                                  ->whereNotNull('datos_bancarios_verificados_at')
                                  ->count(),
            'pendientes' => Usuario::where(function (Builder $query) {
                $query->whereNull('cedula_verificada_at')
                      ->orWhereNull('datos_bancarios_verificados_at');
            })->count(),
        ];

        $segmentacionEstados = Usuario::select('estado', DB::raw('COUNT(*) as total'))
            ->groupBy('estado')
            ->orderByDesc('total')
            ->get()
            ->map(function ($row) {
                return [
                    'estado' => $row->estado ?? 'sin_estado',
                    'total' => (int) $row->total,
                ];
            });

        $topClientes = Usuario::select('id', 'nombre', 'email', 'saldo_total', 'saldo_disponible', 'saldo_pendiente')
            ->orderByDesc('saldo_total')
            ->limit(6)
            ->get();

        $usuariosDisponibles = Usuario::select('id', 'nombre', 'email')
            ->orderBy('nombre')
            ->get();

        return view('admin.usuarios.index', [
            'usuarios' => $usuarios,
            'stats' => $stats,
            'verificaciones' => $verificaciones,
            'segmentacionEstados' => $segmentacionEstados,
            'topClientes' => $topClientes,
            'usuariosDisponibles' => $usuariosDisponibles,
            'filtros' => [
                'estado' => $estado,
                'verificacion' => $verificacion,
                'orden' => $orden,
                'q' => $busqueda,
                'por_pagina' => $porPagina,
            ],
        ]);
    }

    public function show(Usuario $usuario)
    {
        $usuario->load([
            'scawatts' => function ($query) {
                $query->with('granja')->orderByDesc('created_at');
            },
        ]);

        $depositos = $usuario->depositos()->orderByDesc('created_at')->limit(10)->get();
        $retiros = $usuario->retiros()->orderByDesc('created_at')->limit(10)->get();
        $solicitudes = SolicitudContactoScawatt::with(['adminAtendio', 'estadoActualizadoPor'])
            ->where('usuario_id', $usuario->id)
            ->orderByDesc('created_at')
            ->limit(10)
            ->get();
        $notificaciones = $usuario->notificaciones()->orderByDesc('created_at')->limit(8)->get();

        $depositosMontos = DB::table('depositos')
            ->selectRaw("COALESCE(SUM(CASE WHEN estado = 'aprobado' THEN monto ELSE 0 END), 0) AS aprobados")
            ->selectRaw("COALESCE(SUM(CASE WHEN estado = 'pendiente' THEN monto ELSE 0 END), 0) AS pendientes")
            ->selectRaw("COALESCE(SUM(CASE WHEN estado = 'rechazado' THEN monto ELSE 0 END), 0) AS rechazados")
            ->where('usuario_id', $usuario->id)
            ->first();

        $retirosMontos = DB::table('retiros')
            ->selectRaw("COALESCE(SUM(CASE WHEN estado = 'completado' THEN monto ELSE 0 END), 0) AS completados")
            ->selectRaw("COALESCE(SUM(CASE WHEN estado = 'pendiente' THEN monto ELSE 0 END), 0) AS pendientes")
            ->selectRaw("COALESCE(SUM(CASE WHEN estado = 'rechazado' THEN monto ELSE 0 END), 0) AS rechazados")
            ->where('usuario_id', $usuario->id)
            ->first();

        $scawattAggregate = $usuario->scawatts()
            ->selectRaw('COALESCE(SUM(kwh_asignados), 0) AS kwh_total, COALESCE(SUM(valor_actual), 0) AS valor_actual_total, COALESCE(SUM(valor_inicial), 0) AS valor_inicial_total')
            ->first();

        $documentosRegistro = DB::table('documentos')
            ->select('id', 'tipo', 'nombre_archivo', 'ruta_archivo', 'url_publica', 'fecha_generacion', 'created_at')
            ->where('usuario_id', $usuario->id)
            ->orderByDesc('created_at')
            ->get();

        $documentosAdicionales = $documentosRegistro->map(function ($doc) {
            $fecha = !empty($doc->created_at) ? Carbon::parse($doc->created_at) : null;

            return [
                'id' => $doc->id,
                'tipo' => $doc->tipo,
                'nombre' => $doc->nombre_archivo,
                'url' => $doc->url_publica ?: $this->resolvePublicUrl($doc->ruta_archivo),
                'fecha' => $fecha,
                'fecha_generacion' => !empty($doc->fecha_generacion) ? Carbon::parse($doc->fecha_generacion) : $fecha,
            ];
        })->filter(function ($doc) {
            return !empty($doc['url']);
        })->values();

        $documentosIdentidad = [
            'frente_url' => $this->resolvePublicUrl($usuario->cedula_foto_frente),
            'atras_url' => $this->resolvePublicUrl($usuario->cedula_foto_atras),
            'verificada' => (bool) $usuario->cedula_verificada_at,
            'fecha_verificacion' => $usuario->cedula_verificada_at,
        ];

        $numeroCuenta = $usuario->numero_cuenta ? preg_replace('/\s+/', '', (string) $usuario->numero_cuenta) : null;
        $segmentoCuenta = $numeroCuenta && strlen($numeroCuenta) > 4
            ? '****' . substr($numeroCuenta, -4)
            : $numeroCuenta;

        $documentosBancarios = [
            'banco' => $usuario->banco,
            'tipo_cuenta' => $usuario->tipo_cuenta,
            'numero_cuenta' => $numeroCuenta,
            'numero_cuenta_parcial' => $segmentoCuenta,
            'certificado_url' => $this->resolvePublicUrl($usuario->certificado_bancario),
            'verificada' => (bool) $usuario->datos_bancarios_verificados_at,
            'fecha_verificacion' => $usuario->datos_bancarios_verificados_at,
        ];

        $resumenFinanciero = [
            'saldo_disponible' => $this->formatCurrency($usuario->saldo_disponible),
            'saldo_pendiente' => $this->formatCurrency($usuario->saldo_pendiente),
            'saldo_total' => $this->formatCurrency($usuario->saldo_total),
            'depositos_aprobados' => $this->formatCurrency(optional($depositosMontos)->aprobados ?? 0),
            'depositos_pendientes' => $this->formatCurrency(optional($depositosMontos)->pendientes ?? 0),
            'retiros_completados' => $this->formatCurrency(optional($retirosMontos)->completados ?? 0),
            'retiros_pendientes' => $this->formatCurrency(optional($retirosMontos)->pendientes ?? 0),
            'valor_portafolio' => $this->formatCurrency(optional($scawattAggregate)->valor_actual_total ?? 0),
            'valorizacion' => $this->formatCurrency(
                (optional($scawattAggregate)->valor_actual_total ?? 0)
                - (optional($scawattAggregate)->valor_inicial_total ?? 0)
            ),
        ];

        $estadisticasScawatt = [
            'total' => $usuario->scawatts->count(),
            'activos' => $usuario->scawatts->where('estado', 'activo')->count(),
            'inactivos' => $usuario->scawatts->where('estado', '!=', 'activo')->count(),
            'kwh_total' => number_format((float) (optional($scawattAggregate)->kwh_total ?? 0), 2, ',', '.'),
            'valor_actual_total' => $this->formatCurrency(optional($scawattAggregate)->valor_actual_total ?? 0),
            'valor_inicial_total' => $this->formatCurrency(optional($scawattAggregate)->valor_inicial_total ?? 0),
            'valorizacion_total' => $this->formatCurrency(
                (optional($scawattAggregate)->valor_actual_total ?? 0)
                - (optional($scawattAggregate)->valor_inicial_total ?? 0)
            ),
        ];

        $timeline = collect();

        foreach ($depositos as $deposito) {
            $timeline->push([
                'tipo' => 'deposito',
                'titulo' => 'Depósito ' . ucfirst($deposito->estado ?? 'pendiente'),
                'descripcion' => 'Monto ' . $this->formatCurrency($deposito->monto) . ($deposito->referencia ? ' · Ref ' . $deposito->referencia : ''),
                'fecha' => $deposito->created_at,
                'estado' => $deposito->estado,
            ]);
        }

        foreach ($retiros as $retiro) {
            $timeline->push([
                'tipo' => 'retiro',
                'titulo' => 'Retiro ' . ucfirst($retiro->estado ?? 'pendiente'),
                'descripcion' => 'Monto ' . $this->formatCurrency($retiro->monto) . ($retiro->referencia_transaccion ? ' · Ref ' . $retiro->referencia_transaccion : ''),
                'fecha' => $retiro->created_at,
                'estado' => $retiro->estado,
            ]);
        }

        foreach ($solicitudes as $solicitud) {
            $timeline->push([
                'tipo' => 'solicitud',
                'titulo' => 'Solicitud ScaWatt ' . ucfirst($solicitud->estado ?? 'pendiente'),
                'descripcion' => 'Presupuesto ' . ($solicitud->presupuesto_aproximado ? $this->formatCurrency($solicitud->presupuesto_aproximado) : 'Sin definir'),
                'fecha' => $solicitud->created_at,
                'estado' => $solicitud->estado,
            ]);
        }

        foreach ($documentosAdicionales as $doc) {
            $timeline->push([
                'tipo' => 'documento',
                'titulo' => 'Documento ' . ucfirst((string) $doc['tipo']),
                'descripcion' => $doc['nombre'],
                'fecha' => $doc['fecha'],
                'estado' => 'registrado',
            ]);
        }

        $timeline = $timeline
            ->filter(fn ($evento) => !empty($evento['fecha']))
            ->sortByDesc('fecha')
            ->take(15)
            ->values();

        return view('admin.usuarios.show', [
            'usuario' => $usuario,
            'resumenFinanciero' => $resumenFinanciero,
            'documentosIdentidad' => $documentosIdentidad,
            'documentosBancarios' => $documentosBancarios,
            'documentosAdicionales' => $documentosAdicionales,
            'depositos' => $depositos,
            'retiros' => $retiros,
            'solicitudes' => $solicitudes,
            'notificaciones' => $notificaciones,
            'timeline' => $timeline,
            'scawatts' => $usuario->scawatts,
            'estadisticasScawatt' => $estadisticasScawatt,
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'nombre' => ['required', 'string', 'max:255'],
            'email' => ['required', 'email', 'max:255', 'unique:usuarios,email'],
            'telefono' => ['nullable', 'string', 'max:45'],
            'estado' => ['required', 'string', Rule::in(['activo', 'inactivo', 'bloqueado', 'admin'])],
            'password' => ['nullable', 'string', 'min:8', 'max:255'],
        ]);

        $password = $validated['password'] ?? Str::random(12);

        $usuario = Usuario::create([
            'nombre' => $validated['nombre'],
            'email' => $validated['email'],
            'telefono' => $validated['telefono'] ?? null,
            'estado' => $validated['estado'],
            'password' => $password,
        ]);

        $redirect = redirect()
            ->route('admin.usuarios')
            ->with('status', "Usuario {$usuario->nombre} creado exitosamente.");

        if (empty($validated['password'])) {
            $redirect->with('generated_password', $password);
        }

        return $redirect;
    }

    public function update(Request $request, Usuario $usuario)
    {
        $validated = $request->validate([
            'nombre' => ['required', 'string', 'max:255'],
            'email' => ['required', 'email', 'max:255', Rule::unique('usuarios', 'email')->ignore($usuario->id)],
            'telefono' => ['nullable', 'string', 'max:45'],
            'estado' => ['required', 'string', Rule::in(['activo', 'inactivo', 'bloqueado', 'admin'])],
            'password' => ['nullable', 'string', 'min:8', 'max:255'],
        ]);

        $usuario->nombre = $validated['nombre'];
        $usuario->email = $validated['email'];
        $usuario->telefono = $validated['telefono'] ?? null;
        $usuario->estado = $validated['estado'];

        if (!empty($validated['password'])) {
            $usuario->password = $validated['password'];
        }

        $usuario->save();

        return redirect()
            ->back()
            ->with('status', "Usuario {$usuario->nombre} actualizado exitosamente.");
    }

    public function actualizarEstado(Request $request, Usuario $usuario)
    {
        $validated = $request->validate([
            'estado' => ['required', Rule::in(['activo', 'inactivo', 'bloqueado', 'admin'])],
        ]);

        $usuario->estado = $validated['estado'];
        $usuario->save();

        return redirect()
            ->back()
            ->with('status', "Estado de {$usuario->nombre} actualizado a {$usuario->estado}.");
    }

    public function actualizarVerificacionCedula(Request $request, Usuario $usuario)
    {
        $validated = $request->validate([
            'accion' => ['required', Rule::in(['verificar', 'revertir'])],
        ]);

        $usuario->cedula_verificada_at = $validated['accion'] === 'verificar' ? Carbon::now() : null;
        $usuario->save();

        $mensaje = $validated['accion'] === 'verificar'
            ? 'Cedula verificada correctamente.'
            : 'Verificacion de cedula revertida.';

        return redirect()->back()->with('status', $mensaje);
    }

    public function actualizarVerificacionBancaria(Request $request, Usuario $usuario)
    {
        $validated = $request->validate([
            'accion' => ['required', Rule::in(['verificar', 'revertir'])],
        ]);

        $usuario->datos_bancarios_verificados_at = $validated['accion'] === 'verificar' ? Carbon::now() : null;
        $usuario->save();

        $mensaje = $validated['accion'] === 'verificar'
            ? 'Cuenta bancaria verificada correctamente.'
            : 'Verificacion bancaria revertida.';

        return redirect()->back()->with('status', $mensaje);
    }

    public function actualizarEstadoScawatt(Request $request, Usuario $usuario, Scawatt $scawatt)
    {
        if ($scawatt->usuario_id !== $usuario->id) {
            abort(404);
        }

        $validated = $request->validate([
            'estado' => ['required', Rule::in(['activo', 'marketplace', 'reservado', 'vendido', 'bloqueado'])],
        ]);

        $scawatt->estado = $validated['estado'];
        $scawatt->save();

        return redirect()
            ->back()
            ->with('status', "Estado de ScaWatt {$scawatt->scawatt_id} actualizado a {$scawatt->estado}.");
    }

    private function formatCurrency($value): string
    {
        $numeric = $value !== null ? (float) $value : 0.0;
        return '$' . number_format($numeric, 0, ',', '.');
    }

    private function resolvePublicUrl(?string $path): ?string
    {
        if (empty($path)) {
            return null;
        }

        if (filter_var($path, FILTER_VALIDATE_URL)) {
            return $path;
        }

        if (Storage::disk('public')->exists($path)) {
            return Storage::disk('public')->url($path);
        }

        if (Storage::exists($path)) {
            return Storage::url($path);
        }

        return null;
    }
}
