<?php

namespace App\Http\Controllers;

use App\Models\Marketplace;
use App\Models\Scawatt;
use App\Models\Usuario;
use App\Models\Notificacion;
use App\Models\AuditoriaAdmin;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Http\Client\ConnectionException;

class MarketplaceController extends Controller
{
    private $pythonApiUrl = 'http://localhost:5004';
    private $usuariosAppPath = '/home/scalarapp/public_html/scalar-api/api-scawatts/produccion/scausers/usuarios-app';
    private $notificacionesEmail = 'notificaciones@scalar.com.co';

    /**
     * Listar ScaWatts disponibles en marketplace
     * GET /api/marketplace
     */
    public function listar(Request $request)
    {
        $query = Marketplace::with(['scawatt', 'vendedor'])
            ->where('estado', 'en_venta')
            ->whereNull('comprador_id');

        // Filtros
        if ($request->has('precio_min')) {
            $query->where('precio_venta', '>=', $request->precio_min);
        }

        if ($request->has('precio_max')) {
            $query->where('precio_venta', '<=', $request->precio_max);
        }

        if ($request->has('ordenar')) {
            switch ($request->ordenar) {
                case 'precio_asc':
                    $query->orderBy('precio_venta', 'asc');
                    break;
                case 'precio_desc':
                    $query->orderBy('precio_venta', 'desc');
                    break;
                case 'reciente':
                    $query->orderBy('fecha_publicacion', 'desc');
                    break;
                default:
                    $query->orderBy('created_at', 'desc');
            }
        } else {
            $query->orderBy('created_at', 'desc');
        }

        $publicaciones = $query->paginate(20);

        return response()->json($publicaciones);
    }

    /**
     * Ver mis publicaciones en marketplace
     * GET /api/marketplace/mis-publicaciones/{usuario_id}
     */
    public function misPublicaciones($usuarioId)
    {
        $publicaciones = Marketplace::with('scawatt')
            ->where('vendedor_id', $usuarioId)
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return response()->json($publicaciones);
    }

    /**
     * Ver mis compras en marketplace
     * GET /api/marketplace/mis-compras/{usuario_id}
     */
    public function misCompras($usuarioId)
    {
        $compras = Marketplace::with(['scawatt', 'vendedor'])
            ->where('comprador_id', $usuarioId)
            ->where('estado', 'vendido')
            ->orderBy('fecha_venta', 'desc')
            ->paginate(20);

        return response()->json($compras);
    }

    /**
     * Publicar ScaWatt en marketplace
     * POST /api/marketplace/publicar
     * Body: { "scawatt_serial": "SCA-XXX", "usuario_id": 2, "precio_venta": 45000000 }
     */
    public function publicar(Request $request)
    {
        $request->validate([
            'scawatt_serial' => 'required|string',
            'usuario_id' => 'required|exists:usuarios,id',
            'precio_venta' => 'nullable|numeric|min:0.01',
            'precio' => 'nullable|numeric|min:0.01',
            'descripcion' => 'nullable|string|max:500'
        ]);

        $precioVenta = $request->precio_venta ?? $request->precio;

        if ($precioVenta === null) {
            return response()->json([
                'success' => false,
                'message' => 'Debes indicar un precio de venta válido'
            ], 422);
        }

        try {
            // Buscar ScaWatt por serial (no por ID interno)
            $scawatt = Scawatt::where('scawatt_id', $request->scawatt_serial)->first();
            
            if (!$scawatt) {
                return response()->json([
                    'success' => false,
                    'message' => 'ScaWatt no encontrado. Verifica el serial.'
                ], 404);
            }

            // Verificar que el ScaWatt pertenece al vendedor
            if ($scawatt->usuario_id != $request->usuario_id) {
                return response()->json([
                    'success' => false,
                    'message' => 'No tienes permisos para publicar este ScaWatt'
                ], 403);
            }

            // Verificar que el ScaWatt no esté ya en marketplace
            $existente = Marketplace::where('scawatt_id', $scawatt->id)
                ->where('estado', 'en_venta')
                ->whereNull('comprador_id')
                ->first();

            if ($existente) {
                return response()->json([
                    'success' => false,
                    'message' => 'Este ScaWatt ya está publicado en el marketplace'
                ], 400);
            }

            DB::beginTransaction();

            $marketplace = Marketplace::create([
                'scawatt_id' => $scawatt->id,
                'vendedor_id' => $request->usuario_id,
                'precio_venta' => $precioVenta,
                'descripcion' => $request->descripcion ?? "Publicación automática del {$scawatt->scawatt_id}",
                'estado' => 'en_venta',
                'fecha_publicacion' => now()
            ]);

            $marketplace->publicar();

            // Notificar al vendedor
            Notificacion::create([
                'usuario_id' => $request->usuario_id,
                'tipo' => 'success',
                'titulo' => 'ScaWatt publicado',
                'mensaje' => "Tu ScaWatt {$scawatt->scawatt_id} ha sido publicado en el marketplace por $" . number_format($precioVenta, 2),
                'prioridad' => 'media',
                'icono' => 'storefront',
                'datos_adicionales' => [
                    'tipo' => 'marketplace',
                    'marketplace_id' => $marketplace->id,
                    'scawatt_serial' => $scawatt->scawatt_id
                ]
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'ScaWatt publicado exitosamente',
                'publicacion' => $marketplace->fresh(['scawatt'])
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();

            Log::error('Error al publicar ScaWatt en marketplace', [
                'payload' => $request->all(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Error al publicar ScaWatt',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Comprar ScaWatt del marketplace
     * POST /api/marketplace/comprar/{id}
     */
    public function comprar(Request $request, $id)
    {
        $request->validate([
            'comprador_id' => 'required|exists:usuarios,id'
        ]);

        try {
            $publicacion = Marketplace::with(['scawatt', 'vendedor'])->findOrFail($id);

            // Verificar que la publicación esté disponible
            if ($publicacion->estado != 'en_venta' || $publicacion->comprador_id != null) {
                return response()->json([
                    'success' => false,
                    'message' => 'Este ScaWatt ya no está disponible'
                ], 400);
            }

            // Verificar que el comprador no sea el vendedor
            if ($publicacion->vendedor_id == $request->comprador_id) {
                return response()->json([
                    'success' => false,
                    'message' => 'No puedes comprar tu propio ScaWatt'
                ], 400);
            }

            $comprador = Usuario::findOrFail($request->comprador_id);

            // Verificar saldo del comprador
            if ($comprador->saldo_disponible < $publicacion->precio_venta) {
                return response()->json([
                    'success' => false,
                    'message' => 'Saldo insuficiente',
                    'saldo_disponible' => $comprador->saldo_disponible,
                    'precio_venta' => $publicacion->precio_venta
                ], 400);
            }

            DB::beginTransaction();

            $scawattSerial = $publicacion->scawatt->scawatt_id;
            $vendedor = $publicacion->vendedor;

            // Ejecutar la venta (actualiza saldos, cambia propietario, calcula comisión)
            $publicacion->vender($request->comprador_id);

            $comprador->refresh();
            $publicacion->load('scawatt');

            // Registrar auditoría
            if ($request->has('admin_id')) {
                    AuditoriaAdmin::registrar(
                        $request->admin_id,
                        'venta_marketplace',
                        'marketplace',
                        "Venta ejecutada: ScaWatt {$publicacion->scawatt->scawatt_id} vendido por $" . number_format($publicacion->precio_venta, 2) . " (Comisión: $" . number_format($publicacion->comision_plataforma, 2) . ")",
                        'Marketplace',
                        $publicacion->id
                    );
            }

            // Notificar al vendedor
            Notificacion::create([
                'usuario_id' => $publicacion->vendedor_id,
                'tipo' => 'success',
                'titulo' => '¡ScaWatt vendido!',
                'mensaje' => "Tu ScaWatt {$publicacion->scawatt->scawatt_id} se vendió por $" . number_format($publicacion->precio_venta, 2) . ". Has recibido $" . number_format($publicacion->precio_venta - $publicacion->comision_plataforma, 2) . " (comisión del 5%: $" . number_format($publicacion->comision_plataforma, 2) . ")",
                'prioridad' => 'alta',
                'icono' => 'monetization_on',
                'datos_adicionales' => [
                    'tipo' => 'marketplace',
                    'marketplace_id' => $publicacion->id,
                    'monto_recibido' => $publicacion->precio_venta - $publicacion->comision_plataforma,
                    'scawatt_serial' => $publicacion->scawatt->scawatt_id
                ]
            ]);

            // Notificar al comprador
            Notificacion::create([
                'usuario_id' => $request->comprador_id,
                'tipo' => 'success',
                'titulo' => '¡Compra exitosa!',
                'mensaje' => "Has adquirido el ScaWatt {$publicacion->scawatt->scawatt_id} por $" . number_format($publicacion->precio_venta, 2) . ". Ya está en tu cuenta.",
                'prioridad' => 'alta',
                'icono' => 'shopping_cart',
                'datos_adicionales' => [
                    'tipo' => 'marketplace',
                    'marketplace_id' => $publicacion->id,
                    'scawatt_serial' => $publicacion->scawatt->scawatt_id
                ]
            ]);

            $payloadPython = [
                'scawatt_serial' => $scawattSerial,
                'id_laravel' => $comprador->id,
                'id_wordpress' => $comprador->id_wordpress,
                'precio' => (float) $publicacion->precio_venta,
                'marketplace_id' => $publicacion->id,
                'vendedor_id_laravel' => $vendedor?->id,
                'vendedor_id_wordpress' => $vendedor?->id_wordpress,
                'motivo' => "Venta marketplace Laravel #{$publicacion->id}"
            ];

            $responsePython = Http::timeout(120)
                ->retry(2, 2000)
                ->post("{$this->pythonApiUrl}/api/scawatts/transferir", $payloadPython);

            if (!$responsePython->successful()) {
                throw new \Exception('Error sincronizando con servicio Python: ' . $responsePython->body());
            }

            $dataPython = $responsePython->json();

            if (!($dataPython['success'] ?? false)) {
                throw new \Exception('Servicio Python retornó error: ' . ($dataPython['error'] ?? json_encode($dataPython)));
            }

            DB::commit();

            $publicacionActualizada = $publicacion->fresh(['scawatt', 'vendedor', 'comprador']);
            $this->enviarCorreoCompra($publicacionActualizada);

            return response()->json([
                'success' => true,
                'message' => 'ScaWatt comprado exitosamente',
                'publicacion' => $publicacionActualizada
            ]);
        } catch (\Illuminate\Http\Client\ConnectionException $e) {
            DB::rollBack();
            Log::error('Error al comprar ScaWatt en marketplace - conexión Python', [
                'marketplace_id' => $id,
                'payload' => $request->all(),
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'No fue posible sincronizar con el servicio de portafolio. Intenta nuevamente en unos segundos.',
                'error' => $e->getMessage()
            ], 504);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error al comprar ScaWatt en marketplace', [
                'marketplace_id' => $id,
                'payload' => $request->all(),
                'error' => $e->getMessage(),
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Error al comprar ScaWatt',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    private function enviarCorreoCompra(Marketplace $publicacion): void
    {
        try {
            $comprador = $publicacion->comprador;
            $vendedor = $publicacion->vendedor;
            $scawatt = $publicacion->scawatt;

            if (!$comprador || empty($comprador->email) || !$scawatt) {
                return;
            }

            $folderId = $comprador->id_wordpress ?? 'laravel-' . $comprador->id;
            $docsPath = rtrim($this->usuariosAppPath, '/') . '/' . $folderId . '/documentos-scawatt';

            $adjuntos = [];
            $certificado = $docsPath . '/certificado_' . $scawatt->scawatt_id . '.pdf';
            $contrato = $docsPath . '/contrato_' . $scawatt->scawatt_id . '.pdf';

            if (is_readable($certificado)) {
                $adjuntos[] = ['path' => $certificado, 'name' => 'Certificado_' . $scawatt->scawatt_id . '.pdf'];
            }

            if (is_readable($contrato)) {
                $adjuntos[] = ['path' => $contrato, 'name' => 'Contrato_' . $scawatt->scawatt_id . '.pdf'];
            }

            $fechaVenta = $publicacion->fecha_venta ? $publicacion->fecha_venta->copy()->timezone(config('app.timezone', 'UTC')) : null;
            $appUrl = rtrim(config('app.url', 'https://escalar.app'), '/');

            $datosCorreo = [
                'comprador' => $comprador->nombre ?? $comprador->email,
                'vendedor' => $vendedor?->nombre,
                'scawatt_serial' => $scawatt->scawatt_id,
                'marketplace_id' => $publicacion->id,
                'precio' => number_format((float) $publicacion->precio_venta, 0, ',', '.'),
                'comision' => number_format((float) $publicacion->comision_plataforma, 0, ',', '.'),
                'fecha_venta' => $fechaVenta ? $fechaVenta->format('d/m/Y H:i') : null,
                'logo_url' => 'https://escalar.app/brand/Logo%20Scalar%202025.png',
                'token_icon' => 'https://escalar.app/brand/Token-ppal-scawatt.png',
                'dashboard_url' => $appUrl . '/dashboard'
            ];

            $notificacionesEmail = $this->notificacionesEmail;

            Log::info('Enviando correo de compra marketplace', [
                'publicacion_id' => $publicacion->id,
                'destinatario' => $comprador->email,
                'adjuntos' => array_map(fn($adj) => $adj['name'], $adjuntos)
            ]);

            // Correo para el usuario comprador (sin comisión)
            Mail::send('emails.marketplace-compra', $datosCorreo, function ($message) use ($comprador, $adjuntos) {
                $message->to($comprador->email, $comprador->nombre ?? $comprador->email)
                    ->subject('¡Tu nuevo ScaWatt ya es tuyo!');

                foreach ($adjuntos as $adjunto) {
                    $message->attach($adjunto['path'], ['as' => $adjunto['name'], 'mime' => 'application/pdf']);
                }
            });

            // Correo para admin (con comisión visible)
            if ($notificacionesEmail) {
                Mail::send('emails.marketplace-compra-admin', $datosCorreo, function ($message) use ($notificacionesEmail, $adjuntos) {
                    $message->to($notificacionesEmail)
                        ->subject('[ADMIN] Nueva venta Marketplace - Comisión: $' . $datosCorreo['comision']);

                    foreach ($adjuntos as $adjunto) {
                        $message->attach($adjunto['path'], ['as' => $adjunto['name'], 'mime' => 'application/pdf']);
                    }
                });
            }

            if (method_exists(Mail::getFacadeRoot(), 'failures')) {
                $fallidos = Mail::failures();
                if (!empty($fallidos)) {
                    Log::warning('Correo de compra marketplace con destinatarios fallidos', [
                        'publicacion_id' => $publicacion->id,
                        'fallidos' => $fallidos
                    ]);
                }
            }
        } catch (\Throwable $mailException) {
            Log::error('Error enviando correo de compra de marketplace', [
                'publicacion_id' => $publicacion->id ?? null,
                'error' => $mailException->getMessage()
            ]);
        }
    }

    /**
     * Retirar publicación del marketplace
     * PUT /api/marketplace/retirar/{id}
     */
    public function retirar(Request $request, $id)
    {
        try {
            $publicacion = Marketplace::with('scawatt')->findOrFail($id);

            // Verificar que el usuario es el vendedor
            if ($publicacion->vendedor_id != $request->usuario_id) {
                return response()->json([
                    'success' => false,
                    'message' => 'No tienes permisos para retirar esta publicación'
                ], 403);
            }

            // Verificar que no haya sido vendido
            if ($publicacion->estado == 'vendido') {
                return response()->json([
                    'success' => false,
                    'message' => 'No puedes retirar una publicación ya vendida'
                ], 400);
            }

            DB::beginTransaction();

            $publicacion->retirar();

            // Notificar al vendedor
            Notificacion::create([
                'usuario_id' => $request->usuario_id,
                'tipo' => 'info',
                'titulo' => 'Publicación retirada',
                'mensaje' => "Tu ScaWatt {$publicacion->scawatt->scawatt_id} ha sido retirado del marketplace.",
                'prioridad' => 'baja',
                'icono' => 'remove_circle'
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Publicación retirada exitosamente',
                'publicacion' => $publicacion
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error al retirar publicación',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Estadísticas del marketplace (admin)
     * GET /api/admin/marketplace/estadisticas
     */
    public function estadisticas()
    {
        $stats = [
            'publicaciones_activas' => Marketplace::where('estado', 'en_venta')->whereNull('comprador_id')->count(),
            'ventas_totales' => Marketplace::where('estado', 'vendido')->count(),
            'ventas_hoy' => Marketplace::where('estado', 'vendido')->whereDate('fecha_venta', today())->count(),
            'comisiones_totales' => Marketplace::where('estado', 'vendido')->sum('comision_plataforma'),
            'monto_total_ventas' => Marketplace::where('estado', 'vendido')->sum('precio_venta'),
            'precio_promedio' => Marketplace::where('estado', 'en_venta')->avg('precio_venta')
        ];

        return response()->json($stats);
    }

    /**
     * Obtener seriales de ScaWatts disponibles para publicar
     * GET /api/marketplace/debug/seriales-disponibles/{usuario_id}
     * 
     * Retorna los ScaWatts del usuario que NO están publicados en el marketplace
     */
    public function serialesDisponibles($usuarioId)
    {
        try {
            // Obtener todos los ScaWatts del usuario
            $scawatts = Scawatt::where('usuario_id', $usuarioId)
                ->where('estado', 'activo')
                ->get();

            // Obtener IDs de ScaWatts que ya están publicados
            $scawattsPublicados = Marketplace::where('vendedor_id', $usuarioId)
                ->where('estado', 'en_venta')
                ->whereNull('comprador_id')
                ->pluck('scawatt_id')
                ->toArray();

            // Filtrar solo los que NO están publicados
            $disponibles = $scawatts->filter(function($scawatt) use ($scawattsPublicados) {
                return !in_array($scawatt->id, $scawattsPublicados);
            })->map(function($scawatt) {
                return [
                    'id' => $scawatt->id,
                    'scawatt_id' => $scawatt->scawatt_id,
                    'valor_actual' => $scawatt->valor_actual,
                    'fecha_compra' => $scawatt->fecha_compra,
                    'estado' => $scawatt->estado
                ];
            })->values();

            return response()->json([
                'success' => true,
                'seriales' => $disponibles,
                'total' => $disponibles->count()
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al obtener ScaWatts disponibles',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
