<?php

namespace App\Http\Controllers;

use App\Models\Mensaje;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class MensajeController extends Controller
{
    /**
     * Listar mensajes del usuario (bandeja de entrada)
     * GET /api/mensajes/{usuarioId}
     */
    public function index($usuarioId)
    {
        $mensajes = Mensaje::with(['usuario', 'creador'])
            ->paraUsuario($usuarioId)
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return response()->json($mensajes);
    }

    /**
     * Listar mensajes enviados (creados por el usuario)
     * GET /api/mensajes/{usuarioId}/enviados
     */
    public function enviados($usuarioId)
    {
        $mensajes = Mensaje::with(['usuario', 'creador'])
            ->where('creado_por', $usuarioId)
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return response()->json($mensajes);
    }

    /**
     * Ver mensaje específico
     * GET /api/mensajes/show/{id}
     */
    public function show($id)
    {
        try {
            $mensaje = Mensaje::with(['usuario', 'creador'])->findOrFail($id);
            return response()->json($mensaje);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Mensaje no encontrado',
                'error' => $e->getMessage()
            ], 404);
        }
    }

    /**
     * Crear nuevo mensaje/comunicado
     * POST /api/mensajes
     */
    public function store(Request $request)
    {
        $request->validate([
            'titulo' => 'required|string|max:200',
            'contenido' => 'required|string',
            'tipo' => 'required|in:info,alerta,promocion,sistema',
            'destinatario' => 'required|in:todos,individual,grupo',
            'usuario_id' => 'required_if:destinatario,individual|exists:usuarios,id',
            'creado_por' => 'required|exists:usuarios,id',
            'fecha_programada' => 'nullable|date'
        ]);

        try {
            DB::beginTransaction();

            $mensaje = Mensaje::create([
                'titulo' => $request->titulo,
                'contenido' => $request->contenido,
                'tipo' => $request->tipo,
                'destinatario' => $request->destinatario,
                'usuario_id' => $request->usuario_id,
                'fecha_programada' => $request->fecha_programada,
                'enviado' => $request->fecha_programada ? false : true,
                'fecha_enviado' => $request->fecha_programada ? null : now(),
                'creado_por' => $request->creado_por
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Mensaje creado exitosamente',
                'mensaje' => $mensaje->load('usuario', 'creador')
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error al crear mensaje',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Actualizar mensaje
     * PUT /api/mensajes/{id}
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'titulo' => 'string|max:200',
            'contenido' => 'string',
            'tipo' => 'in:info,alerta,promocion,sistema',
            'destinatario' => 'in:todos,individual,grupo',
            'usuario_id' => 'exists:usuarios,id',
            'fecha_programada' => 'nullable|date'
        ]);

        try {
            $mensaje = Mensaje::findOrFail($id);
            $mensaje->update($request->all());

            return response()->json([
                'success' => true,
                'message' => 'Mensaje actualizado',
                'mensaje' => $mensaje
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al actualizar mensaje',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Marcar mensaje como enviado
     * PUT /api/mensajes/{id}/marcar-enviado
     */
    public function marcarEnviado($id)
    {
        try {
            $mensaje = Mensaje::findOrFail($id);
            $mensaje->marcarEnviado();

            return response()->json([
                'success' => true,
                'message' => 'Mensaje marcado como enviado'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al marcar mensaje',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Eliminar mensaje
     * DELETE /api/mensajes/{id}
     */
    public function destroy($id)
    {
        try {
            $mensaje = Mensaje::findOrFail($id);
            $mensaje->delete();

            return response()->json([
                'success' => true,
                'message' => 'Mensaje eliminado'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al eliminar mensaje',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Contar mensajes pendientes de envío
     * GET /api/mensajes/pendientes/count
     */
    public function contarPendientes()
    {
        $count = Mensaje::pendientes()->count();
        return response()->json(['count' => $count]);
    }

    /**
     * Contar mensajes por tipo para un usuario
     * GET /api/mensajes/{usuarioId}/stats
     */
    public function estadisticas($usuarioId)
    {
        $stats = [
            'total' => Mensaje::paraUsuario($usuarioId)->count(),
            'info' => Mensaje::paraUsuario($usuarioId)->tipo('info')->count(),
            'alerta' => Mensaje::paraUsuario($usuarioId)->tipo('alerta')->count(),
            'promocion' => Mensaje::paraUsuario($usuarioId)->tipo('promocion')->count(),
            'sistema' => Mensaje::paraUsuario($usuarioId)->tipo('sistema')->count()
        ];

        return response()->json($stats);
    }
}
