<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use Illuminate\Foundation\Application;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use App\Models\Scawatt;
use App\Models\Granja;
use App\Models\Usuario;
use App\Models\Marketplace;
use App\Models\SolicitudContactoScawatt;
use App\Models\Deposito;
use App\Models\Retiro;

class MonitorController extends Controller
{
    /**
     * Mostrar página de monitor
     */
    public function index()
    {
        return view('monitor.index');
    }

    /**
     * Obtener el estado de procesos y detalles del servidor.
     */
    public function obtenerProcesos(): JsonResponse
    {
        try {
            $tokensPath = '/home/scalarapp/scalar-data/scawatt-server/produccion-scawatts/data_scawatts/raiz-scawatt/scawatts_tokens.json';
            $publicJsonPath = '/home/scalarapp/public_html/develop/code-test/ScaWatt.json';

            $procesos = [
                'monitor_scawatt' => $this->checkProcess('monitor_scawatt'),
                'sincronizacion' => $this->checkFileIsFresh($tokensPath, 120),
                'cron_tasks' => $this->checkProcess('cron'),
                'mysql_db' => $this->checkDatabaseConnection(),
            ];

            $detalles = [
                'servidor' => gethostname() ?: php_uname('n'),
                'usuario' => get_current_user(),
                'timestamp' => now()->toDateTimeString(),
                'uptime' => $this->getSystemUptime(),
                'php_version' => PHP_VERSION,
                'laravel_version' => Application::VERSION,
            ];

            $archivos = [
                'scawatts_tokens' => $this->fileStatus($tokensPath, 120),
                'scawatt_public' => $this->fileStatus($publicJsonPath, 120),
            ];

            $metricas = $this->obtenerMetricasSistema();
            $dashboard = $this->obtenerResumenDashboard($metricas, $archivos['scawatts_tokens'] ?? null);
            $solicitudes = $this->obtenerResumenSolicitudes();

            return response()->json([
                'success' => true,
                'procesos' => $procesos,
                'detalles' => $detalles,
                'archivos' => $archivos,
                'metricas' => $metricas,
                'dashboard' => $dashboard,
                'solicitudes' => $solicitudes,
            ]);
        } catch (\Throwable $exception) {
            Log::error('MonitorController::obtenerProcesos error', [
                'message' => $exception->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'No se pudo obtener el estado del monitor.',
            ], 500);
        }
    }

    private function obtenerMetricasSistema(): array
    {
        $totalesScawatt = Scawatt::selectRaw('
                COUNT(*) as total,
                COALESCE(SUM(valor_actual), 0) as valor_actual_total,
                COALESCE(SUM(valor_inicial), 0) as valor_inicial_total,
                COALESCE(SUM(valor_actual - valor_inicial), 0) as valorizacion_total
            ')
            ->first();

        $valorActualTotal = (float) ($totalesScawatt->valor_actual_total ?? 0);
        $valorInicialTotal = (float) ($totalesScawatt->valor_inicial_total ?? 0);
        $valorizacionTotal = (float) ($totalesScawatt->valorizacion_total ?? ($valorActualTotal - $valorInicialTotal));

        return [
            'scawatts_total' => (int) ($totalesScawatt->total ?? 0),
            'usuarios_total' => (int) Usuario::count(),
            'usuarios_con_scawatts' => (int) Scawatt::distinct('usuario_id')->count('usuario_id'),
            'granjas_total' => (int) Granja::count(),
            'capacidad_total_mw' => (float) Granja::sum('capacidad_mw'),
            'valor_actual_total' => $valorActualTotal,
            'valor_inicial_total' => $valorInicialTotal,
            'valorizacion_total' => $valorizacionTotal,
            'valoracion_porcentaje' => $valorInicialTotal > 0
                ? round(($valorizacionTotal / $valorInicialTotal) * 100, 2)
                : 0,
            'marketplace_en_venta' => (int) Marketplace::where('estado', 'en_venta')->count(),
            'marketplace_total' => (int) Marketplace::count(),
            'depositos_pendientes' => (int) Deposito::whereIn('estado', ['pendiente', 'en_proceso'])->count(),
            'retiros_pendientes' => (int) Retiro::whereIn('estado', ['pendiente', 'en_proceso'])->count(),
        ];
    }

    private function obtenerResumenDashboard(array $metricas, ?array $archivoTokens): array
    {
        return [
            'total_scawatts' => $metricas['scawatts_total'] ?? 0,
            'total_granjas' => $metricas['granjas_total'] ?? 0,
            'total_usuarios' => $metricas['usuarios_total'] ?? 0,
            'valor_actual_total' => $metricas['valor_actual_total'] ?? 0,
            'valorizacion_total' => $metricas['valorizacion_total'] ?? 0,
            'porcentaje_valorizacion' => $metricas['valoracion_porcentaje'] ?? 0,
            'capacidad_total_mw' => $metricas['capacidad_total_mw'] ?? 0,
            'marketplace_publicaciones_en_venta' => $metricas['marketplace_en_venta'] ?? 0,
            'depositos_pendientes' => $metricas['depositos_pendientes'] ?? 0,
            'retiros_pendientes' => $metricas['retiros_pendientes'] ?? 0,
            'ultima_sync' => $archivoTokens['last_modified'] ?? null,
        ];
    }

    private function obtenerResumenSolicitudes(): array
    {
        $pendientes = SolicitudContactoScawatt::pendientes()->count();
        $recientes = SolicitudContactoScawatt::orderByDesc('created_at')
            ->limit(20)
            ->get()
            ->map(function (SolicitudContactoScawatt $solicitud) {
                return [
                    'id' => $solicitud->id,
                    'nombre_completo' => $solicitud->nombre_completo,
                    'cedula' => $solicitud->cedula,
                    'email' => $solicitud->email,
                    'telefono' => $solicitud->telefono,
                    'cantidad_scawatts' => $solicitud->cantidad_scawatts,
                    'presupuesto_aproximado' => $solicitud->presupuesto_aproximado,
                    'mensaje' => $solicitud->mensaje,
                    'estado' => $solicitud->estado,
                    'created_at' => optional($solicitud->created_at)->toDateTimeString(),
                ];
            });

        return [
            'pendientes' => (int) $pendientes,
            'total' => (int) SolicitudContactoScawatt::count(),
            'recientes' => $recientes,
            'acciones_habilitadas' => false,
        ];
    }

    /**
     * Determina si un proceso está activo utilizando pgrep.
     */
    private function checkProcess(string $pattern): string
    {
        $output = [];
        $status = 1;
        @exec('pgrep -f ' . escapeshellarg($pattern), $output, $status);

        return $status === 0 && !empty($output) ? 'activo' : 'inactivo';
    }

    /**
     * Considera activo el proceso de sincronización si el archivo asociado es reciente.
     */
    private function checkFileIsFresh(string $path, int $maxMinutes): string
    {
        $status = $this->fileStatus($path, $maxMinutes);

        return $status['exists'] && $status['status'] === 'reciente' ? 'activo' : 'inactivo';
    }

    /**
     * Verifica el estado de un archivo relevante para el monitor.
     */
    private function fileStatus(string $path, int $maxMinutes): array
    {
        if (!file_exists($path)) {
            return [
                'exists' => false,
                'status' => 'faltante',
                'last_modified' => null,
                'age_minutes' => null,
                'size_bytes' => null,
                'path' => $path,
            ];
        }

        $lastModified = Carbon::createFromTimestamp(filemtime($path));
        $ageMinutes = $lastModified->diffInMinutes(now());

        return [
            'exists' => true,
            'status' => $ageMinutes <= $maxMinutes ? 'reciente' : 'desactualizado',
            'last_modified' => $lastModified->toDateTimeString(),
            'age_minutes' => $ageMinutes,
            'size_bytes' => filesize($path),
            'path' => $path,
        ];
    }

    /**
     * Verifica la conexión a la base de datos.
     */
    private function checkDatabaseConnection(): string
    {
        try {
            DB::connection()->getPdo();
            return 'activo';
        } catch (\Throwable $exception) {
            Log::warning('MonitorController::checkDatabaseConnection falló', [
                'message' => $exception->getMessage(),
            ]);
            return 'inactivo';
        }
    }

    /**
     * Obtiene una cadena amigable con el uptime del sistema.
     */
    private function getSystemUptime(): ?string
    {
        $uptimeFile = '/proc/uptime';

        if (!is_readable($uptimeFile)) {
            return null;
        }

        $content = trim((string) file_get_contents($uptimeFile));
        if ($content === '') {
            return null;
        }

        $seconds = (int) floor((float) explode(' ', $content)[0] ?? 0);

        if ($seconds <= 0) {
            return null;
        }

        $days = intdiv($seconds, 86400);
        $hours = intdiv($seconds % 86400, 3600);
        $minutes = intdiv($seconds % 3600, 60);

        $parts = [];
        if ($days > 0) {
            $parts[] = $days . 'd';
        }
        if ($hours > 0) {
            $parts[] = $hours . 'h';
        }
        if ($minutes > 0) {
            $parts[] = $minutes . 'm';
        }

        return empty($parts) ? $seconds . 's' : implode(' ', $parts);
    }

    /**
     * Obtener estadísticas de verificaciones de ScaWatt QR
     * NUEVO - No afecta funcionalidad existente
     */
    public function obtenerEstadisticasVerificaciones(): JsonResponse
    {
        try {
            // Leer cache de Laravel para verificaciones (últimas 24h)
            $verificacionesRecientes = [];
            $totalVerificaciones = 0;
            $verificacionesExitosas = 0;
            $verificacionesFallidas = 0;

            // Intentar obtener estadísticas del cache
            // Las verificaciones guardan cache con prefijo "verificacion_scawatt_"
            $cacheKeys = $this->obtenerClavesVerificacionCache();
            
            foreach ($cacheKeys as $key) {
                $data = \Cache::get($key);
                if ($data && is_array($data)) {
                    $totalVerificaciones++;
                    if (isset($data['verificado']) && $data['verificado'] === true) {
                        $verificacionesExitosas++;
                    } else {
                        $verificacionesFallidas++;
                    }
                    
                    // Agregar a log reciente (últimos 10)
                    if (count($verificacionesRecientes) < 10) {
                        $verificacionesRecientes[] = [
                            'serial' => $data['serial'] ?? 'N/A',
                            'timestamp' => $data['timestamp'] ?? now()->toDateTimeString(),
                            'exitosa' => $data['verificado'] ?? false,
                        ];
                    }
                }
            }

            return response()->json([
                'success' => true,
                'estadisticas' => [
                    'total_24h' => $totalVerificaciones,
                    'exitosas' => $verificacionesExitosas,
                    'fallidas' => $verificacionesFallidas,
                    'tasa_exito' => $totalVerificaciones > 0 
                        ? round(($verificacionesExitosas / $totalVerificaciones) * 100, 1)
                        : 0,
                ],
                'recientes' => $verificacionesRecientes,
            ]);
        } catch (\Throwable $exception) {
            Log::error('MonitorController::obtenerEstadisticasVerificaciones error', [
                'message' => $exception->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Error al obtener estadísticas de verificaciones',
            ], 500);
        }
    }

    /**
     * Helper para obtener claves de cache de verificaciones
     * NUEVO - Método auxiliar
     */
    private function obtenerClavesVerificacionCache(): array
    {
        // Por ahora retornamos array vacío
        // En producción se podría implementar con Redis SCAN
        // o guardar las claves en una tabla temporal
        return [];
    }

    /**
     * Obtener últimas líneas del log de Laravel
     * NUEVO - No afecta funcionalidad existente
     */
    public function obtenerLogErrores(): JsonResponse
    {
        try {
            $logPath = storage_path('logs/laravel.log');
            
            if (!file_exists($logPath)) {
                return response()->json([
                    'success' => true,
                    'errores' => [],
                    'mensaje' => 'No hay archivo de log disponible',
                ]);
            }

            // Leer últimas 50 líneas
            $lineas = $this->leerUltimasLineasArchivo($logPath, 50);
            
            // Parsear errores
            $errores = $this->parsearLogLaravel($lineas);

            return response()->json([
                'success' => true,
                'errores' => array_slice($errores, 0, 20), // Solo últimos 20 errores
                'total' => count($errores),
            ]);
        } catch (\Throwable $exception) {
            Log::error('MonitorController::obtenerLogErrores error', [
                'message' => $exception->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Error al leer log de errores',
            ], 500);
        }
    }

    /**
     * Leer últimas N líneas de un archivo
     * NUEVO - Método auxiliar
     */
    private function leerUltimasLineasArchivo(string $path, int $numLineas): array
    {
        $file = new \SplFileObject($path, 'r');
        $file->seek(PHP_INT_MAX);
        $lastLine = $file->key();
        $lines = [];
        
        $linesToRead = min($numLineas, $lastLine);
        $startLine = max(0, $lastLine - $linesToRead);
        
        $file->seek($startLine);
        
        while (!$file->eof()) {
            $line = $file->current();
            if (trim($line) !== '') {
                $lines[] = $line;
            }
            $file->next();
        }
        
        return $lines;
    }

    /**
     * Parsear log de Laravel
     * NUEVO - Método auxiliar
     */
    private function parsearLogLaravel(array $lineas): array
    {
        $errores = [];
        $errorActual = null;
        
        foreach ($lineas as $linea) {
            // Detectar inicio de nuevo log: [YYYY-MM-DD HH:MM:SS]
            if (preg_match('/^\[(\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2})\].*?\.(\w+):/', $linea, $matches)) {
                // Guardar error anterior si existe
                if ($errorActual) {
                    $errores[] = $errorActual;
                }
                
                $errorActual = [
                    'timestamp' => $matches[1],
                    'nivel' => strtoupper($matches[2]),
                    'mensaje' => trim(substr($linea, strpos($linea, ':') + 1)),
                    'linea_completa' => $linea,
                ];
            } elseif ($errorActual) {
                // Continuar mensaje de error multi-línea
                $errorActual['mensaje'] .= ' ' . trim($linea);
            }
        }
        
        // Agregar último error
        if ($errorActual) {
            $errores[] = $errorActual;
        }
        
        return array_reverse($errores); // Más recientes primero
    }

    /**
     * Health check de servicios críticos
     * NUEVO - No afecta funcionalidad existente
     */
    public function healthCheckServicios(): JsonResponse
    {
        try {
            $servicios = [];

            // 1. API Python Producción (puerto 5001)
            $servicios['python_api'] = $this->checkService('http://localhost:5001/api/health', 5);

            // 2. Generador de Certificados (puerto 5002)
            $servicios['generador_pdf'] = $this->checkService('http://localhost:5002/health', 5);

            // 3. Base de datos
            $servicios['database'] = [
                'status' => $this->checkDatabaseConnection() === 'activo' ? 'ok' : 'error',
                'time_ms' => 0,
            ];

            return response()->json([
                'success' => true,
                'servicios' => $servicios,
                'timestamp' => now()->toDateTimeString(),
            ]);
        } catch (\Throwable $exception) {
            Log::error('MonitorController::healthCheckServicios error', [
                'message' => $exception->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Error al verificar servicios',
            ], 500);
        }
    }

    /**
     * Verificar disponibilidad de un servicio HTTP
     * NUEVO - Método auxiliar
     */
    private function checkService(string $url, int $timeout = 5): array
    {
        $start = microtime(true);
        
        try {
            $ch = curl_init($url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, $timeout);
            curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, $timeout);
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, false);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $error = curl_error($ch);
            curl_close($ch);
            
            $timeMs = round((microtime(true) - $start) * 1000, 2);
            
            if ($error) {
                return [
                    'status' => 'error',
                    'time_ms' => $timeMs,
                    'error' => $error,
                ];
            }
            
            if ($httpCode >= 200 && $httpCode < 300) {
                return [
                    'status' => 'ok',
                    'time_ms' => $timeMs,
                    'http_code' => $httpCode,
                ];
            }
            
            return [
                'status' => 'warning',
                'time_ms' => $timeMs,
                'http_code' => $httpCode,
            ];
        } catch (\Throwable $e) {
            $timeMs = round((microtime(true) - $start) * 1000, 2);
            
            return [
                'status' => 'error',
                'time_ms' => $timeMs,
                'error' => $e->getMessage(),
            ];
        }
    }
}
