<?php

namespace App\Http\Controllers;

use App\Models\Notificacion;
use App\Models\Usuario;
use App\Services\FirebaseService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class NotificacionController extends Controller
{
    protected $firebaseService;

    public function __construct(FirebaseService $firebaseService)
    {
        $this->firebaseService = $firebaseService;
    }
    /**
     * Listar notificaciones de un usuario
     * GET /api/notificaciones/{usuarioId}
     */
    public function index($usuarioId)
    {
        $notificaciones = Notificacion::where('usuario_id', $usuarioId)
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return response()->json($notificaciones);
    }

    /**
     * Contar notificaciones no leídas
     * GET /api/notificaciones/{usuarioId}/no-leidas/count
     */
    public function contarNoLeidas($usuarioId)
    {
        $count = Notificacion::where('usuario_id', $usuarioId)
            ->noLeidas()
            ->count();

        return response()->json(['count' => $count]);
    }

    /**
     * Crear notificación
     * POST /api/admin/notificaciones
     */
    public function store(Request $request)
    {
        $request->validate([
            'usuario_id' => 'required|exists:usuarios,id',
            'tipo' => 'required|string|in:info,success,warning,error',
            'titulo' => 'required|string|max:255',
            'mensaje' => 'required|string',
            'prioridad' => 'nullable|string|in:baja,media,alta',
            'icono' => 'nullable|string|max:50',
            'url_accion' => 'nullable|url',
            'datos_adicionales' => 'nullable|array'
        ]);

        try {
            $notificacion = Notificacion::create($request->all());

            return response()->json([
                'success' => true,
                'message' => 'Notificación creada exitosamente',
                'notificacion' => $notificacion
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al crear notificación',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Marcar notificación como leída
     * PUT /api/notificaciones/{id}/marcar-leida
     */
    public function marcarLeida($id)
    {
        try {
            $notificacion = Notificacion::findOrFail($id);
            $notificacion->marcarLeida();

            return response()->json([
                'success' => true,
                'message' => 'Notificación marcada como leída'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al marcar notificación',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Marcar todas las notificaciones como leídas
     * PUT /api/notificaciones/{usuarioId}/marcar-todas-leidas
     */
    public function marcarTodasLeidas($usuarioId)
    {
        try {
            Notificacion::where('usuario_id', $usuarioId)
                ->noLeidas()
                ->update([
                    'leido' => true,
                    'fecha_leido' => now()
                ]);

            return response()->json([
                'success' => true,
                'message' => 'Todas las notificaciones marcadas como leídas'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al marcar notificaciones',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Eliminar notificación
     * DELETE /api/notificaciones/{id}
     */
    public function destroy($id)
    {
        try {
            $notificacion = Notificacion::findOrFail($id);
            $notificacion->delete();

            return response()->json([
                'success' => true,
                'message' => 'Notificación eliminada'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al eliminar notificación',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Notificación masiva a múltiples usuarios
     * POST /api/admin/notificaciones/masiva
     */
    public function notificacionMasiva(Request $request)
    {
        $request->validate([
            'usuarios_ids' => 'required|array',
            'usuarios_ids.*' => 'exists:usuarios,id',
            'tipo' => 'required|string|in:info,success,warning,error',
            'titulo' => 'required|string|max:255',
            'mensaje' => 'required|string',
            'prioridad' => 'nullable|string|in:baja,media,alta'
        ]);

        try {
            DB::beginTransaction();

            $notificaciones = [];
            foreach ($request->usuarios_ids as $usuarioId) {
                $notificaciones[] = [
                    'usuario_id' => $usuarioId,
                    'tipo' => $request->tipo,
                    'titulo' => $request->titulo,
                    'mensaje' => $request->mensaje,
                    'prioridad' => $request->prioridad ?? 'media',
                    'leido' => false,
                    'created_at' => now(),
                    'updated_at' => now()
                ];
            }

            Notificacion::insert($notificaciones);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Notificaciones enviadas',
                'total' => count($notificaciones)
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error al enviar notificaciones masivas',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
