<?php

namespace App\Http\Controllers;

use App\Mail\DocumentosScawattCompartidosMail;
use App\Mail\SolicitudDocumentosScawattMail;
use App\Models\Usuario;
use App\Models\Notificacion;
use App\Models\Scawatt;
use App\Models\Marketplace;
use App\Models\SolicitudContactoScawatt;
use App\Models\SolicitudDocumentosScawatt;
use App\Services\ScawattDocumentManager;
use App\Services\SolicitudScawattWorkflowService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Cache;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Carbon\Carbon;

class ScawattController extends Controller
{
    private string $pythonApiUrl;

    public function __construct()
    {
        $this->pythonApiUrl = config('services.scawatt_python.url', 'http://localhost:5004');
    }
    
    /**
     * Crear un nuevo ScaWatt para un usuario
     * 
     * POST /api/admin/scawatts/crear
     * {
     *   "usuario_id": 2,
     *   "fecha_inicio": "2025-09-01" (opcional)
     * }
     */
    public function crear(Request $request)
    {
        $request->validate([
            'usuario_id' => 'required|exists:usuarios,id',
            'valor_inicial' => 'nullable|numeric|min:0',
            'fecha_inicio' => 'nullable|date',
            'años_contrato' => 'nullable|integer|in:10,15,20,25',
            'kwh_asignados' => 'nullable|numeric|min:0',
            'tipo' => 'nullable|string|in:estandar,premium'
        ]);

        $usuario = Usuario::findOrFail($request->usuario_id);

        try {
            DB::beginTransaction();

            // 1. Llamar a Python para crear el ScaWatt
            // NUEVO: Enviar id_laravel como maestro, id_wordpress solo por compatibilidad
            $pythonPayload = [
                'id_laravel' => $usuario->id,  // ID Laravel (sistema maestro)
                'id_wordpress' => $usuario->id_wordpress  // WordPress (temporal)
            ];
            
            // Agregar parámetros opcionales si se proporcionan
            if ($request->has('valor_inicial')) {
                $pythonPayload['valor_inicial'] = $request->valor_inicial;
            }
            if ($request->has('fecha_inicio')) {
                $pythonPayload['fecha_inicio'] = $request->fecha_inicio;
            }
            if ($request->has('años_contrato')) {
                $pythonPayload['años_contrato'] = $request->años_contrato;
            }
            if ($request->has('kwh_asignados')) {
                $pythonPayload['kwh_asignados'] = $request->kwh_asignados;
            }
            if ($request->has('tipo')) {
                $pythonPayload['tipo'] = $request->tipo;
            }
            
            $response = Http::timeout(30)->post("{$this->pythonApiUrl}/api/scawatts/crear", $pythonPayload);

            if (!$response->successful()) {
                throw new \Exception('Error al crear ScaWatt en Python: ' . $response->body());
            }

            $pythonData = $response->json();

            if (!$pythonData['success']) {
                throw new \Exception('Python retornó error: ' . ($pythonData['error'] ?? 'Desconocido'));
            }

            // NUEVO: Usar scawatt_serial como ID principal
            $scawattSerial = $pythonData['scawatt_serial'];
            $scawattId = $pythonData['scawatt_id'];  // Mantener por compatibilidad
            $scawattData = $pythonData['data'];

            // 2. Registrar en MySQL (usar scawatt_serial como ID principal)
            $scawatt = Scawatt::create([
                'scawatt_id' => $scawattSerial,
                'usuario_id' => $usuario->id,
                'kwh_asignados' => $scawattData['kwh_asignados'] ?? 45000,
                'valor_actual' => ($scawattData['valor'] ?? 0) + ($scawattData['valorizacion'] ?? 0),
                'valor_inicial' => $scawattData['valor'] ?? 0,
                'valorizaciones' => [$scawattData['valorizacion'] ?? 0],
                'fecha_inicio' => $scawattData['fecha_inicio'] ?? now()->toDateString(),
                'fecha_final' => $scawattData['fecha_final'] ?? now()->toDateString(),
                'estado' => 'activo',
                'tipo' => $scawattData['tipo'] ?? 'normal'
            ]);

            // 3. Copiar documentos generados por Python a Laravel
            if ($pythonData['documentos_generados']) {
                $this->copiarDocumentos($usuario, $scawattSerial);
            }

            // 4. Registrar en trazabilidad
            DB::table('trazabilidad')->insert([
                'scawatt_id' => $scawatt->id,
                'usuario_id' => $usuario->id,
                'evento' => 'creacion',
                'descripcion' => 'ScaWatt creado desde panel Laravel',
                'datos_antes' => null,
                'datos_despues' => json_encode([
                    'valor_inicial' => $scawattData['valor'] ?? null,
                    'valorizacion' => $scawattData['valorizacion'] ?? null,
                    'valor_actual' => ($scawattData['valor'] ?? 0) + ($scawattData['valorizacion'] ?? 0)
                ]),
                'monto' => $scawattData['valor'] ?? null,
                'fecha_evento' => now(),
                'created_at' => now(),
                'updated_at' => now()
            ]);

            // 5. Registrar propietario
            DB::table('propietarios')->insert([
                'scawatt_id' => $scawatt->id,
                'usuario_id' => $usuario->id,
                'fecha_inicio' => $scawattData['fecha_inicio'],
                'tipo_adquisicion' => 'compra_inicial',
                'precio_adquisicion' => $scawattData['valor'],
                'es_propietario_actual' => true,
                'created_at' => now(),
                'updated_at' => now()
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'mensaje' => 'ScaWatt creado exitosamente',
                'data' => [
                    'scawatt' => $scawatt,
                    'usuario' => $usuario,
                    'python_response' => $pythonData
                ]
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            
            Log::error('Error al crear ScaWatt', [
                'error' => $e->getMessage(),
                'payload' => $request->all()
            ]);

            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Obtener proyección estimada de generación y rentabilidad para los ScaWatts de un usuario
     * GET /api/scawatts/{usuarioId}/proyeccion
     */
    public function obtenerProyeccion(Request $request, $scawattId)
    {
        try {
            $scawatt = Scawatt::with(['granja', 'usuario'])->findOrFail($scawattId);

            $usuario = $request->user();
            $esPropietario = $usuario && $usuario->id === $scawatt->usuario_id;
            $esAdmin = $usuario && method_exists($usuario, 'esAdministrador') && $usuario->esAdministrador();
            $publicacionMarketplaceActiva = Marketplace::where('scawatt_id', $scawatt->id)
                ->where('estado', 'en_venta')
                ->whereNull('comprador_id')
                ->exists();

            if (!$esPropietario && !$esAdmin && !$publicacionMarketplaceActiva) {
                Log::info('Proyección consultada por usuario sin relación directa', [
                    'solicitante_id' => $usuario ? $usuario->id : null,
                    'scawatt_id' => $scawatt->id,
                ]);
            }

            $granja = $scawatt->granja;

            $valorInicial = (float) $scawatt->valor_inicial;
            $valorActual = (float) $scawatt->valor_actual;

            $defaultRate = (float) config('scawatt.proyeccion.tasa_valorizacion_anual', 3.5);
            $rate = $granja && $granja->tasa_valorizacion_anual ? (float) $granja->tasa_valorizacion_anual : $defaultRate;
            if ($rate <= 0) {
                $rate = $defaultRate;
            }

            $defaultGananciaMensual = (float) config('scawatt.proyeccion.ganancia_mensual_base', 550000);
            $gananciaMensualBase = $granja && $granja->ganancia_mensual_base ? (float) $granja->ganancia_mensual_base : $defaultGananciaMensual;
            if ($gananciaMensualBase <= 0) {
                $gananciaMensualBase = $defaultGananciaMensual;
            }

            $kwhPorScawattMes = (float) config('scawatt.proyeccion.kwh_por_scawatt_mes', 3);
            $precioKwh = (float) config('scawatt.proyeccion.precio_kwh', 400);

            $fechaInicio = $scawatt->fecha_inicio ? Carbon::parse($scawatt->fecha_inicio) : null;
            $fechaFinal = $scawatt->fecha_final ? Carbon::parse($scawatt->fecha_final) : null;

            $contratoMeses = null;
            $añosRestantes = null;
            if ($fechaInicio && $fechaFinal) {
                $contratoMeses = max($fechaInicio->floatDiffInMonths($fechaFinal, false), 0);
                $añosRestantes = round(max(now()->floatDiffInMonths($fechaFinal, false) / 12, 0), 2);
            }

            $añosProyectados = $request->query('años', $request->query('anios', $request->query('years')));
            $añosProyectados = $añosProyectados ? (int) $añosProyectados : 10;
            if ($añosProyectados <= 0) {
                $añosProyectados = 1;
            }

            if ($añosRestantes !== null && $añosRestantes > 0) {
                $añosProyectados = (int) min($añosProyectados, max((int) ceil($añosRestantes), 1));
            }

            $gananciaAnual = $gananciaMensualBase * 12;
            $energiaMensualEstimado = null;
            if ($scawatt->kwh_asignados && $contratoMeses && $contratoMeses > 0) {
                $energiaMensualEstimado = (float) $scawatt->kwh_asignados / $contratoMeses;
            } else {
                $energiaMensualEstimado = $kwhPorScawattMes;
            }

            $detalleAnual = [];
            $ingresosAcumulados = 0.0;
            $valorProyectado = $valorActual;

            for ($year = 1; $year <= $añosProyectados; $year++) {
                $ingresosAcumulados += $gananciaAnual;
                $valorInversion = $valorActual * pow(1 + ($rate / 100), $year);
                $valorEstimadoTotal = $valorInversion + $ingresosAcumulados;
                $roi = $valorInicial > 0 ? (($valorEstimadoTotal - $valorInicial) / $valorInicial) * 100 : 0;

                $detalleAnual[] = [
                    'anio' => $year,
                    'valor_inversion_estimado' => round($valorInversion, 2),
                    'ingresos_acumulados' => round($ingresosAcumulados, 2),
                    'valor_estimado_total' => round($valorEstimadoTotal, 2),
                    'roi_porcentaje' => round($roi, 2)
                ];

                $valorProyectado = $valorEstimadoTotal;
            }

            $gananciaFutura = $valorProyectado - $valorActual;
            $roiFinal = $valorInicial > 0 ? (($valorProyectado - $valorInicial) / $valorInicial) * 100 : 0;

            $energiaAnualEstimado = round($energiaMensualEstimado * 12, 2);
            $energiaTotalEstimado = round($energiaAnualEstimado * $añosProyectados, 2);

            $factorEmision = (float) config('scawatt.proyeccion.factor_emision_co2', 0.5);
            $co2AnualEstimado = round($energiaAnualEstimado * $factorEmision, 2);
            $co2TotalEstimado = round($co2AnualEstimado * $añosProyectados, 2);

            return response()->json([
                'success' => true,
                'scawatt' => [
                    'id' => $scawatt->id,
                    'serial' => $scawatt->scawatt_id,
                    'valor_inicial' => round($valorInicial, 2),
                    'valor_actual' => round($valorActual, 2),
                    'valorizacion_actual' => round($valorActual - $valorInicial, 2),
                    'estado' => $scawatt->estado,
                ],
                'granja' => $granja ? [
                    'id' => $granja->id,
                    'nombre' => $granja->nombre,
                    'departamento' => $granja->departamento,
                    'municipio' => $granja->municipio,
                    'años_restantes' => $añosRestantes,
                    'tasa_valorizacion_anual' => round($rate, 2),
                    'ganancia_mensual_base' => round($gananciaMensualBase, 2),
                ] : null,
                'proyeccion' => [
                    'años_proyectados' => $añosProyectados,
                    'valor_final_estimado' => round($valorProyectado, 2),
                    'ganancia_futura_estimada' => round($gananciaFutura, 2),
                    'roi_final_porcentaje' => round($roiFinal, 2),
                    'detalle_anual' => $detalleAnual,
                ],
                'impacto_ambiental' => [
                    'energia_anual_estimado' => $energiaAnualEstimado,
                    'energia_total_estimado' => $energiaTotalEstimado,
                    'co2_anual_estimado' => $co2AnualEstimado,
                    'co2_total_estimado' => $co2TotalEstimado,
                    'factor_emision_co2' => $factorEmision,
                    'unidad_factor' => 'kg_CO2_por_kWh',
                ],
                'supuestos' => [
                    'kwh_estimados_mes' => round($energiaMensualEstimado, 2),
                    'precio_kwh' => $precioKwh,
                    'ganancia_mensual_base' => round($gananciaMensualBase, 2),
                    'ganancia_anual_base' => round($gananciaAnual, 2),
                    'tasa_valorizacion_anual' => round($rate, 2),
                    'contrato_meses' => $contratoMeses ? round($contratoMeses, 2) : null,
                    'años_restantes' => $añosRestantes,
                    'factor_emision_co2' => $factorEmision,
                    'energia_anual_estimado' => $energiaAnualEstimado,
                ],
                'fecha_calculo' => now()->toIso8601String(),
            ]);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'ScaWatt no encontrado'
            ], 404);
        } catch (\Exception $e) {
            Log::error('Error al calcular proyección de ScaWatts', [
                'error' => $e->getMessage(),
                'scawatt_id' => $scawattId,
                'payload' => $request->all(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'No fue posible obtener la proyección',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Obtener documentos (certificado/contrato) de un ScaWatt sin requerir pertenencia
     * GET /api/public/scawatts/{identifier}/documentos
     */
    public function documentosPublicos($identifier)
    {
        try {
            $scawatt = $this->buscarScawattPorIdentificador($identifier);

            $documentos = $this->obtenerDocumentosDeScawatt($scawatt->id);

            if ($documentos->isEmpty()) {
                Log::warning('Documentos no encontrados para ScaWatt', [
                    'scawatt_id' => $scawatt->id,
                    'identificador_busqueda' => $identifier,
                ]);
            }

            return response()->json([
                'success' => true,
                'scawatt' => [
                    'id' => $scawatt->id,
                    'serial' => $scawatt->scawatt_id,
                    'estado' => $scawatt->estado,
                    'tipo' => $scawatt->tipo,
                    'propietario_id' => $scawatt->usuario ? $scawatt->usuario->id : null,
                    'propietario_nombre' => $scawatt->usuario ? $scawatt->usuario->nombre : null,
                ],
                'documentos' => $this->mapearDocumentos($documentos),
            ]);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'ScaWatt no encontrado'
            ], 404);
        } catch (\Exception $e) {
            Log::error('Error al obtener documentos públicos de ScaWatt', [
                'error' => $e->getMessage(),
                'identificador' => $identifier,
            ]);

            return response()->json([
                'success' => false,
                'message' => 'No fue posible obtener los documentos del ScaWatt',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function solicitarDocumentos(Request $request, $identifier)
    {
        $payload = Validator::make(
            $this->normalizarDatosSolicitante($request),
            [
                'nombre' => ['nullable', 'string', 'max:120'],
                'email' => ['nullable', 'string', 'email', 'max:160'],
                'telefono' => ['nullable', 'string', 'max:30'],
                'motivo' => ['nullable', 'string', 'max:500'],
            ]
        )->validate();

        try {
            $scawatt = $this->buscarScawattPorIdentificador($identifier);
            if (!$scawatt->usuario || empty($scawatt->usuario->email)) {
                return response()->json([
                    'success' => false,
                    'message' => 'No se puede contactar al propietario para autorizar la descarga de documentos.',
                ], 422);
            }

            $this->expirarSolicitudesPendientes($scawatt->id);

            $codigo = $this->generarCodigoVerificacion();

            $solicitud = SolicitudDocumentosScawatt::create([
                'scawatt_id' => $scawatt->id,
                'propietario_id' => $scawatt->usuario->id,
                'codigo_hash' => hash('sha256', $codigo),
                'codigo_preview' => substr($codigo, -4),
                'estado' => 'pendiente',
                'solicitante_nombre' => $payload['nombre'] ?? null,
                'solicitante_email' => $payload['email'] ?? null,
                'solicitante_telefono' => $payload['telefono'] ?? null,
                'motivo' => $payload['motivo'] ?? null,
                'expires_at' => now()->addHours(6),
                'enviado_at' => now(),
            ]);

            $this->enviarCodigoVerificacion($scawatt, $codigo, $payload);

            Log::info('Solicitud de documentos de ScaWatt registrada', [
                'serial' => $scawatt->scawatt_id,
                'scawatt_id' => $scawatt->id,
                'solicitud_id' => $solicitud->id,
                'propietario' => $scawatt->usuario->email,
                'payload' => $payload,
                'ip' => $request->ip(),
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Enviamos un código de verificación al propietario. Compártelo cuando lo recibas para desbloquear los documentos.',
                'estado' => 'pendiente',
                'documentos' => [],
                'solicitud_id' => $solicitud->id,
                'expira_en' => optional($solicitud->expires_at)->toIso8601String(),
                'scawatt' => [
                    'id' => $scawatt->id,
                    'serial' => $scawatt->scawatt_id,
                    'estado' => $scawatt->estado,
                    'propietario' => [
                        'id' => $scawatt->usuario->id,
                        'nombre' => $scawatt->usuario->nombre,
                        'email' => $scawatt->usuario->email,
                    ],
                ],
            ]);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'ScaWatt no encontrado'
            ], 404);
        } catch (\Exception $e) {
            Log::error('Error al solicitar documentos de ScaWatt', [
                'error' => $e->getMessage(),
                'identifier' => $identifier,
                'payload' => $request->all(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'No fue posible registrar tu solicitud en este momento.',
                'error' => config('app.debug') ? $e->getMessage() : null,
            ], 500);
        }
    }

    public function verificarCodigoDocumentos(Request $request, $identifier)
    {
        $data = $request->validate([
            'codigo' => ['required', 'string', 'min:4', 'max:12'],
            'solicitud_id' => ['nullable', 'integer'],
        ]);

        $codigoIngresado = preg_replace('/\s+/', '', $data['codigo']);

        try {
            $scawatt = $this->buscarScawattPorIdentificador($identifier);

            $solicitudQuery = SolicitudDocumentosScawatt::where('scawatt_id', $scawatt->id)
                ->where('estado', 'pendiente');

            if (!empty($data['solicitud_id'])) {
                $solicitudQuery->where('id', $data['solicitud_id']);
            }

            $solicitud = $solicitudQuery->orderByDesc('created_at')->first();

            if (!$solicitud) {
                return response()->json([
                    'success' => false,
                    'message' => 'No hay solicitudes pendientes para este ScaWatt.',
                ], 404);
            }

            if ($solicitud->estaExpirada()) {
                $solicitud->update([
                    'estado' => 'expirado',
                    'finalizado_at' => now(),
                ]);

                return response()->json([
                    'success' => false,
                    'message' => 'El código ha expirado. Solicita uno nuevo.',
                    'estado' => 'expirado',
                ], 422);
            }

            $codigoHash = hash('sha256', $codigoIngresado);

            if (!hash_equals($solicitud->codigo_hash, $codigoHash)) {
                $solicitud->increment('intentos');

                return response()->json([
                    'success' => false,
                    'message' => 'Código incorrecto. Intenta nuevamente.',
                    'estado' => 'codigo_invalido',
                ], 422);
            }

            $solicitud->update([
                'estado' => 'autorizado',
                'autorizado_at' => now(),
                'finalizado_at' => now(),
            ]);

            $documentos = $this->obtenerDocumentosDeScawatt($scawatt->id);
            $documentosDisponibles = $this->filtrarDocumentosDisponibles($documentos);

            if ($documentosDisponibles->isEmpty() && $scawatt->usuario) {
                $this->copiarDocumentos($scawatt->usuario, $scawatt->scawatt_id);
                $documentos = $this->obtenerDocumentosDeScawatt($scawatt->id);
                $documentosDisponibles = $this->filtrarDocumentosDisponibles($documentos);
            }

            $solicitanteTieneEmail = !empty($solicitud->solicitante_email);
            $solicitanteEsPropietario = $solicitanteTieneEmail && $this->solicitanteCoincideConPropietario($solicitud->solicitante_email, $scawatt->usuario);

            $documentosParaRespuesta = (!$solicitanteTieneEmail || $solicitanteEsPropietario)
                ? $documentosDisponibles
                : $documentosDisponibles->filter(function ($doc) {
                    return strtolower($doc->tipo) === 'certificado';
                })->values();

            $enlacesCompartidos = $this->prepararEnlacesParaSolicitante($documentosDisponibles, $solicitud, $solicitanteEsPropietario);
            $correoEnviadoSolicitante = false;

            if ($solicitanteTieneEmail && !empty($enlacesCompartidos)) {
                $correoEnviadoSolicitante = $this->enviarDocumentosAlSolicitante($solicitud, $scawatt, $enlacesCompartidos);
            }

            $estado = $documentosDisponibles->isNotEmpty() ? 'disponible' : 'pendiente';

            Log::info('Código verificado para documentos de ScaWatt', [
                'serial' => $scawatt->scawatt_id,
                'scawatt_id' => $scawatt->id,
                'solicitud_id' => $solicitud->id,
                'estado' => $estado,
                'documentos_disponibles' => $documentosDisponibles->count(),
                'solicitante_email' => $solicitud->solicitante_email,
                'correo_enviado_al_solicitante' => $correoEnviadoSolicitante,
                'documentos_compartidos' => array_keys($enlacesCompartidos),
            ]);

            $this->registrarNotificacionesDocumentos(
                $solicitud,
                $scawatt,
                $documentosDisponibles,
                $enlacesCompartidos,
                $correoEnviadoSolicitante,
                $solicitanteEsPropietario
            );

            if ($correoEnviadoSolicitante) {
                $message = 'Código verificado. Enviamos un correo al solicitante con los documentos autorizados.';
            } else {
                $message = $estado === 'disponible'
                    ? 'Código verificado. Documentos listos para descargar.'
                    : 'Código verificado. Estamos preparando los documentos.';
            }

            return response()->json([
                'success' => true,
                'estado' => $estado,
                'message' => $message,
                'documentos' => $this->mapearDocumentos($documentosParaRespuesta),
                'solicitud_id' => $solicitud->id,
                'correo_enviado_al_solicitante' => $correoEnviadoSolicitante,
                'documentos_compartidos' => $enlacesCompartidos,
                'scawatt' => [
                    'id' => $scawatt->id,
                    'serial' => $scawatt->scawatt_id,
                    'estado' => $scawatt->estado,
                    'propietario' => $scawatt->usuario ? [
                        'id' => $scawatt->usuario->id,
                        'nombre' => $scawatt->usuario->nombre,
                        'email' => $scawatt->usuario->email,
                    ] : null,
                    'codigo_preview' => $solicitud->codigo_preview,
                    'codigo_expira' => optional($solicitud->expires_at)->toIso8601String(),
                ],
            ]);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'ScaWatt no encontrado'
            ], 404);
        } catch (\Exception $e) {
            Log::error('Error al verificar código de documentos de ScaWatt', [
                'error' => $e->getMessage(),
                'identifier' => $identifier,
                'payload' => $request->all(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'No fue posible validar el código en este momento.',
                'error' => config('app.debug') ? $e->getMessage() : null,
            ], 500);
        }
    }

    /**
     * Copiar documentos desde Python a Laravel storage
     */
    private function expirarSolicitudesPendientes(int $scawattId): void
    {
        SolicitudDocumentosScawatt::where('scawatt_id', $scawattId)
            ->where('estado', 'pendiente')
            ->update([
                'estado' => 'expirado',
                'finalizado_at' => now(),
            ]);
    }

    private function generarCodigoVerificacion(): string
    {
        return str_pad((string) random_int(0, 999999), 6, '0', STR_PAD_LEFT);
    }

    private function enviarCodigoVerificacion(Scawatt $scawatt, string $codigo, array $solicitante = []): void
    {
        try {
            Mail::to($scawatt->usuario->email)
                ->send(new SolicitudDocumentosScawattMail($scawatt, $codigo, $solicitante));
        } catch (\Throwable $exception) {
            Log::error('No se pudo enviar correo de código de documentos', [
                'error' => $exception->getMessage(),
                'serial' => $scawatt->scawatt_id,
            ]);
        }
    }

    private function solicitanteCoincideConPropietario(?string $solicitanteEmail, ?Usuario $propietario): bool
    {
        if (empty($solicitanteEmail) || !$propietario || empty($propietario->email)) {
            return false;
        }

        return strcasecmp(trim($solicitanteEmail), trim($propietario->email)) === 0;
    }

    private function prepararEnlacesParaSolicitante($documentosDisponibles, SolicitudDocumentosScawatt $solicitud, bool $solicitanteEsPropietario): array
    {
        if (empty($solicitud->solicitante_email)) {
            return [];
        }

        $mapa = [];
        foreach ($documentosDisponibles as $doc) {
            $tipo = strtolower($doc->tipo);
            $mapa[$tipo] = $doc->url_publica;
        }

        if (empty($mapa['certificado'])) {
            return [];
        }

        $enlaces = [
            'certificado' => $mapa['certificado'],
        ];

        if ($solicitanteEsPropietario) {
            foreach ($mapa as $tipo => $url) {
                if ($tipo === 'certificado') {
                    continue;
                }
                $enlaces[$tipo] = $url;
            }
        }

        return $enlaces;
    }

    private function normalizarDatosSolicitante(Request $request): array
    {
        $mapa = [
            'nombre' => ['nombre', 'solicitante_nombre', 'nombre_solicitante', 'nombreSolicitante'],
            'email' => ['email', 'correo', 'correo_electronico', 'solicitante_email', 'email_solicitante', 'emailSolicitante', 'correo_solicitante', 'correoSolicitante'],
            'telefono' => ['telefono', 'tel', 'celular', 'solicitante_telefono', 'telefono_solicitante', 'telefonoSolicitante'],
            'motivo' => ['motivo', 'motivo_solicitud', 'motivoSolicitud', 'comentario', 'comentarios', 'razon', 'razon_solicitud', 'razonSolicitud'],
        ];

        $resultado = [];

        foreach ($mapa as $destino => $fuentes) {
            foreach ($fuentes as $fuente) {
                $valor = $request->input($fuente);

                if ($valor === null) {
                    continue;
                }

                if (is_string($valor)) {
                    $valor = trim($valor);
                }

                if ($valor === '' || $valor === null) {
                    continue;
                }

                $resultado[$destino] = $valor;
                break;
            }
        }

        return $resultado;
    }

    private function enviarDocumentosAlSolicitante(SolicitudDocumentosScawatt $solicitud, Scawatt $scawatt, array $enlaces): bool
    {
        try {
            Mail::to($solicitud->solicitante_email)
                ->send(new DocumentosScawattCompartidosMail([
                    'scawatt_serial' => $scawatt->scawatt_id,
                    'propietario' => [
                        'nombre' => optional($scawatt->usuario)->nombre,
                        'email' => optional($scawatt->usuario)->email,
                    ],
                    'solicitante' => [
                        'nombre' => $solicitud->solicitante_nombre,
                        'email' => $solicitud->solicitante_email,
                    ],
                    'enlaces' => $enlaces,
                ]));

            Log::info('Documentos de ScaWatt enviados al solicitante', [
                'solicitud_id' => $solicitud->id,
                'solicitante_email' => $solicitud->solicitante_email,
                'documentos' => array_keys($enlaces),
            ]);

            return true;
        } catch (\Throwable $exception) {
            Log::error('No se pudieron enviar documentos al solicitante', [
                'solicitud_id' => $solicitud->id,
                'solicitante_email' => $solicitud->solicitante_email,
                'error' => $exception->getMessage(),
            ]);

            return false;
        }
    }

    private function registrarNotificacionesDocumentos(
        SolicitudDocumentosScawatt $solicitud,
        Scawatt $scawatt,
        $documentosDisponibles,
        array $enlacesCompartidos,
        bool $correoEnviadoSolicitante,
        bool $solicitanteEsPropietario
    ): void {
        try {
            $coleccionDocumentos = $documentosDisponibles instanceof \Illuminate\Support\Collection
                ? $documentosDisponibles
                : collect($documentosDisponibles);

            $destinatarios = collect();

            if ($scawatt->usuario_id) {
                $destinatarios->push($scawatt->usuario_id);
            }

            $masterCorreos = Usuario::correosMaster();
            $masterQuery = Usuario::query()->where('estado', 'master');
            if (!empty($masterCorreos)) {
                $masterQuery->orWhereIn('email', $masterCorreos);
            }

            $destinatarios = $destinatarios->merge($masterQuery->pluck('id'));
            $destinatarios = $destinatarios->unique()->filter();

            if ($destinatarios->isEmpty()) {
                return;
            }

            $documentosDisponiblesTipos = $coleccionDocumentos
                ->pluck('tipo')
                ->map(fn ($tipo) => strtolower((string) $tipo))
                ->unique()
                ->values()
                ->all();

            $documentosCompartidosTipos = array_values(array_unique(array_map('strtolower', array_keys($enlacesCompartidos))));

            $solicitanteLabel = $solicitud->solicitante_nombre
                ? trim($solicitud->solicitante_nombre . ' (' . ($solicitud->solicitante_email ?? 'sin email') . ')')
                : ($solicitud->solicitante_email ?? 'Solicitante sin email');

            $mensaje = sprintf(
                '%s autorizó la solicitud de %s sobre el ScaWatt %s.',
                optional($scawatt->usuario)->nombre ?? 'Un propietario',
                $solicitanteLabel,
                $scawatt->scawatt_id
            );

            if ($correoEnviadoSolicitante && !empty($documentosCompartidosTipos)) {
                $mensaje .= ' Documentos enviados: ' . implode(', ', $documentosCompartidosTipos) . '.';
            } elseif (!$correoEnviadoSolicitante) {
                $mensaje .= ' No se envió correo automático al solicitante.';
            }

            foreach ($destinatarios as $usuarioId) {
                Notificacion::create([
                    'usuario_id' => $usuarioId,
                    'tipo' => 'documentos_scawatt',
                    'titulo' => 'Autorización de documentos de ScaWatt',
                    'mensaje' => $mensaje,
                    'prioridad' => 'media',
                    'icono' => 'fa-file-signature',
                    'url_accion' => null,
                    'datos_adicionales' => [
                        'solicitud_id' => $solicitud->id,
                        'scawatt_id' => $scawatt->id,
                        'scawatt_serial' => $scawatt->scawatt_id,
                        'solicitante' => [
                            'nombre' => $solicitud->solicitante_nombre,
                            'email' => $solicitud->solicitante_email,
                            'es_propietario' => $solicitanteEsPropietario,
                        ],
                        'documentos_disponibles' => $documentosDisponiblesTipos,
                        'documentos_compartidos' => $documentosCompartidosTipos,
                        'correo_enviado' => $correoEnviadoSolicitante,
                    ],
                    'leido' => false,
                ]);
            }
        } catch (\Throwable $exception) {
            Log::error('No fue posible registrar notificación de documentos de ScaWatt', [
                'solicitud_id' => $solicitud->id,
                'error' => $exception->getMessage(),
            ]);
        }
    }

    private function copiarDocumentos($usuario, $scawattId)
    {
        return app(ScawattDocumentManager::class)->copyDocuments($usuario, $scawattId);
    }

    private function buscarScawattPorIdentificador($identifier)
    {
        return Scawatt::with('usuario')
            ->when(is_numeric($identifier), function ($query) use ($identifier) {
                return $query->where('id', $identifier);
            }, function ($query) use ($identifier) {
                return $query->where('scawatt_id', $identifier);
            })
            ->firstOrFail();
    }

    private function obtenerDocumentosDeScawatt(int $scawattId)
    {
        return DB::table('documentos')
            ->select('id', 'tipo', 'nombre_archivo', 'ruta_archivo', 'url_publica', 'fecha_generacion', 'created_at')
            ->where('scawatt_id', $scawattId)
            ->orderBy('created_at', 'desc')
            ->get();
    }

    private function filtrarDocumentosDisponibles($documentos)
    {
        return collect($documentos)->filter(function ($doc) {
            if (!$doc->ruta_archivo) {
                return false;
            }

            if (Storage::exists($doc->ruta_archivo)) {
                return true;
            }

            $rutaAbsoluta = storage_path('app/' . ltrim($doc->ruta_archivo, '/'));

            return file_exists($rutaAbsoluta);
        })->values();
    }

    private function mapearDocumentos($documentos)
    {
        return collect($documentos)->map(function ($doc) {
            return [
                'id' => $doc->id,
                'tipo' => $doc->tipo,
                'nombre_archivo' => $doc->nombre_archivo,
                'url' => $doc->url_publica,
                'fecha_generacion' => $doc->fecha_generacion,
                'registrado_en' => $doc->created_at,
            ];
        })->values();
    }

    /**
     * Sincronizar ScaWatts desde JSON a MySQL
     * GET /api/admin/scawatts/sync
     */
    public function sincronizar()
    {
        try {
            $jsonPath = '/home/scalarapp/scalar-data/scawatt-server/produccion-scawatts/data_scawatts/raiz-scawatt/scawatts_tokens.json';
            
            if (!file_exists($jsonPath)) {
                return response()->json([
                    'success' => false,
                    'error' => 'JSON de ScaWatts no encontrado'
                ], 404);
            }

            $scawattsJson = json_decode(file_get_contents($jsonPath), true);
            $sincronizados = 0;
            $errores = [];

            foreach ($scawattsJson as $scawattId => $data) {
                try {
                    // No validar usuario si no existe id_laravel
                    $usuarioId = null;
                    
                    if (isset($data['id_laravel'])) {
                        $usuario = Usuario::find($data['id_laravel']);
                        if ($usuario) {
                            $usuarioId = $usuario->id;
                        }
                    }

                    // Actualizar o crear en MySQL
                    Scawatt::updateOrCreate(
                        ['scawatt_id' => $scawattId],
                        [
                            'usuario_id' => $usuarioId,
                            'kwh_asignados' => 1000, // Valor por defecto
                            'valor_actual' => $data['valor'] + $data['valorizacion'],
                            'valor_inicial' => $data['valor'],
                            'valorizaciones' => json_encode([$data['valorizacion']]),
                            'fecha_inicio' => $data['fecha_inicio'],
                            'fecha_final' => $data['fecha_final'],
                            'estado' => $data['estado'] === 'Activo' ? 'activo' : 'finalizado'
                        ]
                    );

                    $sincronizados++;

                } catch (\Exception $e) {
                    $errores[] = "Error en {$scawattId}: " . $e->getMessage();
                }
            }

            return response()->json([
                'success' => true,
                'sincronizados' => $sincronizados,
                'total' => count($scawattsJson),
                'errores' => $errores
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Listar ScaWatts
     * GET /api/admin/scawatts
     */
    public function index()
    {
        $scawatts = Scawatt::with('usuario')->get();
        
        return response()->json([
            'success' => true,
            'data' => $scawatts
        ]);
    }

    /**
     * Listar ScaWatts pertenecientes a un usuario específico
     * GET /api/usuarios/{id}/scawatts
     */
    public function listarPorUsuario($usuarioId)
    {
        try {
            $usuario = Usuario::findOrFail($usuarioId);

            $scawatts = Scawatt::with('granja')
                ->where('usuario_id', $usuarioId)
                ->orderBy('created_at', 'desc')
                ->get();

            if ($scawatts->isEmpty()) {
                return response()->json([
                    'success' => true,
                    'usuario' => [
                        'id' => $usuario->id,
                        'nombre' => $usuario->nombre,
                        'email' => $usuario->email,
                    ],
                    'scawatts' => []
                ]);
            }

            $documentos = DB::table('documentos')
                ->select('id', 'scawatt_id', 'tipo', 'nombre_archivo', 'url_publica', 'fecha_generacion', 'created_at')
                ->whereIn('scawatt_id', $scawatts->pluck('id'))
                ->get()
                ->groupBy('scawatt_id');

            $scawattsFormateados = $scawatts->map(function (Scawatt $scawatt) use ($documentos) {
                $docs = $documentos->get($scawatt->id, collect())->map(function ($doc) {
                    return [
                        'id' => $doc->id,
                        'tipo' => $doc->tipo,
                        'nombre_archivo' => $doc->nombre_archivo,
                        'url' => $doc->url_publica,
                        'fecha_generacion' => $doc->fecha_generacion,
                        'registrado_en' => $doc->created_at,
                    ];
                })->values();

                $granja = $scawatt->granja;

                return [
                    'id' => $scawatt->id,
                    'serial' => $scawatt->scawatt_id,
                    'estado' => $scawatt->estado,
                    'tipo' => $scawatt->tipo,
                    'kwh_asignados' => $scawatt->kwh_asignados,
                    'valor_inicial' => $scawatt->valor_inicial,
                    'valor_actual' => $scawatt->valor_actual,
                    'valorizacion_total' => $scawatt->valorizacion_total,
                    'fecha_inicio' => $scawatt->fecha_inicio,
                    'fecha_final' => $scawatt->fecha_final,
                    'documentos' => $docs,
                    'granja' => $granja ? [
                        'id' => $granja->id,
                        'nombre' => $granja->nombre,
                        'departamento' => $granja->departamento,
                        'municipio' => $granja->municipio,
                        'direccion_completa' => $granja->direccion_completa,
                        'latitud' => $granja->latitud,
                        'longitud' => $granja->longitud,
                    ] : null,
                ];
            })->values();

            return response()->json([
                'success' => true,
                'usuario' => [
                    'id' => $usuario->id,
                    'nombre' => $usuario->nombre,
                    'email' => $usuario->email,
                ],
                'scawatts' => $scawattsFormateados
            ]);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Usuario no encontrado'
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al listar ScaWatts del usuario',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Registrar solicitud de contacto para adquirir ScaWatts
     * POST /api/admin/scawatts/solicitar-contacto
     */
    public function solicitarContacto(Request $request)
    {
        $validated = $request->validate([
            'usuario_id' => 'required|exists:usuarios,id',
            'nombre_completo' => 'nullable|string|max:255',
            'cedula' => 'nullable|string|max:50',
            'email' => 'nullable|email|max:255',
            'telefono' => 'nullable|string|max:30',
            'cantidad_scawatts' => 'nullable|integer|min:1',
            'presupuesto_aproximado' => 'nullable|numeric|min:0',
            'mensaje' => 'nullable|string|max:1000'
        ]);

        try {
            $usuario = Usuario::findOrFail($validated['usuario_id']);

            $solicitud = SolicitudContactoScawatt::create([
                'usuario_id' => $usuario->id,
                'nombre_completo' => $validated['nombre_completo'] ?? $usuario->nombre,
                'cedula' => $validated['cedula'] ?? $usuario->cedula,
                'email' => $validated['email'] ?? $usuario->email,
                'telefono' => $validated['telefono'] ?? $usuario->telefono,
                'cantidad_scawatts' => $validated['cantidad_scawatts'] ?? null,
                'presupuesto_aproximado' => $validated['presupuesto_aproximado'] ?? null,
                'mensaje' => $validated['mensaje'] ?? null,
                'estado' => 'pendiente'
            ]);

            app(SolicitudScawattWorkflowService::class)->handleNuevaSolicitud($solicitud);

            Log::info('Solicitud de contacto ScaWatt registrada', [
                'solicitud_id' => $solicitud->id,
                'usuario_id' => $usuario->id,
                'cantidad_scawatts' => $solicitud->cantidad_scawatts,
                'presupuesto' => $solicitud->presupuesto_aproximado
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Solicitud de contacto registrada exitosamente',
                'solicitud' => $solicitud
            ], 201);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Usuario no encontrado'
            ], 404);
        } catch (\Exception $e) {
            Log::error('Error al registrar solicitud de contacto ScaWatt', [
                'error' => $e->getMessage(),
                'payload' => $request->all()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Error al registrar la solicitud de contacto',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Obtener estadísticas de inversión de ScaWatts
     * GET /api/admin/scawatts/estadisticas
     * 
     * Retorna:
     * - total_inversion_inicial: Suma de todos los valores iniciales
     * - total_valor_actual: Suma de todos los valores actuales
     * - total_valorizacion: Ganancia total (valor_actual - valor_inicial)
     * - total_scawatts: Cantidad total de ScaWatts
     * - promedio_valorizacion: Valorización promedio por ScaWatt
     * - scawatts_activos: Cantidad de ScaWatts activos
     * - scawatts_inactivos: Cantidad de ScaWatts inactivos
     */
    public function estadisticas()
    {
        try {
            // Obtener todos los ScaWatts con sus usuarios
            $scawatts = Scawatt::with('usuario')->get();
            
            // Calcular estadísticas
            $totalInversionInicial = $scawatts->sum('valor_inicial');
            $totalValorActual = $scawatts->sum('valor_actual');
            $totalValorizacion = $totalValorActual - $totalInversionInicial;
            $totalScawatts = $scawatts->count();
            $promedioValorizacion = $totalScawatts > 0 ? $totalValorizacion / $totalScawatts : 0;
            
            // Estadísticas por estado
            $scawattsActivos = $scawatts->where('estado', 'activo')->count();
            $scawattsInactivos = $scawatts->where('estado', '!=', 'activo')->count();
            
            // Valorización promedio solo de ScaWatts activos
            $scawattsActivosData = $scawatts->where('estado', 'activo');
            $valorizacionActivos = $scawattsActivosData->sum(function ($s) {
                return $s->valor_actual - $s->valor_inicial;
            });
            $promedioValorizacionActivos = $scawattsActivos > 0 ? 
                $valorizacionActivos / $scawattsActivos : 0;
            
            return response()->json([
                'success' => true,
                'data' => [
                    'total_inversion_inicial' => round($totalInversionInicial, 2),
                    'total_valor_actual' => round($totalValorActual, 2),
                    'total_valorizacion' => round($totalValorizacion, 2),
                    'porcentaje_valorizacion' => $totalInversionInicial > 0 ? 
                        round(($totalValorizacion / $totalInversionInicial) * 100, 2) : 0,
                    'total_scawatts' => $totalScawatts,
                    'promedio_valorizacion' => round($promedioValorizacion, 2),
                    'scawatts_activos' => $scawattsActivos,
                    'scawatts_inactivos' => $scawattsInactivos,
                    'promedio_valorizacion_activos' => round($promedioValorizacionActivos, 2)
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al obtener estadísticas de ScaWatts',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Obtener estadísticas de inversión de un usuario específico
     * GET /api/usuarios/{id}/scawatts/estadisticas
     * 
     * Retorna las mismas métricas pero solo para los ScaWatts de ese usuario
     */
    public function estadisticasUsuario($usuarioId)
    {
        try {
            $usuario = Usuario::findOrFail($usuarioId);
            
            // Obtener ScaWatts del usuario
            $scawatts = Scawatt::where('usuario_id', $usuarioId)->get();
            
            if ($scawatts->isEmpty()) {
                return response()->json([
                    'success' => true,
                    'data' => [
                        'total_inversion_inicial' => 0,
                        'total_valor_actual' => 0,
                        'total_valorizacion' => 0,
                        'porcentaje_valorizacion' => 0,
                        'total_scawatts' => 0,
                        'promedio_valorizacion' => 0,
                        'scawatts_activos' => 0,
                        'scawatts_inactivos' => 0
                    ]
                ]);
            }
            
            // Calcular estadísticas
            $totalInversionInicial = $scawatts->sum('valor_inicial');
            $totalValorActual = $scawatts->sum('valor_actual');
            $totalValorizacion = $totalValorActual - $totalInversionInicial;
            $totalScawatts = $scawatts->count();
            $promedioValorizacion = $totalScawatts > 0 ? $totalValorizacion / $totalScawatts : 0;
            
            // Estadísticas por estado
            $scawattsActivos = $scawatts->where('estado', 'activo')->count();
            $scawattsInactivos = $scawatts->where('estado', '!=', 'activo')->count();
            
            return response()->json([
                'success' => true,
                'data' => [
                    'usuario' => [
                        'id' => $usuario->id,
                        'nombre' => $usuario->nombre,
                        'email' => $usuario->email
                    ],
                    'total_inversion_inicial' => round($totalInversionInicial, 2),
                    'total_valor_actual' => round($totalValorActual, 2),
                    'total_valorizacion' => round($totalValorizacion, 2),
                    'porcentaje_valorizacion' => $totalInversionInicial > 0 ? 
                        round(($totalValorizacion / $totalInversionInicial) * 100, 2) : 0,
                    'total_scawatts' => $totalScawatts,
                    'promedio_valorizacion' => round($promedioValorizacion, 2),
                    'scawatts_activos' => $scawattsActivos,
                    'scawatts_inactivos' => $scawattsInactivos
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al obtener estadísticas del usuario',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Verificar información pública de un ScaWatt
     * GET /api/public/scawatts/{serial}/verificar
     * 
     * Retorna información básica del ScaWatt sin revelar el propietario
     */
    public function verificarScawatt($serial)
    {
        try {
            // Buscar ScaWatt en base de datos
            $scawatt = Scawatt::where('scawatt_id', $serial)->first();
            
            if (!$scawatt) {
                return response()->json([
                    'success' => false,
                    'message' => 'ScaWatt no encontrado',
                    'valido' => false
                ], 404);
            }

            // Obtener información del usuario
            $usuario = Usuario::find($scawatt->usuario_id);

            return response()->json([
                'success' => true,
                'valido' => true,
                'data' => [
                    'serial' => $scawatt->scawatt_id,
                    'estado' => $scawatt->estado,
                    'fecha_creacion' => Carbon::parse($scawatt->fecha_inicio)->format('d/m/Y'),
                    'capacidad' => $scawatt->kwh_asignados ?? 0,
                    'tipo' => $scawatt->tipo ?? 'estándar',
                    'valor_inicial' => round($scawatt->valor_inicial, 2),
                    'valor_actual' => round($scawatt->valor_actual, 2),
                    'propietario_iniciales' => $usuario ? $this->obtenerIniciales($usuario->nombre) : 'N/A',
                    'requiere_codigo' => true
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al verificar ScaWatt',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Solicitar código de verificación por email
     * POST /api/public/scawatts/{serial}/solicitar-codigo
     */
    public function solicitarCodigoVerificacion($serial)
    {
        try {
            // Buscar ScaWatt
            $scawatt = Scawatt::where('scawatt_id', $serial)->first();
            
            if (!$scawatt) {
                return response()->json([
                    'success' => false,
                    'message' => 'ScaWatt no encontrado'
                ], 404);
            }

            // Obtener usuario
            $usuario = Usuario::find($scawatt->usuario_id);
            
            if (!$usuario || !$usuario->email) {
                return response()->json([
                    'success' => false,
                    'message' => 'Información del propietario no disponible'
                ], 404);
            }

            // Generar código aleatorio de 4 dígitos
            $codigo = str_pad(rand(0, 9999), 4, '0', STR_PAD_LEFT);
            
            // Guardar en cache por 10 minutos
            $cacheKey = "verificacion_scawatt_{$serial}";
            Cache::put($cacheKey, [
                'codigo' => $codigo,
                'usuario_id' => $usuario->id,
                'intentos' => 0
            ], now()->addMinutes(10));

            // Enviar email
            try {
                Mail::raw(
                    "Hola {$usuario->nombre},\n\n" .
                    "Alguien está solicitando verificar la propiedad de tu ScaWatt {$serial}.\n\n" .
                    "Tu código de verificación es: {$codigo}\n\n" .
                    "Este código expirará en 10 minutos.\n\n" .
                    "Si no fuiste tú quien solicitó este código, puedes ignorar este mensaje.\n\n" .
                    "Saludos,\nEquipo Scalar",
                    function ($message) use ($usuario, $serial) {
                        $message->to($usuario->email)
                                ->subject("Código de verificación ScaWatt {$serial}");
                    }
                );
            } catch (\Exception $e) {
                Log::error('Error enviando email de verificación', [
                    'serial' => $serial,
                    'error' => $e->getMessage()
                ]);
            }

            return response()->json([
                'success' => true,
                'message' => 'Código enviado al correo del propietario',
                'email_preview' => substr($usuario->email, 0, 3) . '***@' . explode('@', $usuario->email)[1],
                'expira_en' => 10 // minutos
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al solicitar código',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Verificar propietario con código
     * POST /api/public/scawatts/{serial}/verificar-propietario
     * 
     * Body: { "codigo": "1234" }
     * Código = código aleatorio de 4 dígitos enviado por email
     */

    public function verificarPropietario($serial, Request $request)
    {
        $request->validate([
            'codigo' => 'required|string|min:4|max:4'
        ]);

        try {
            // Buscar ScaWatt
            $scawatt = Scawatt::where('scawatt_id', $serial)->first();
            
            if (!$scawatt) {
                return response()->json([
                    'success' => false,
                    'message' => 'ScaWatt no encontrado'
                ], 404);
            }

            // Obtener usuario
            $usuario = Usuario::find($scawatt->usuario_id);
            
            if (!$usuario) {
                return response()->json([
                    'success' => false,
                    'message' => 'Información del propietario no disponible'
                ], 404);
            }

            // Verificar código desde cache
            $cacheKey = "verificacion_scawatt_{$serial}";
            $datosVerificacion = Cache::get($cacheKey);
            
            if (!$datosVerificacion) {
                return response()->json([
                    'success' => false,
                    'message' => 'Código expirado o no solicitado. Por favor solicita un nuevo código.'
                ], 401);
            }

            // Verificar intentos
            if ($datosVerificacion['intentos'] >= 3) {
                Cache::forget($cacheKey);
                return response()->json([
                    'success' => false,
                    'message' => 'Demasiados intentos fallidos. Solicita un nuevo código.'
                ], 401);
            }

            // Verificar código
            if ($request->codigo !== $datosVerificacion['codigo']) {
                // Incrementar intentos
                $datosVerificacion['intentos']++;
                Cache::put($cacheKey, $datosVerificacion, now()->addMinutes(10));
                
                return response()->json([
                    'success' => false,
                    'message' => 'Código incorrecto',
                    'intentos_restantes' => 3 - $datosVerificacion['intentos']
                ], 401);
            }

            // Código correcto - limpiar cache
            Cache::forget($cacheKey);

            // Revelar propietario
            return response()->json([
                'success' => true,
                'verificado' => true,
                'data' => [
                    'propietario' => [
                        'nombre' => $usuario->nombre,
                        'email' => $usuario->email,
                        'tipo_documento' => 'CC',
                        'documento' => $usuario->cedula
                    ],
                    'scawatt' => [
                        'serial' => $scawatt->scawatt_id,
                        'estado' => $scawatt->estado,
                        'fecha_creacion' => Carbon::parse($scawatt->fecha_inicio)->format('d/m/Y'),
                        'capacidad' => $scawatt->kwh_asignados ?? 0,
                        'tipo' => $scawatt->tipo ?? 'estándar',
                        'valor_inicial' => round($scawatt->valor_inicial, 2),
                        'valor_actual' => round($scawatt->valor_actual, 2),
                        'valorizacion' => round($scawatt->valor_actual - $scawatt->valor_inicial, 2),
                        'porcentaje_valorizacion' => $scawatt->valor_inicial > 0 ? 
                            round((($scawatt->valor_actual - $scawatt->valor_inicial) / $scawatt->valor_inicial) * 100, 2) : 0
                    ]
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al verificar propietario',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Helper: Obtener iniciales del nombre
     */
    private function obtenerIniciales($nombre)
    {
        $palabras = explode(' ', trim($nombre));
        $iniciales = '';
        
        foreach (array_slice($palabras, 0, 2) as $palabra) {
            if (strlen($palabra) > 0) {
                $iniciales .= strtoupper($palabra[0]) . '.';
            }
        }
        
        return $iniciales ?: 'N/A';
    }
}
