<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Carbon\Carbon;

class ScawattValorController extends Controller
{
    /**
     * Ruta al archivo JSON con valores históricos
     */
    private $jsonPath = '/home/scalarapp/public_html/develop/code-test/ScaWatt.json';

    /**
     * Obtener valor actual del ScaWatt (tiempo real)
     * GET /api/scawatt/actual
     */
    public function valorActual()
    {
        try {
            // Ruta al archivo JSON
            $jsonPath = '/home/scalarapp/public_html/develop/code-test/ScaWatt.json';
            
            if (!file_exists($jsonPath)) {
                return response()->json([
                    'error' => 'Archivo de datos no encontrado'
                ], 404);
            }
            
            // Leer JSON y limpiar formato (remover comas extras al final)
            $jsonContent = file_get_contents($jsonPath);
            // Remover coma antes de ] o }
            $jsonContent = preg_replace('/,\s*([\]}])/m', '$1', $jsonContent);
            
            $data = json_decode($jsonContent, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                return response()->json([
                    'error' => 'Error al leer datos: ' . json_last_error_msg()
                ], 500);
            }
            
            // Obtener hora actual en Bogotá (America/Bogota = UTC-5)
            $now = new \DateTime('now', new \DateTimeZone('America/Bogota'));
            $horaActual = $now->format('Y-m-d H:i:s');
            
            // Buscar el registro más cercano a la hora actual
            $registroActual = null;
            $menorDiferencia = PHP_INT_MAX;
            
            foreach ($data as $registro) {
                // Parsear fecha formato dd/mm/yyyy HH:mm:ss
                $fechaRegistro = \DateTime::createFromFormat('d/m/Y H:i:s', $registro['fecha'], new \DateTimeZone('America/Bogota'));
                if (!$fechaRegistro) {
                    continue; // Saltar si no se puede parsear
                }
                
                $diferencia = abs($now->getTimestamp() - $fechaRegistro->getTimestamp());
                
                if ($diferencia < $menorDiferencia) {
                    $menorDiferencia = $diferencia;
                    $registroActual = $registro;
                }
            }
            
            if (!$registroActual) {
                return response()->json([
                    'error' => 'No se encontraron registros'
                ], 404);
            }
            
            // Formato de respuesta
            return response()->json([
                'registro' => [
                    'valor' => $registroActual['valor'],
                    'fecha' => $registroActual['fecha']
                ]
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Error al procesar solicitud: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Obtener historial de valores del ScaWatt
     * GET /api/scawatt/historial?days=1&limit=10
     * 
     * Devuelve los registros de los últimos N días desde la hora actual de Colombia
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function historial(Request $request)
    {
        try {
            // Validar parámetro days (por defecto 1 día)
            $days = $request->input('days', 1);
            
            // Validar parámetro limit (opcional, sin límite por defecto)
            $limit = $request->input('limit', null);
            
            // Validar que sea un número válido entre 1 y 365
            if (!is_numeric($days) || $days < 1 || $days > 365) {
                return response()->json([
                    'success' => false,
                    'message' => 'El parámetro days debe ser un número entre 1 y 365'
                ], 400);
            }
            
            // Validar limit si se proporciona
            if ($limit !== null && (!is_numeric($limit) || $limit < 1)) {
                return response()->json([
                    'success' => false,
                    'message' => 'El parámetro limit debe ser un número mayor a 0'
                ], 400);
            }

            // Verificar que el archivo existe
            if (!file_exists($this->jsonPath)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Archivo de datos no encontrado'
                ], 404);
            }

            // Leer JSON y limpiar formato (remover coma al final si existe)
            $jsonContent = file_get_contents($this->jsonPath);
            $jsonContent = preg_replace('/,\s*([\]}])/m', '$1', $jsonContent);
            $registros = json_decode($jsonContent, true);

            if (empty($registros)) {
                return response()->json([
                    'success' => false,
                    'message' => 'No hay datos disponibles'
                ], 404);
            }

            // Obtener hora actual de Bogotá
            $ahora = Carbon::now('America/Bogota');
            
            // Calcular fecha límite (ahora - N días)
            $fechaLimite = $ahora->copy()->subDays($days);

            // Filtrar registros dentro del rango
            $historial = [];
            foreach ($registros as $registro) {
                // Parsear fecha del registro
                $fechaRegistro = Carbon::createFromFormat('d/m/Y H:i:s', $registro['fecha'], 'America/Bogota');
                
                // Solo incluir si está dentro del rango (últimos N días)
                if ($fechaRegistro->greaterThanOrEqualTo($fechaLimite) && $fechaRegistro->lessThanOrEqualTo($ahora)) {
                    $historial[] = [
                        'valor' => (string) $registro['valor'],
                        'fecha' => $registro['fecha']
                    ];
                }
            }

            // Ordenar por fecha descendente (más reciente primero)
            usort($historial, function($a, $b) {
                $fechaA = Carbon::createFromFormat('d/m/Y H:i:s', $a['fecha']);
                $fechaB = Carbon::createFromFormat('d/m/Y H:i:s', $b['fecha']);
                return $fechaB->timestamp - $fechaA->timestamp;
            });

            // Aplicar límite si se especificó
            if ($limit !== null) {
                $historial = array_slice($historial, 0, (int) $limit);
            }

            // Devolver array directo (sin wrapper)
            return response()->json($historial);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al obtener historial',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Estadísticas del día actual
     * GET /api/scawatt/estadisticas-dia
     * 
     * Devuelve min, max, promedio del día actual
     */
    public function estadisticasDia()
    {
        try {
            if (!file_exists($this->jsonPath)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Archivo de datos no encontrado'
                ], 404);
            }

            $jsonContent = file_get_contents($this->jsonPath);
            $registros = json_decode($jsonContent, true);

            if (empty($registros)) {
                return response()->json([
                    'success' => false,
                    'message' => 'No hay datos disponibles'
                ], 404);
            }

            // Obtener registros del día actual
            $hoy = Carbon::now('America/Bogota')->startOfDay();
            $registrosHoy = [];

            foreach ($registros as $registro) {
                $fechaRegistro = Carbon::createFromFormat('d/m/Y H:i:s', $registro['fecha'], 'America/Bogota');
                
                if ($fechaRegistro->isSameDay($hoy)) {
                    $registrosHoy[] = (float) $registro['valor'];
                }
            }

            if (empty($registrosHoy)) {
                return response()->json([
                    'success' => false,
                    'message' => 'No hay datos del día actual'
                ], 404);
            }

            $estadisticas = [
                'fecha' => $hoy->format('d/m/Y'),
                'total_registros' => count($registrosHoy),
                'valor_minimo' => min($registrosHoy),
                'valor_maximo' => max($registrosHoy),
                'valor_promedio' => round(array_sum($registrosHoy) / count($registrosHoy), 4),
                'ultimo_valor' => end($registrosHoy)
            ];

            return response()->json($estadisticas);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al obtener estadísticas',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
