<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

class SolarController extends Controller
{
    /**
     * Obtener datos solares actuales
     * GET /api/solar/current
     */
    public function current()
    {
        try {
            // Obtener datos de la granja desde la base de datos
            $granja = \App\Models\Granja::where('estado', 'operativa')->first();
            
            if (!$granja) {
                throw new \Exception('No hay granjas operativas');
            }
            
            // Leer datos solares más recientes de Stormglass
            $solarDataPath = '/home/scalarapp/scalar-data/api_weather_google/la-gloria-scalar/stormglass/weather_data/';
            
            // Buscar el archivo más reciente
            $files = glob($solarDataPath . 'solar_report_*.json');
            if (empty($files)) {
                throw new \Exception('No se encontraron datos solares');
            }
            
            // Ordenar por fecha de modificación (más reciente primero)
            usort($files, function($a, $b) {
                return filemtime($b) - filemtime($a);
            });
            
            $latestFile = $files[0];
            $solarData = json_decode(file_get_contents($latestFile), true);
            
            // Obtener hora actual local
            $currentHour = now('America/Bogota')->hour;
            
            // Buscar datos de la hora actual en el forecast
            $currentConditions = $solarData['current_conditions'] ?? null;
            $hourlyData = null;
            
            if (isset($solarData['daily_forecast'])) {
                foreach ($solarData['daily_forecast'] as $forecast) {
                    if ($forecast['hour_local'] == $currentHour) {
                        $hourlyData = $forecast;
                        break;
                    }
                }
            }
            
            // Usar datos de la hora actual o condiciones actuales
            $solarRadiation = $hourlyData['solar_radiation'] ?? $currentConditions['solar_radiation'] ?? 0;
            $uvIndex = $hourlyData['uv_index'] ?? $currentConditions['uv_index'] ?? 0;
            $uvLevel = $hourlyData['uv_level'] ?? $currentConditions['uv_level'] ?? null;
            $solarLevel = $hourlyData['solar_level'] ?? $currentConditions['solar_level'] ?? null;
            
            // Calcular potencia basada en capacidad real de la granja
            $capacidadMW = (float) $granja->capacidad_mw; // 1 MW = 1000 kW
            $systemEfficiency = 0.85; // 85% de eficiencia del sistema
            
            // Potencia actual basada en radiación solar (irradiancia estándar = 1000 W/m²)
            $factorRadiacion = $solarRadiation / 1000; // Factor de radiación actual vs estándar
            $potenciaActualKW = $capacidadMW * 1000 * $factorRadiacion * $systemEfficiency;
            
            // Producción estimada del día (asumiendo 5 horas pico de sol)
            $horasPico = 5;
            $produccionDiariaKWh = $capacidadMW * 1000 * $horasPico * $systemEfficiency;
            
            // DEVOLVER DATOS DIRECTOS SIN WRAPPER (formato original)
            return response()->json([
                // Datos de la granja
                'granja' => [
                    'id' => $granja->id,
                    'nombre' => $granja->nombre,
                    'capacidad_mw' => $capacidadMW,
                    'capacidad_kw' => $capacidadMW * 1000,
                    'produccion_anual_estimada_kwh' => (float) $granja->produccion_estimada_anual,
                    'estado' => $granja->estado,
                    'scawatts_activos' => $granja->scawatts()->where('estado', 'activo')->count()
                ],
                
                // Datos solares actuales
                'solar' => [
                    'irradiance' => round($solarRadiation, 2), // W/m²
                    'uv_index' => round($uvIndex, 2),
                    'uv_level' => $uvLevel,
                    'solar_level' => $solarLevel
                ],
                
                // Generación actual
                'generation' => [
                    'current_power_kw' => round($potenciaActualKW, 2),
                    'current_power_mw' => round($potenciaActualKW / 1000, 3),
                    'max_power_kw' => $capacidadMW * 1000,
                    'efficiency_percent' => $systemEfficiency * 100,
                    'capacity_factor' => round($factorRadiacion * 100, 2), // % de capacidad actual
                    'estimated_daily_production_kwh' => round($produccionDiariaKWh, 2)
                ],
                
                // Metadata
                'timestamp' => $solarData['timestamp'] ?? now()->toIso8601String(),
                'location' => $solarData['location'] ?? [
                    'name' => $granja->nombre,
                    'latitude' => (float) $granja->latitud ?: 8.629667,
                    'longitude' => (float) $granja->longitud ?: -73.782861,
                    'municipio' => $granja->municipio,
                    'departamento' => $granja->departamento
                ]
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Error al obtener datos solares', ['exception' => $e->getMessage()]);

            return response()->json([
                'success' => false,
                'message' => 'No fue posible obtener datos solares en tiempo real.',
                'error' => $e->getMessage()
            ], 503);
        }
    }
    
    /**
     * Calcular irradiancia según la hora del día
     * @param int $hour Hora del día (0-23)
     * @return float Irradiancia en W/m²
     */
    private function calculateIrradiance($hour)
    {
        // La irradiancia solar varía durante el día
        // Máximo alrededor del mediodía (12-14h), mínimo en la noche
        
        if ($hour >= 6 && $hour <= 18) {
            // Durante el día
            // Usar una función seno para simular la curva solar
            $t = ($hour - 6) / 12; // Normalizar entre 0 y 1
            $base = 1000; // Irradiancia máxima (W/m²)
            $irradiance = $base * sin($t * pi());
            
            // Agregar variación aleatoria ±50 W/m²
            $irradiance += rand(-50, 50);
            
            return max(0, $irradiance);
        } else {
            // Durante la noche
            return 0;
        }
    }
}
