<?php

namespace App\Http\Controllers;

use App\Models\Deposito;
use App\Models\Retiro;
use App\Models\Usuario;
use App\Models\AuditoriaAdmin;
use App\Models\Notificacion;
use App\Services\FirebaseService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;

class TransaccionController extends Controller
{
    protected $firebaseService;

    public function __construct(FirebaseService $firebaseService)
    {
        $this->firebaseService = $firebaseService;
    }
    /**
     * Listar depósitos
     * GET /api/admin/transacciones/depositos
     */
    public function listarDepositos(Request $request)
    {
        $query = Deposito::with('usuario');

        // Filtros
        if ($request->has('estado')) {
            $query->where('estado', $request->estado);
        }

        if ($request->has('usuario_id')) {
            $query->where('usuario_id', $request->usuario_id);
        }

        if ($request->has('fecha_desde')) {
            $query->whereDate('created_at', '>=', $request->fecha_desde);
        }

        if ($request->has('fecha_hasta')) {
            $query->whereDate('created_at', '<=', $request->fecha_hasta);
        }

        $depositos = $query->orderBy('created_at', 'desc')->paginate(20);

        return response()->json($depositos);
    }

    /**
     * Listar retiros
     * GET /api/admin/transacciones/retiros
     */
    public function listarRetiros(Request $request)
    {
        $query = Retiro::with('usuario');

        // Filtros
        if ($request->has('estado')) {
            $query->where('estado', $request->estado);
        }

        if ($request->has('usuario_id')) {
            $query->where('usuario_id', $request->usuario_id);
        }

        if ($request->has('fecha_desde')) {
            $query->whereDate('created_at', '>=', $request->fecha_desde);
        }

        if ($request->has('fecha_hasta')) {
            $query->whereDate('created_at', '<=', $request->fecha_hasta);
        }

        $retiros = $query->orderBy('created_at', 'desc')->paginate(20);

        return response()->json($retiros);
    }

    /**
     * Crear depósito
     * POST /api/transacciones/depositos
     */
    public function crearDeposito(Request $request)
    {
        $request->validate([
            'usuario_id' => 'required|exists:usuarios,id',
            'monto' => 'required|numeric|min:0.01',
            'metodo_pago' => 'required|in:transferencia,tarjeta,efectivo,otro',
            'referencia_pago' => 'nullable|string|max:255',
            'referencia' => 'nullable|string|max:255',
            'comprobante' => 'nullable|string'
        ]);

        try {
            DB::beginTransaction();

            $deposito = Deposito::create([
                'usuario_id' => $request->usuario_id,
                'monto' => $request->monto,
                'metodo_pago' => $request->metodo_pago,
                'referencia' => $request->input('referencia_pago') ?? $request->input('referencia'),
                'estado' => 'pendiente'
            ]);

            // Crear notificación con PUSH
            $this->crearNotificacionConPush(
                $request->usuario_id,
                'info',
                'Depósito registrado',
                "Tu depósito de $" . number_format($deposito->monto, 2) . " está pendiente de aprobación.",
                ['tipo' => 'deposito', 'deposito_id' => $deposito->id]
            );

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Depósito creado exitosamente',
                'deposito' => $deposito
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error al crear depósito',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Aprobar depósito
     * PUT /api/admin/transacciones/depositos/{id}/aprobar
     */
    public function aprobarDeposito(Request $request, $id)
    {
        $request->validate([
            'admin_id' => 'required|exists:usuarios,id',
            'notas_admin' => 'nullable|string'
        ]);

        try {
            $deposito = Deposito::findOrFail($id);

            DB::beginTransaction();

            $deposito->aprobar($request->admin_id, $request->notas_admin);

            // Registrar auditoría
            AuditoriaAdmin::registrar(
                $request->admin_id,
                'aprobar_deposito',
                'transacciones',
                "Depósito aprobado por $" . number_format($deposito->monto, 2) . " para usuario ID: {$deposito->usuario_id}",
                'Deposito',
                $deposito->id
            );

            // Notificar al usuario con PUSH
            $this->crearNotificacionConPush(
                $deposito->usuario_id,
                'success',
                'Depósito aprobado',
                "Tu depósito de $" . number_format($deposito->monto, 2) . " ha sido aprobado y acreditado.",
                [
                    'tipo' => 'deposito',
                    'deposito_id' => $deposito->id,
                    'monto' => $deposito->monto
                ]
            );

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Depósito aprobado exitosamente',
                'deposito' => $deposito
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error al aprobar depósito',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Rechazar depósito
     * PUT /api/admin/transacciones/depositos/{id}/rechazar
     */
    public function rechazarDeposito(Request $request, $id)
    {
        $request->validate([
            'admin_id' => 'required|exists:usuarios,id',
            'motivo' => 'required|string'
        ]);

        try {
            $deposito = Deposito::findOrFail($id);

            DB::beginTransaction();

            $deposito->rechazar($request->admin_id, $request->motivo);

            // Registrar auditoría
            AuditoriaAdmin::registrar(
                $request->admin_id,
                'rechazar_deposito',
                'transacciones',
                "Depósito rechazado: {$request->motivo}",
                'Deposito',
                $deposito->id
            );

            // Notificar al usuario con PUSH
            $this->crearNotificacionConPush(
                $deposito->usuario_id,
                'error',
                'Depósito rechazado',
                "Tu depósito de $" . number_format($deposito->monto, 2) . " fue rechazado. Motivo: {$request->motivo}",
                ['tipo' => 'deposito', 'deposito_id' => $deposito->id]
            );

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Depósito rechazado',
                'deposito' => $deposito
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error al rechazar depósito',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Solicitar retiro
     * POST /api/transacciones/retiros
     */
    public function solicitarRetiro(Request $request)
    {
        $request->validate([
            'usuario_id' => 'required|exists:usuarios,id',
            'monto' => 'required|numeric|min:0.01',
            'metodo_retiro' => 'required|in:transferencia,cheque,efectivo',
            'datos_bancarios' => 'required|array'
        ]);

        try {
            $usuario = Usuario::findOrFail($request->usuario_id);

            // Verificar saldo disponible
            if ($usuario->saldo_disponible < $request->monto) {
                return response()->json([
                    'success' => false,
                    'message' => 'Saldo insuficiente',
                    'saldo_disponible' => $usuario->saldo_disponible
                ], 400);
            }

            DB::beginTransaction();

            $retiro = Retiro::create([
                'usuario_id' => $request->usuario_id,
                'monto' => $request->monto,
                'metodo_retiro' => $request->metodo_retiro,
                'datos_bancarios' => $request->datos_bancarios,
                'estado' => 'pendiente'
            ]);

            // Congelar saldo
            $usuario->saldo_disponible -= $request->monto;
            $usuario->saldo_pendiente += $request->monto;
            $usuario->save();

            // Notificar al usuario con PUSH
            $this->crearNotificacionConPush(
                $request->usuario_id,
                'info',
                'Retiro solicitado',
                "Tu solicitud de retiro por $" . number_format($retiro->monto, 2) . " está en proceso.",
                ['tipo' => 'retiro', 'retiro_id' => $retiro->id]
            );

            // Aviso interno por correo para el equipo administrador
            $this->avisarRetiroPorCorreo($usuario, $retiro);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Retiro solicitado exitosamente',
                'retiro' => $retiro
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error al solicitar retiro',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Aprobar retiro
     * PUT /api/admin/transacciones/retiros/{id}/aprobar
     */
    public function aprobarRetiro(Request $request, $id)
    {
        $request->validate([
            'admin_id' => 'required|exists:usuarios,id',
            'referencia_transaccion' => 'required|string',
            'notas_admin' => 'nullable|string'
        ]);

        try {
            $retiro = Retiro::findOrFail($id);

            DB::beginTransaction();

            $retiro->aprobar(
                $request->admin_id,
                $request->referencia_transaccion,
                $request->notas_admin
            );

            // Registrar auditoría
            AuditoriaAdmin::registrar(
                $request->admin_id,
                'aprobar_retiro',
                'transacciones',
                "Retiro aprobado por $" . number_format($retiro->monto, 2) . " para usuario ID: {$retiro->usuario_id}. Ref: {$request->referencia_transaccion}",
                'Retiro',
                $retiro->id
            );

            // Notificar al usuario con PUSH
            $this->crearNotificacionConPush(
                $retiro->usuario_id,
                'success',
                'Retiro procesado',
                "Tu retiro de $" . number_format($retiro->monto, 2) . " ha sido procesado. Referencia: {$request->referencia_transaccion}",
                [
                    'tipo' => 'retiro',
                    'retiro_id' => $retiro->id,
                    'monto' => $retiro->monto,
                    'referencia' => $request->referencia_transaccion
                ]
            );

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Retiro aprobado exitosamente',
                'retiro' => $retiro
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error al aprobar retiro',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Rechazar retiro
     * PUT /api/admin/transacciones/retiros/{id}/rechazar
     */
    public function rechazarRetiro(Request $request, $id)
    {
        $request->validate([
            'admin_id' => 'required|exists:usuarios,id',
            'motivo' => 'required|string'
        ]);

        try {
            $retiro = Retiro::findOrFail($id);

            DB::beginTransaction();

            $retiro->rechazar($request->admin_id, $request->motivo);

            // Registrar auditoría
            AuditoriaAdmin::registrar(
                $request->admin_id,
                'rechazar_retiro',
                'transacciones',
                "Retiro rechazado: {$request->motivo}",
                'Retiro',
                $retiro->id
            );

            // Notificar al usuario con PUSH
            $this->crearNotificacionConPush(
                $retiro->usuario_id,
                'warning',
                'Retiro rechazado',
                "Tu solicitud de retiro por $" . number_format($retiro->monto, 2) . " fue rechazada. Motivo: {$request->motivo}. El saldo ha sido reintegrado.",
                ['tipo' => 'retiro', 'retiro_id' => $retiro->id]
            );

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Retiro rechazado',
                'retiro' => $retiro
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error al rechazar retiro',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Estadísticas de transacciones
     * GET /api/admin/transacciones/estadisticas
     */
    public function estadisticas()
    {
        $stats = [
            'depositos' => [
                'pendientes' => Deposito::pendientes()->count(),
                'aprobados_hoy' => Deposito::aprobados()->whereDate('fecha_aprobacion', today())->count(),
                'monto_total_aprobados' => Deposito::aprobados()->sum('monto')
            ],
            'retiros' => [
                'pendientes' => Retiro::pendientes()->count(),
                'procesados_hoy' => Retiro::procesados()->whereDate('fecha_procesado', today())->count(),
                'monto_total_procesados' => Retiro::procesados()->sum('monto')
            ]
        ];

        return response()->json($stats);
    }

    /**
     * Helper: Crear notificación con PUSH automático
     */
    private function crearNotificacionConPush($usuarioId, $tipo, $titulo, $mensaje, $datosAdicionales = [])
    {
        $usuario = Usuario::find($usuarioId);
        
        // Crear notificación en DB
        $notificacion = Notificacion::create([
            'usuario_id' => $usuarioId,
            'tipo' => $tipo,
            'titulo' => $titulo,
            'mensaje' => $mensaje,
            'prioridad' => $tipo == 'error' ? 'alta' : 'media',
            'icono' => $this->obtenerIconoPorTipo($tipo),
            'datos_adicionales' => $datosAdicionales
        ]);

        // Enviar PUSH si tiene token
        if ($usuario && $usuario->fcm_token) {
            $this->firebaseService->enviarNotificacion(
                $usuario->fcm_token,
                $titulo,
                $mensaje,
                array_merge($datosAdicionales, [
                    'tipo' => $tipo,
                    'notificacion_id' => $notificacion->id
                ])
            );
        }

        return $notificacion;
    }

    /**
     * Enviar correo a los administradores al generarse un retiro
     */
    private function avisarRetiroPorCorreo(Usuario $usuario, Retiro $retiro): void
    {
        $destinatarios = ['notificaciones@scalar.com.co'];
        $copia = ['desarrollo@wosh.com.co'];

        $datosBancarios = $retiro->datos_bancarios ?? [];

        try {
            Log::info('Avisar retiro por correo: preparando envío', [
                'retiro_id' => $retiro->id,
                'usuario_id' => $usuario->id,
                'monto' => $retiro->monto,
                'destinatarios' => $destinatarios,
                'copia' => $copia,
            ]);

            Mail::send(
                'emails.retiro-solicitado',
                [
                    'usuario' => $usuario,
                    'retiro' => $retiro,
                    'datosBancarios' => $datosBancarios,
                ],
                function ($message) use ($destinatarios, $copia, $retiro, $usuario) {
                    $message->to($destinatarios)
                        ->cc($copia)
                        ->subject('Nueva solicitud de retiro #' . $retiro->id . ' - ' . $usuario->nombre);
                }
            );

            Log::info('Avisar retiro por correo: envío finalizado', [
                'retiro_id' => $retiro->id,
                'usuario_id' => $usuario->id,
            ]);

            $mailer = Mail::getFacadeRoot();
            if ($mailer && method_exists($mailer, 'failures')) {
                $fallidos = $mailer->failures();
                if (!empty($fallidos)) {
                    Log::warning('Avisar retiro por correo: destinatarios fallidos', [
                        'retiro_id' => $retiro->id,
                        'usuario_id' => $usuario->id,
                        'fallidos' => $fallidos,
                    ]);
                }
            }
        } catch (\Throwable $th) {
            Log::error('No se pudo enviar el correo de alerta de retiro', [
                'retiro_id' => $retiro->id,
                'usuario_id' => $usuario->id,
                'error' => $th->getMessage(),
            ]);
        }
    }

    /**
     * Obtener icono según tipo de notificación
     */
    private function obtenerIconoPorTipo($tipo)
    {
        return match($tipo) {
            'success' => 'check_circle',
            'error' => 'cancel',
            'warning' => 'warning',
            'info' => 'info',
            default => 'notifications'
        };
    }

    /**
     * Registrar depósito manual por admin (auto-aprobado)
     * POST /api/admin/transacciones/deposito-manual
     */
    public function registrarDepositoManual(Request $request)
    {
        $request->validate([
            'usuario_id' => 'required|exists:usuarios,id',
            'monto' => 'required|numeric|min:1000',
            'metodo_pago' => 'required|string|max:50',
            'descripcion' => 'nullable|string|max:500'
        ]);

        try {
            DB::beginTransaction();

            // Crear depósito con estado aprobado directamente
            $deposito = Deposito::create([
                'usuario_id' => $request->usuario_id,
                'monto' => $request->monto,
                'metodo_pago' => $request->metodo_pago,
                'referencia' => 'ADMIN-' . strtoupper(uniqid()),
                'estado' => 'pendiente',
                'notas_admin' => $request->descripcion
            ]);

            // Aprobar inmediatamente (admin_id puede ser null)
            $deposito->aprobar(null, 'Depósito manual registrado y aprobado por admin');

            DB::commit();

            // Recargar el usuario para obtener el saldo actualizado
            $usuario = Usuario::find($request->usuario_id);

            return response()->json([
                'success' => true,
                'message' => 'Depósito registrado y aprobado exitosamente',
                'deposito' => $deposito,
                'nuevo_saldo' => $usuario->saldo_disponible
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error al registrar depósito',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Consulta pública de depósitos con totales
     * GET /api/public/depositos
     */
    public function consultarDepositos(Request $request)
    {
        $query = Deposito::with('usuario:id,nombre,email');

        // Filtros opcionales
        if ($request->has('estado')) {
            $query->where('estado', $request->estado);
        }

        if ($request->has('usuario_id')) {
            $query->where('usuario_id', $request->usuario_id);
        }

        if ($request->has('fecha_desde')) {
            $query->whereDate('created_at', '>=', $request->fecha_desde);
        }

        if ($request->has('fecha_hasta')) {
            $query->whereDate('created_at', '<=', $request->fecha_hasta);
        }

        // Obtener lista con paginación
        $depositos = $query->orderBy('created_at', 'desc')
            ->paginate($request->get('per_page', 20));

        // Calcular totales (sin filtros de paginación)
        $queryTotales = clone $query;
        $totales = [
            'total_depositado' => $queryTotales->sum('monto'),
            'total_registros' => $queryTotales->count(),
            'por_estado' => [
                'aprobados' => Deposito::where('estado', 'aprobado')->sum('monto'),
                'pendientes' => Deposito::where('estado', 'pendiente')->sum('monto'),
                'rechazados' => Deposito::where('estado', 'rechazado')->sum('monto'),
            ],
            'conteo_por_estado' => [
                'aprobados' => Deposito::where('estado', 'aprobado')->count(),
                'pendientes' => Deposito::where('estado', 'pendiente')->count(),
                'rechazados' => Deposito::where('estado', 'rechazado')->count(),
            ]
        ];

        return response()->json([
            'success' => true,
            'depositos' => $depositos->items(),
            'totales' => $totales,
            'pagination' => [
                'current_page' => $depositos->currentPage(),
                'last_page' => $depositos->lastPage(),
                'per_page' => $depositos->perPage(),
                'total' => $depositos->total(),
            ]
        ]);
    }

    /**
     * Consulta pública de retiros con totales
     * GET /api/public/retiros
     */
    public function consultarRetiros(Request $request)
    {
        $query = Retiro::with('usuario:id,nombre,email');

        // Filtros opcionales
        if ($request->has('estado')) {
            $query->where('estado', $request->estado);
        }

        if ($request->has('usuario_id')) {
            $query->where('usuario_id', $request->usuario_id);
        }

        if ($request->has('fecha_desde')) {
            $query->whereDate('created_at', '>=', $request->fecha_desde);
        }

        if ($request->has('fecha_hasta')) {
            $query->whereDate('created_at', '<=', $request->fecha_hasta);
        }

        // Obtener lista con paginación
        $retiros = $query->orderBy('created_at', 'desc')
            ->paginate($request->get('per_page', 20));

        // Calcular totales
        $queryTotales = clone $query;
        $totales = [
            'total_retirado' => $queryTotales->sum('monto'),
            'total_registros' => $queryTotales->count(),
            'por_estado' => [
                'completados' => Retiro::where('estado', 'completado')->sum('monto'),
                'pendientes' => Retiro::where('estado', 'pendiente')->sum('monto'),
                'en_proceso' => Retiro::where('estado', 'en_proceso')->sum('monto'),
                'rechazados' => Retiro::where('estado', 'rechazado')->sum('monto'),
            ],
            'conteo_por_estado' => [
                'completados' => Retiro::where('estado', 'completado')->count(),
                'pendientes' => Retiro::where('estado', 'pendiente')->count(),
                'en_proceso' => Retiro::where('estado', 'en_proceso')->count(),
                'rechazados' => Retiro::where('estado', 'rechazado')->count(),
            ]
        ];

        return response()->json([
            'success' => true,
            'retiros' => $retiros->items(),
            'totales' => $totales,
            'pagination' => [
                'current_page' => $retiros->currentPage(),
                'last_page' => $retiros->lastPage(),
                'per_page' => $retiros->perPage(),
                'total' => $retiros->total(),
            ]
        ]);
    }
}
