<?php

namespace App\Http\Controllers;

use App\Models\Usuario;
use App\Models\AuditoriaAdmin;
use App\Models\SolicitudContactoScawatt;
use Illuminate\Http\Request;
use App\Services\ScawattValuationService;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Carbon;

class UsuarioController extends Controller
{
    private const USUARIOS_ALIADOS_PATH = '/home/scalarapp/scalar-data/scawatt-server/produccion-scawatts/data_scawatts/usuarios-aliados.json';
    private const SCAWATTS_ACTIVE_USERS_PATH = '/home/scalarapp/scalar-data/scawatt-server/produccion-scawatts/data_scawatts/scawatts_active_user.json';
    private const SCAWATTS_TOKENS_PATH = '/home/scalarapp/scalar-data/scawatt-server/produccion-scawatts/data_scawatts/raiz-scawatt/scawatts_tokens.json';

    public function __construct(private readonly ScawattValuationService $valuationService)
    {
    }

    /**
     * Listar todos los usuarios
     * GET /api/admin/usuarios
     */
    public function index(Request $request)
    {
        $query = Usuario::query();

        // Filtros opcionales
        if ($request->has('estado')) {
            $query->where('estado', $request->estado);
        }

        if ($request->has('buscar')) {
            $buscar = $request->buscar;
            $query->where(function($q) use ($buscar) {
                $q->where('nombre', 'like', "%{$buscar}%")
                  ->orWhere('email', 'like', "%{$buscar}%");
            });
        }

        $usuarios = $query->with(['scawatts', 'notificaciones', 'mensajesRecibidos'])
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return response()->json($usuarios);
    }

    /**
     * Ver detalle de usuario
     * GET /api/admin/usuarios/{id}
     */
    public function show($id)
    {
        try {
            $usuario = Usuario::with([
                'scawatts',
                'depositos',
                'retiros',
                'notificaciones',
                'mensajesRecibidos',
                'mensajesEnviados',
                'documentos',
                'trazabilidad'
            ])->findOrFail($id);

            // Estadísticas del usuario
            $estadisticas = [
                'total_scawatts' => $usuario->scawatts()->count(),
                'total_depositos' => $usuario->depositos()->sum('monto'),
                'total_retiros' => $usuario->retiros()->where('estado', 'completado')->sum('monto'),
                'notificaciones_no_leidas' => $usuario->notificacionesNoLeidas()->count(),
                'mensajes_no_leidos' => $usuario->mensajesNoLeidos()->count(),
                'valor_total_scawatts' => $usuario->scawatts()->sum('valor_actual')
            ];

            return response()->json([
                'usuario' => $usuario,
                'estadisticas' => $estadisticas
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Usuario no encontrado',
                'error' => $e->getMessage()
            ], 404);
        }
    }

    /**
     * Snapshot integral del usuario combinando base de datos y archivos JSON.
     * GET /api/admin/usuarios/{id}/detalle
     */
    public function detalle($id)
    {
        try {
            $usuario = Usuario::with([
                'scawatts' => function ($query) {
                    $query->with('granja')->orderByDesc('created_at');
                },
                'depositos' => function ($query) {
                    $query->orderByDesc('created_at')->limit(25);
                },
                'retiros' => function ($query) {
                    $query->orderByDesc('created_at')->limit(25);
                },
                'notificaciones' => function ($query) {
                    $query->orderByDesc('created_at')->limit(20);
                },
                'mensajesRecibidos' => function ($query) {
                    $query->orderByDesc('created_at')->limit(20);
                },
                'mensajesEnviados' => function ($query) {
                    $query->orderByDesc('created_at')->limit(20);
                },
            ])->findOrFail($id);

            $solicitudes = SolicitudContactoScawatt::with([
                    'adminAtendio:id,nombre,email',
                    'estadoActualizadoPor:id,nombre,email',
                ])
                ->where('usuario_id', $usuario->id)
                ->orderByDesc('created_at')
                ->limit(20)
                ->get();

            $documentosRegistro = DB::table('documentos')
                ->select('id', 'tipo', 'nombre_archivo', 'ruta_archivo', 'url_publica', 'fecha_generacion', 'created_at')
                ->where('usuario_id', $usuario->id)
                ->orderByDesc('created_at')
                ->limit(40)
                ->get();

            $depositosMontos = DB::table('depositos')
                ->selectRaw("COALESCE(SUM(CASE WHEN estado = 'aprobado' THEN monto ELSE 0 END), 0) AS aprobados")
                ->selectRaw("COALESCE(SUM(CASE WHEN estado = 'pendiente' THEN monto ELSE 0 END), 0) AS pendientes")
                ->selectRaw("COALESCE(SUM(CASE WHEN estado = 'rechazado' THEN monto ELSE 0 END), 0) AS rechazados")
                ->where('usuario_id', $usuario->id)
                ->first();

            $retirosMontos = DB::table('retiros')
                ->selectRaw("COALESCE(SUM(CASE WHEN estado = 'completado' THEN monto ELSE 0 END), 0) AS completados")
                ->selectRaw("COALESCE(SUM(CASE WHEN estado = 'pendiente' THEN monto ELSE 0 END), 0) AS pendientes")
                ->selectRaw("COALESCE(SUM(CASE WHEN estado = 'rechazado' THEN monto ELSE 0 END), 0) AS rechazados")
                ->where('usuario_id', $usuario->id)
                ->first();

            $scawattAggregate = $usuario->scawatts()
                ->selectRaw('COALESCE(SUM(kwh_asignados), 0) AS kwh_total, COALESCE(SUM(valor_actual), 0) AS valor_actual_total, COALESCE(SUM(valor_inicial), 0) AS valor_inicial_total')
                ->first();

            $resumenFinanciero = [
                'saldo_disponible' => (float) ($usuario->saldo_disponible ?? 0),
                'saldo_pendiente' => (float) ($usuario->saldo_pendiente ?? 0),
                'saldo_total' => (float) ($usuario->saldo_total ?? 0),
                'depositos_aprobados' => (float) ($depositosMontos->aprobados ?? 0),
                'depositos_pendientes' => (float) ($depositosMontos->pendientes ?? 0),
                'depositos_rechazados' => (float) ($depositosMontos->rechazados ?? 0),
                'retiros_completados' => (float) ($retirosMontos->completados ?? 0),
                'retiros_pendientes' => (float) ($retirosMontos->pendientes ?? 0),
                'retiros_rechazados' => (float) ($retirosMontos->rechazados ?? 0),
                'valor_portafolio' => (float) ($scawattAggregate->valor_actual_total ?? 0),
                'valor_inicial_portafolio' => (float) ($scawattAggregate->valor_inicial_total ?? 0),
                'valorizacion_portafolio' => (float) (($scawattAggregate->valor_actual_total ?? 0) - ($scawattAggregate->valor_inicial_total ?? 0)),
            ];

            $estadisticasScawatt = [
                'total' => $usuario->scawatts->count(),
                'activos' => $usuario->scawatts->where('estado', 'activo')->count(),
                'inactivos' => $usuario->scawatts->where('estado', '!=', 'activo')->count(),
                'kwh_total' => (float) ($scawattAggregate->kwh_total ?? 0),
                'valor_actual_total' => (float) ($scawattAggregate->valor_actual_total ?? 0),
                'valor_inicial_total' => (float) ($scawattAggregate->valor_inicial_total ?? 0),
                'valorizacion_total' => (float) (($scawattAggregate->valor_actual_total ?? 0) - ($scawattAggregate->valor_inicial_total ?? 0)),
            ];

            $scawattsListado = $usuario->scawatts->map(function ($scawatt) {
                return [
                    'id' => $scawatt->id,
                    'serial' => $scawatt->scawatt_id,
                    'estado' => $scawatt->estado,
                    'valor_inicial' => (float) ($scawatt->valor_inicial ?? 0),
                    'valor_actual' => (float) ($scawatt->valor_actual ?? 0),
                    'valorizacion_total' => (float) ($scawatt->valorizacion_total ?? 0),
                    'kwh_asignados' => (float) ($scawatt->kwh_asignados ?? 0),
                    'fecha_inicio' => $scawatt->fecha_inicio ? $scawatt->fecha_inicio->toDateString() : null,
                    'fecha_final' => $scawatt->fecha_final ? $scawatt->fecha_final->toDateString() : null,
                    'granja' => $scawatt->granja ? [
                        'id' => $scawatt->granja->id,
                        'nombre' => $scawatt->granja->nombre,
                        'ubicacion' => $scawatt->granja->ubicacion ?? null,
                    ] : null,
                ];
            })->values()->toArray();

            $depositosRecientes = $usuario->depositos->map(function ($deposito) {
                return [
                    'id' => $deposito->id,
                    'monto' => (float) ($deposito->monto ?? 0),
                    'estado' => $deposito->estado,
                    'metodo_pago' => $deposito->metodo_pago,
                    'referencia' => $deposito->referencia,
                    'referencia_pago' => $deposito->referencia_pago,
                    'fecha_aprobacion' => $this->toIsoString($deposito->fecha_aprobacion),
                    'creado_en' => $this->toIsoString($deposito->created_at),
                    'actualizado_en' => $this->toIsoString($deposito->updated_at),
                    'comprobante' => [
                        'ruta' => $deposito->comprobante_ruta,
                        'url_publica' => $deposito->comprobante_url,
                        'url_resuelta' => $this->resolvePublicUrl($deposito->comprobante_url ?: $deposito->comprobante_ruta),
                    ],
                    'datos_pago' => $deposito->datos_pago,
                ];
            })->values()->toArray();

            $retirosRecientes = $usuario->retiros->map(function ($retiro) {
                return [
                    'id' => $retiro->id,
                    'monto' => (float) ($retiro->monto ?? 0),
                    'estado' => $retiro->estado,
                    'metodo' => $retiro->metodo_retiro,
                    'referencia_transaccion' => $retiro->referencia_transaccion,
                    'fecha_solicitud' => $this->toIsoString($retiro->fecha_solicitud),
                    'fecha_procesado' => $this->toIsoString($retiro->fecha_procesado),
                    'creado_en' => $this->toIsoString($retiro->created_at),
                    'actualizado_en' => $this->toIsoString($retiro->updated_at),
                    'datos_bancarios' => $retiro->datos_bancarios,
                    'notas_admin' => $retiro->notas_admin,
                    'notas_usuario' => $retiro->notas_usuario,
                ];
            })->values()->toArray();

            $solicitudesRecientes = $solicitudes->map(function ($solicitud) {
                return [
                    'id' => $solicitud->id,
                    'estado' => $solicitud->estado,
                    'cantidad_scawatts' => $solicitud->cantidad_scawatts,
                    'presupuesto_aproximado' => $solicitud->presupuesto_aproximado !== null ? (float) $solicitud->presupuesto_aproximado : null,
                    'mensaje' => $solicitud->mensaje,
                    'link_pago' => $solicitud->link_pago,
                    'monto_pago' => $solicitud->monto_pago !== null ? (float) $solicitud->monto_pago : null,
                    'fecha_contacto' => $this->toIsoString($solicitud->fecha_contacto),
                    'fecha_pago' => $this->toIsoString($solicitud->fecha_pago),
                    'created_at' => $this->toIsoString($solicitud->created_at),
                    'updated_at' => $this->toIsoString($solicitud->updated_at),
                    'admin_atendio' => $solicitud->adminAtendio ? [
                        'id' => $solicitud->adminAtendio->id,
                        'nombre' => $solicitud->adminAtendio->nombre,
                        'email' => $solicitud->adminAtendio->email,
                    ] : null,
                    'estado_actualizado_por' => $solicitud->estadoActualizadoPor ? [
                        'id' => $solicitud->estadoActualizadoPor->id,
                        'nombre' => $solicitud->estadoActualizadoPor->nombre,
                        'email' => $solicitud->estadoActualizadoPor->email,
                    ] : null,
                    'notas_admin' => $solicitud->notas_admin,
                    'historial_eventos' => $solicitud->historial_eventos,
                ];
            })->values()->toArray();

            $notificacionesRecientes = $usuario->notificaciones->map(function ($notificacion) {
                return [
                    'id' => $notificacion->id,
                    'titulo' => $notificacion->titulo,
                    'mensaje' => $notificacion->mensaje,
                    'tipo' => $notificacion->tipo,
                    'leido' => (bool) $notificacion->leido,
                    'icono' => $notificacion->icono,
                    'prioridad' => $notificacion->prioridad,
                    'url_accion' => $notificacion->url_accion,
                    'datos_adicionales' => $notificacion->datos_adicionales,
                    'fecha_leido' => $this->toIsoString($notificacion->fecha_leido),
                    'created_at' => $this->toIsoString($notificacion->created_at),
                    'updated_at' => $this->toIsoString($notificacion->updated_at),
                ];
            })->values()->toArray();

            $mensajesRecibidos = $usuario->mensajesRecibidos->map(function ($mensaje) {
                return [
                    'id' => $mensaje->id,
                    'titulo' => $mensaje->titulo,
                    'contenido' => $mensaje->contenido,
                    'tipo' => $mensaje->tipo,
                    'destinatario' => $mensaje->destinatario,
                    'enviado' => (bool) $mensaje->enviado,
                    'fecha_programada' => $this->toIsoString($mensaje->fecha_programada),
                    'fecha_enviado' => $this->toIsoString($mensaje->fecha_enviado),
                    'creado_en' => $this->toIsoString($mensaje->created_at),
                    'actualizado_en' => $this->toIsoString($mensaje->updated_at),
                ];
            })->values()->toArray();

            $mensajesEnviados = $usuario->mensajesEnviados->map(function ($mensaje) {
                return [
                    'id' => $mensaje->id,
                    'titulo' => $mensaje->titulo,
                    'contenido' => $mensaje->contenido,
                    'tipo' => $mensaje->tipo,
                    'destinatario' => $mensaje->destinatario,
                    'enviado' => (bool) $mensaje->enviado,
                    'fecha_programada' => $this->toIsoString($mensaje->fecha_programada),
                    'fecha_enviado' => $this->toIsoString($mensaje->fecha_enviado),
                    'creado_en' => $this->toIsoString($mensaje->created_at),
                    'actualizado_en' => $this->toIsoString($mensaje->updated_at),
                ];
            })->values()->toArray();

            $documentos = $documentosRegistro->map(function ($doc) {
                $urlBase = $doc->url_publica ?: $doc->ruta_archivo;

                return [
                    'id' => $doc->id,
                    'tipo' => $doc->tipo,
                    'nombre' => $doc->nombre_archivo,
                    'ruta' => $doc->ruta_archivo,
                    'url_publica' => $doc->url_publica,
                    'url_resuelta' => $this->resolvePublicUrl($urlBase),
                    'fecha_generacion' => $this->toIsoString($doc->fecha_generacion),
                    'registrado_en' => $this->toIsoString($doc->created_at),
                ];
            })->values()->toArray();

            $timeline = collect();

            foreach ($usuario->depositos as $deposito) {
                $timeline->push([
                    'tipo' => 'deposito',
                    'estado' => $deposito->estado,
                    'monto' => (float) ($deposito->monto ?? 0),
                    'referencia' => $deposito->referencia,
                    'descripcion' => $deposito->metodo_pago,
                    'fecha' => $this->toIsoString($deposito->created_at),
                ]);
            }

            foreach ($usuario->retiros as $retiro) {
                $timeline->push([
                    'tipo' => 'retiro',
                    'estado' => $retiro->estado,
                    'monto' => (float) ($retiro->monto ?? 0),
                    'referencia' => $retiro->referencia_transaccion,
                    'descripcion' => $retiro->metodo_retiro,
                    'fecha' => $this->toIsoString($retiro->created_at),
                ]);
            }

            foreach ($solicitudes as $solicitud) {
                $timeline->push([
                    'tipo' => 'solicitud',
                    'estado' => $solicitud->estado,
                    'presupuesto' => $solicitud->presupuesto_aproximado !== null ? (float) $solicitud->presupuesto_aproximado : null,
                    'descripcion' => $solicitud->mensaje,
                    'fecha' => $this->toIsoString($solicitud->created_at),
                ]);
            }

            foreach ($documentosRegistro as $doc) {
                $timeline->push([
                    'tipo' => 'documento',
                    'estado' => 'registrado',
                    'descripcion' => $doc->nombre_archivo,
                    'fecha' => $this->toIsoString($doc->created_at),
                ]);
            }

            $timeline = $timeline
                ->filter(fn ($evento) => !empty($evento['fecha']))
                ->sortByDesc('fecha')
                ->take(30)
                ->values()
                ->toArray();

            $jsonErrors = [];
            $usuariosAliadosRegistro = $this->obtenerRegistroUsuariosAliados($usuario, $jsonErrors);
            $scawattsActivosRegistro = $this->obtenerRegistroScawattsActivos($usuario, $jsonErrors);
            $tokens = $this->obtenerTokensPorUsuario($usuario, $jsonErrors);
            $jsonErrors = array_values(array_unique($jsonErrors));

            return response()->json([
                'success' => true,
                'meta' => [
                    'generated_at' => now()->toIso8601String(),
                    'usuario_id' => $usuario->id,
                    'id_wordpress' => $usuario->id_wordpress,
                ],
                'database' => [
                    'perfil' => [
                        'id' => $usuario->id,
                        'id_wordpress' => $usuario->id_wordpress,
                        'nombre' => $usuario->nombre,
                        'email' => $usuario->email,
                        'telefono' => $usuario->telefono,
                        'estado' => $usuario->estado,
                        'cedula' => $usuario->cedula,
                        'created_at' => $this->toIsoString($usuario->created_at),
                        'updated_at' => $this->toIsoString($usuario->updated_at),
                    ],
                    'verificaciones' => [
                        'cedula_verificada' => (bool) $usuario->cedula_verificada_at,
                        'cedula_verificada_at' => $this->toIsoString($usuario->cedula_verificada_at),
                        'datos_bancarios_verificados' => (bool) $usuario->datos_bancarios_verificados_at,
                        'datos_bancarios_verificados_at' => $this->toIsoString($usuario->datos_bancarios_verificados_at),
                    ],
                    'finanzas' => $resumenFinanciero,
                    'scawatts' => [
                        'estadisticas' => $estadisticasScawatt,
                        'listado' => $scawattsListado,
                    ],
                    'movimientos' => [
                        'depositos' => $depositosRecientes,
                        'retiros' => $retirosRecientes,
                    ],
                    'solicitudes' => $solicitudesRecientes,
                    'notificaciones' => $notificacionesRecientes,
                    'mensajes' => [
                        'recibidos' => $mensajesRecibidos,
                        'enviados' => $mensajesEnviados,
                    ],
                    'documentos' => $documentos,
                    'timeline' => $timeline,
                ],
                'json_sources' => [
                    'usuarios_aliados' => [
                        'path' => self::USUARIOS_ALIADOS_PATH,
                        'registro' => $usuariosAliadosRegistro,
                    ],
                    'scawatts_active_user' => [
                        'path' => self::SCAWATTS_ACTIVE_USERS_PATH,
                        'registro' => $scawattsActivosRegistro,
                    ],
                    'scawatts_tokens' => [
                        'path' => self::SCAWATTS_TOKENS_PATH,
                        'total' => count($tokens),
                        'registros' => $tokens,
                    ],
                    'errores' => $jsonErrors,
                ],
            ]);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Usuario no encontrado',
            ], 404);
        } catch (\Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al generar detalle',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Crear nuevo usuario
     * POST /api/admin/usuarios
     * 
     * Lógica de contraseña:
     * - Si se proporciona password, se usa ese
     * - Si no hay password pero sí cédula, usa la cédula como password
     * - Si no hay ni password ni cédula, genera password aleatoria
     */
    public function store(Request $request)
    {
        $request->validate([
            'nombre' => 'required|string|max:255',
            'email' => 'required|email|unique:usuarios,email',
            'password' => 'nullable|string|min:6',
            'telefono' => 'nullable|string|max:20',
            'cedula' => 'nullable|string|max:20',
            'id_wordpress' => 'nullable|integer|unique:usuarios,id_wordpress',
            'estado' => 'nullable|in:activo,inactivo,bloqueado'
        ]);

        try {
            DB::beginTransaction();

            // Determinar la contraseña a usar
            $passwordGenerada = null;
            $passwordTextoPlano = null;

            if ($request->filled('password')) {
                // Si el admin proporciona una contraseña, usarla
                $passwordTextoPlano = $request->password;
            } elseif ($request->filled('cedula')) {
                // Si no hay password pero sí cédula, usar la cédula como password
                $passwordTextoPlano = $request->cedula;
                $passwordGenerada = $request->cedula;
            } else {
                // Si no hay ni password ni cédula, generar una aleatoria
                $passwordTextoPlano = $this->generarPasswordAleatoria();
                $passwordGenerada = $passwordTextoPlano;
            }

            $usuario = Usuario::create([
                'nombre' => $request->nombre,
                'email' => $request->email,
                'password' => \Hash::make($passwordTextoPlano),
                'telefono' => $request->telefono,
                'cedula' => $request->cedula,
                'id_wordpress' => $request->id_wordpress,
                'saldo_disponible' => 0,
                'saldo_pendiente' => 0,
                'saldo_total' => 0,
                'estado' => $request->estado ?? 'activo'
            ]);

            // Si se generó una contraseña, enviar notificación
            if ($passwordGenerada) {
                $this->enviarPasswordPorEmail($usuario, $passwordGenerada);
            }

            // Registrar auditoría
            if ($request->has('admin_id')) {
                AuditoriaAdmin::registrar(
                    $request->admin_id,
                    'crear',
                    'usuarios',
                    "Usuario creado: {$usuario->nombre}" . ($passwordGenerada ? " - Password generada" : ""),
                    'Usuario',
                    $usuario->id,
                    null,
                    $usuario->toArray()
                );
            }

            DB::commit();

            $responseData = [
                'success' => true,
                'message' => 'Usuario creado exitosamente',
                'usuario' => $usuario
            ];

            // Incluir password generada en la respuesta para que el admin la vea
            if ($passwordGenerada) {
                $responseData['password_generada'] = $passwordGenerada;
                $responseData['mensaje_password'] = 'Se ha enviado una notificación al usuario con su contraseña';
            }

            return response()->json($responseData, 201);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error al crear usuario',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Actualizar usuario
     * PUT /api/admin/usuarios/{id}
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'nombre' => 'sometimes|string|max:255',
            'email' => 'sometimes|email|unique:usuarios,email,' . $id,
            'telefono' => 'nullable|string|max:20',
            'estado' => 'sometimes|in:activo,inactivo,bloqueado'
        ]);

        try {
            $usuario = Usuario::findOrFail($id);
            $datosAnteriores = $usuario->toArray();

            DB::beginTransaction();

            $usuario->update($request->all());

            // Registrar auditoría
            if ($request->has('admin_id')) {
                AuditoriaAdmin::registrar(
                    $request->admin_id,
                    'actualizar',
                    'usuarios',
                    "Usuario actualizado: {$usuario->nombre}",
                    'Usuario',
                    $usuario->id,
                    $datosAnteriores,
                    $usuario->toArray()
                );
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Usuario actualizado exitosamente',
                'usuario' => $usuario
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error al actualizar usuario',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Cambiar estado de usuario
     * PUT /api/admin/usuarios/{id}/estado
     */
    public function cambiarEstado(Request $request, $id)
    {
        $request->validate([
            'estado' => 'required|in:activo,inactivo,bloqueado',
            'admin_id' => 'required|exists:usuarios,id',
            'motivo' => 'nullable|string'
        ]);

        try {
            $usuario = Usuario::findOrFail($id);
            $estadoAnterior = $usuario->estado;

            DB::beginTransaction();

            $usuario->update(['estado' => $request->estado]);

            // Registrar auditoría
            AuditoriaAdmin::registrar(
                $request->admin_id,
                'cambiar_estado_usuario',
                'usuarios',
                "Estado cambiado de '{$estadoAnterior}' a '{$request->estado}'. Motivo: " . ($request->motivo ?? 'No especificado'),
                'Usuario',
                $usuario->id
            );

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Estado de usuario actualizado',
                'usuario' => $usuario
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error al cambiar estado',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Dashboard del usuario (vista completa)
     * GET /api/usuarios/{id}/dashboard
     */
    public function dashboard($id)
    {
        try {
            $usuario = Usuario::with([
                'scawatts' => function($q) {
                    $q->where('estado', 'activo');
                },
                'depositos' => function($q) {
                    $q->orderBy('created_at', 'desc')->limit(10);
                },
                'retiros' => function($q) {
                    $q->orderBy('created_at', 'desc')->limit(10);
                }
            ])->findOrFail($id);

            $realtimePortfolio = $this->valuationService->buildRealtimePortfolio($usuario);
            $realtimeScawatts = collect($realtimePortfolio['scawatts'] ?? [])->keyBy('scawatt_id');

            $scawattListado = $usuario->scawatts->map(function ($scawatt) use ($realtimeScawatts) {
                $payload = $scawatt->toArray();
                $realtime = $realtimeScawatts->get($scawatt->scawatt_id);

                if ($realtime) {
                    $payload['valor_inicial'] = $this->formatDecimalString((float) $realtime['valor_inicial']);
                    $payload['valor_actual'] = $this->formatDecimalString((float) $realtime['valor_actual']);
                    $payload['valorizaciones'] = [$realtime['valorizacion']];
                    $payload['valorizacion_total'] = (float) $payload['valor_actual'] - (float) $payload['valor_inicial'];
                }

                return $payload;
            })->values();

            $scawattValorTotal = $scawattListado->sum(fn ($item) => (float) $item['valor_actual']);

            // DEVOLVER DATOS DIRECTOS SIN WRAPPER (formato original)
            $dashboard = [
                'usuario' => [
                    'id' => $usuario->id,
                    'nombre' => $usuario->nombre,
                    'email' => $usuario->email,
                    'estado' => $usuario->estado,
                    'saldo_disponible' => $usuario->saldo_disponible,
                    'saldo_total' => $usuario->saldo_total
                ],
                'scawatts' => [
                    'total' => $scawattListado->count(),
                    'valor_total' => $scawattValorTotal,
                    'listado' => $scawattListado
                ],
                'transacciones' => [
                    'depositos_total' => $usuario->depositos()->where('estado', 'aprobado')->sum('monto'),
                    'retiros_total' => $usuario->retiros()->where('estado', 'completado')->sum('monto'),
                    'ultimos_depositos' => $usuario->depositos,
                    'ultimos_retiros' => $usuario->retiros
                ],
                'notificaciones_no_leidas' => $usuario->notificacionesNoLeidas()->count(),
                'mensajes_no_leidos' => 0  // Tabla mensajes no tiene estado de lectura por usuario
            ];

            return response()->json($dashboard);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al cargar dashboard',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Obtener perfil general del usuario
     * GET /api/usuarios/{id}/perfil
     */
    public function obtenerPerfil($id)
    {
        try {
            $usuario = Usuario::findOrFail($id);

            return response()->json([
                'success' => true,
                'perfil' => $this->formatearPerfilUsuario($usuario)
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Usuario no encontrado',
                'error' => $e->getMessage()
            ], 404);
        }
    }

    /**
     * Actualizar perfil general del usuario
     * PUT /api/usuarios/{id}/perfil
     */
    public function actualizarPerfil(Request $request, $id)
    {
        $validated = $request->validate([
            'nombre' => 'sometimes|string|max:255',
            'email' => 'sometimes|email|unique:usuarios,email,' . $id,
            'telefono' => 'sometimes|nullable|string|max:20',
            'picture' => 'sometimes|nullable|string|max:500'
        ]);

        try {
            $usuario = Usuario::findOrFail($id);

            $usuario->fill($validated);
            $usuario->save();

            return response()->json([
                'success' => true,
                'message' => 'Perfil actualizado correctamente',
                'perfil' => $this->formatearPerfilUsuario($usuario->fresh())
            ]);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Usuario no encontrado'
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al actualizar el perfil',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Subir avatar del usuario (almacena archivo y devuelve URL pública)
     * POST /api/usuarios/{id}/avatar/upload
     */
    public function subirAvatar(Request $request, $id)
    {
        $request->validate([
            'avatar' => 'required|image|mimes:jpg,jpeg,png,heic,webp|max:10240'
        ]);

        try {
            $usuario = Usuario::findOrFail($id);

            $archivo = $request->file('avatar');
            $directorio = "avatars/user-{$usuario->id}";

            // Guardamos la ruta anterior para intentar limpiarla si pertenece a nuestro storage
            $rutaAnterior = $usuario->picture;

            // Aseguramos la existencia del directorio destino
            Storage::disk('public')->makeDirectory($directorio);

            $path = $archivo->store($directorio, 'public');
            $url = Storage::disk('public')->url($path);

            // Persistimos la nueva URL del avatar
            $usuario->picture = $url;
            $usuario->save();

            // Intentamos eliminar el archivo anterior si estaba dentro del storage público
            if ($rutaAnterior && str_contains($rutaAnterior, '/storage/')) {
                $relativePath = ltrim(str_replace(url('/storage'), '', $rutaAnterior), '/');
                if ($relativePath && Storage::disk('public')->exists($relativePath)) {
                    Storage::disk('public')->delete($relativePath);
                }
            }

            return response()->json([
                'success' => true,
                'message' => 'Avatar subido correctamente',
                'data' => [
                    'url' => $url,
                    'path' => $path,
                    'user' => $this->formatearPerfilUsuario($usuario->fresh()),
                ]
            ], 201);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Usuario no encontrado'
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al subir el avatar',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Registrar token FCM para notificaciones push
     * POST /api/usuarios/{id}/register-token
     */
    public function registrarTokenFCM(Request $request, $id)
    {
        $request->validate([
            'fcm_token' => 'required|string|max:255'
        ]);

        try {
            $usuario = Usuario::findOrFail($id);
            
            $usuario->fcm_token = $request->fcm_token;
            $usuario->fcm_token_updated_at = now();
            $usuario->save();

            return response()->json([
                'success' => true,
                'message' => 'Token FCM registrado exitosamente'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al registrar token',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Eliminar token FCM (logout)
     * DELETE /api/usuarios/{id}/unregister-token
     */
    public function eliminarTokenFCM($id)
    {
        try {
            $usuario = Usuario::findOrFail($id);
            
            $usuario->fcm_token = null;
            $usuario->fcm_token_updated_at = null;
            $usuario->save();

            return response()->json([
                'success' => true,
                'message' => 'Token FCM eliminado'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al eliminar token',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Obtener todos los movimientos (depósitos, retiros, compras y ventas) de un usuario
     * GET /api/usuarios/{id}/movimientos
     * 
     * ESPECIFICACIÓN:
     * - tipo: "ingreso" (compras ScaWatt, depósitos, ventas) | "egreso" (retiros)
     * - monto: SIEMPRE POSITIVO
     * - Ordenado por fecha descendente
     */
    public function movimientos($id)
    {
        try {
            $usuario = Usuario::findOrFail($id);
            
            // INGRESOS: Depósitos/Abonos
            $depositos = $usuario->depositos()
                ->where('estado', 'aprobado')
                ->get()
                ->map(function($deposito) {
                    return [
                        'id' => $deposito->id,
                        'tipo' => 'ingreso',
                        'concepto' => 'Abono',
                        'descripcion' => $deposito->metodo_pago ?? 'Abono a billetera',
                        'monto' => abs((float) $deposito->monto),  // SIEMPRE POSITIVO
                        'fecha' => $deposito->created_at->format('Y-m-d H:i:s'),
                        'scawatt_id' => null,
                        'detalle' => $deposito->metodo_pago,
                        'estado' => 'completado'
                    ];
                });
            
            // EGRESOS: Retiros
            $retiros = $usuario->retiros()
                ->where('estado', 'completado')
                ->get()
                ->map(function($retiro) {
                    return [
                        'id' => $retiro->id,
                        'tipo' => 'egreso',
                        'concepto' => 'Retiro',
                        'descripcion' => 'Retiro a cuenta bancaria',
                        'monto' => abs((float) $retiro->monto),  // SIEMPRE POSITIVO
                        'fecha' => $retiro->created_at->format('Y-m-d H:i:s'),
                        'scawatt_id' => null,
                        'detalle' => 'Retiro a cuenta bancaria',
                        'estado' => 'completado'
                    ];
                });
            
            // Compras de ScaWatts NO afectan el saldo de billetera
            // Se muestran solo como registro informativo
            $compras_list = \DB::table('marketplace')
                ->join('scawatts', 'marketplace.scawatt_id', '=', 'scawatts.id')
                ->where('marketplace.comprador_id', $id)
                ->where('marketplace.estado', 'vendido')
                ->select('marketplace.*', 'scawatts.scawatt_id as scawatt_serial')
                ->orderBy('marketplace.fecha_venta', 'ASC')
                ->get();
            
            $primera_compra_id = $compras_list->first()->id ?? null;
            
            $compras = $compras_list->map(function($compra) use ($primera_compra_id) {
                $es_primera = ($compra->id === $primera_compra_id);
                return [
                    'id' => $compra->id,
                    'tipo' => 'egreso',  // EGRESO: el dinero sale de la billetera al comprar
                    'concepto' => 'Compra ScaWatt',
                    'descripcion' => $es_primera ? 'Compra de tu primer ScaWatt' : 'Compra de ScaWatt',
                    'monto' => abs((float) $compra->precio_venta),  // SIEMPRE POSITIVO
                    'fecha' => $compra->fecha_venta,
                    'scawatt_id' => $compra->scawatt_serial,
                    'detalle' => 'ScaWatt ' . $compra->scawatt_serial,
                    'estado' => 'completado',
                    'es_primera_compra' => $es_primera,
                    'afecta_saldo' => false  // NO afecta saldo en el cálculo (ya está descontado)
                ];
            });
            
            // Ventas de ScaWatts en el marketplace (NO afectan saldo inmediatamente)
            // El dinero se acredita en el "Cierre Mensual" (depósito automático)
            $ventas = \DB::table('marketplace')
                ->join('scawatts', 'marketplace.scawatt_id', '=', 'scawatts.id')
                ->where('marketplace.vendedor_id', $id)
                ->where('marketplace.estado', 'vendido')
                ->select('marketplace.*', 'scawatts.scawatt_id as scawatt_serial')
                ->get()
                ->map(function($venta) {
                    $monto_neto = (float) $venta->precio_venta - (float) $venta->comision_plataforma;
                    return [
                        'id' => $venta->id,
                        'tipo' => 'ingreso',
                        'concepto' => 'Ganancia',
                        'descripcion' => 'Venta de ScaWatt (Se acreditará en Cierre Mensual)',
                        'monto' => abs($monto_neto),  // SIEMPRE POSITIVO
                        'fecha' => $venta->fecha_venta,
                        'scawatt_id' => $venta->scawatt_serial,
                        'detalle' => 'Venta de ScaWatt ' . $venta->scawatt_serial,
                        'estado' => 'pendiente_liquidacion',
                        'es_primera_compra' => false,
                        'afecta_saldo' => false  // NO afecta saldo (se paga en Cierre Mensual)
                    ];
                });
            
            // Combinar todos los movimientos
            $movimientos = collect()
                ->merge($depositos)
                ->merge($retiros)
                ->merge($compras)
                ->merge($ventas)
                ->sortByDesc('fecha')
                ->values();
            
            // Calcular resumen financiero
            // INCLUIR TODOS LOS MOVIMIENTOS en el historial (incluyendo compras y ventas)
            $ingresos_total = $movimientos->where('tipo', 'ingreso')->sum('monto');
            $egresos_total = $movimientos->where('tipo', 'egreso')->sum('monto');
            
            // El saldo_actual es el saldo_disponible real del usuario (fuente de verdad)
            $saldo_actual = $usuario->saldo_disponible;            return response()->json([
                'resumen' => [
                    'ingresos_total' => $ingresos_total,
                    'egresos_total' => $egresos_total,
                    'saldo_actual' => $saldo_actual
                ],
                'movimientos' => $movimientos
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al obtener movimientos',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Resumen de retiros del usuario
     * GET /api/usuarios/{id}/retiros/resumen
     */
    public function resumenRetiros($id)
    {
        try {
            $usuario = Usuario::findOrFail($id);
            
            $retiros = $usuario->retiros();
            
            $resumen = [
                'totales' => [
                    'total_retirado' => $retiros->where('estado', 'completado')->sum('monto'),
                    'total_pendiente' => $retiros->where('estado', 'pendiente')->sum('monto'),
                    'total_rechazado' => $retiros->where('estado', 'rechazado')->sum('monto')
                ],
                'cantidad' => [
                    'completados' => $retiros->where('estado', 'completado')->count(),
                    'pendientes' => $retiros->where('estado', 'pendiente')->count(),
                    'rechazados' => $retiros->where('estado', 'rechazado')->count()
                ],
                'ultimo_retiro' => $retiros->where('estado', 'completado')
                    ->orderBy('created_at', 'desc')
                    ->first()
            ];
            
            return response()->json($resumen);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al obtener resumen de retiros',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Exportar historial de movimientos a PDF
     * GET /api/usuarios/{id}/movimientos/pdf
     */
    public function movimientosPDF($id)
    {
        try {
            $usuario = Usuario::findOrFail($id);
            
            // Obtener los mismos datos que el endpoint movimientos()
            $response = $this->movimientos($id);
            $data = json_decode($response->getContent(), true);
            
            // Generar HTML para el PDF
            $html = $this->generarHTMLMovimientos($usuario, $data);
            
            // Crear PDF con HTML y CSS
            return response($html)
                ->header('Content-Type', 'application/pdf')
                ->header('Content-Disposition', 'attachment; filename="movimientos_' . $usuario->id . '_' . date('Y-m-d') . '.pdf"');
                
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al generar PDF',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    private function readJson(string $path, array &$errors = []): array
    {
        if (!is_file($path)) {
            $errors[] = "Archivo no encontrado: {$path}";
            return [];
        }

        try {
            $contents = file_get_contents($path);
            if ($contents === false) {
                $errors[] = "No se pudo leer: {$path}";
                return [];
            }

            $decoded = json_decode($contents, true);
            if (json_last_error() !== JSON_ERROR_NONE || !is_array($decoded)) {
                $errors[] = "Contenido JSON inválido en: {$path}";
                return [];
            }

            return $decoded;
        } catch (\Throwable $e) {
            $errors[] = "Excepción leyendo {$path}: " . $e->getMessage();
            return [];
        }
    }

    private function obtenerRegistroUsuariosAliados(Usuario $usuario, array &$errors): ?array
    {
        $data = $this->readJson(self::USUARIOS_ALIADOS_PATH, $errors);
        if (empty($data)) {
            return null;
        }

        $key = (string) $usuario->id;
        $registro = $data[$key] ?? null;
        if (is_array($registro)) {
            return $registro;
        }

        foreach ($data as $entry) {
            if (!is_array($entry)) {
                continue;
            }

            if ((int) ($entry['id_laravel'] ?? 0) === (int) $usuario->id) {
                return $entry;
            }

            if ($usuario->id_wordpress && (int) ($entry['id_wordpress'] ?? 0) === (int) $usuario->id_wordpress) {
                return $entry;
            }
        }

        return null;
    }

    private function obtenerRegistroScawattsActivos(Usuario $usuario, array &$errors): ?array
    {
        $data = $this->readJson(self::SCAWATTS_ACTIVE_USERS_PATH, $errors);
        if (empty($data)) {
            return null;
        }

        if ($usuario->id_wordpress) {
            $key = (string) $usuario->id_wordpress;
            if (isset($data[$key]) && is_array($data[$key])) {
                return $data[$key];
            }
        }

        foreach ($data as $entry) {
            if (!is_array($entry)) {
                continue;
            }

            if ((int) ($entry['id_laravel'] ?? 0) === (int) $usuario->id) {
                return $entry;
            }
        }

        return null;
    }

    private function obtenerTokensPorUsuario(Usuario $usuario, array &$errors): array
    {
        $data = $this->readJson(self::SCAWATTS_TOKENS_PATH, $errors);
        if (empty($data)) {
            return [];
        }

        $tokens = [];

        foreach ($data as $serial => $payload) {
            if (!is_array($payload)) {
                continue;
            }

            $laravelId = (int) ($payload['id_laravel'] ?? 0);
            $wordpressId = (int) ($payload['id_wordpress'] ?? 0);

            $coincideLaravel = $laravelId === (int) $usuario->id;
            $coincideWordpress = $usuario->id_wordpress && $wordpressId === (int) $usuario->id_wordpress;

            if (!$coincideLaravel && !$coincideWordpress) {
                continue;
            }

            if (!isset($payload['scawatt_serial']) && is_string($serial)) {
                $payload['scawatt_serial'] = $serial;
            }

            $tokens[] = $payload;
        }

        return $tokens;
    }

    private function resolvePublicUrl(?string $path): ?string
    {
        if (empty($path)) {
            return null;
        }

        if (filter_var($path, FILTER_VALIDATE_URL)) {
            return $path;
        }

        if (Storage::disk('public')->exists($path)) {
            return Storage::disk('public')->url($path);
        }

        if (Storage::exists($path)) {
            return Storage::url($path);
        }

        return null;
    }

    private function toIsoString($value): ?string
    {
        if ($value === null || $value === '') {
            return null;
        }

        if ($value instanceof Carbon) {
            return $value->toIso8601String();
        }

        if ($value instanceof \DateTimeInterface) {
            return Carbon::instance($value)->toIso8601String();
        }

        try {
            return Carbon::parse($value)->toIso8601String();
        } catch (\Throwable $e) {
            return null;
        }
    }
    
    /**
     * Generar HTML formateado para el PDF de movimientos
     */
    private function generarHTMLMovimientos($usuario, $data)
    {
        $resumen = $data['resumen'];
        $movimientos = $data['movimientos'];
        
        $html = '
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Historial de Movimientos - ScaWatts</title>
    <style>
        body { font-family: Arial, sans-serif; font-size: 12px; margin: 20px; }
        .header { text-align: center; margin-bottom: 30px; border-bottom: 2px solid #333; padding-bottom: 10px; }
        .header h1 { margin: 0; color: #2c3e50; }
        .header p { margin: 5px 0; color: #7f8c8d; }
        .resumen { background: #f8f9fa; padding: 15px; margin-bottom: 20px; border-radius: 5px; }
        .resumen table { width: 100%; }
        .resumen td { padding: 8px; }
        .resumen .label { font-weight: bold; width: 150px; }
        .movimientos { margin-top: 20px; }
        .movimientos table { width: 100%; border-collapse: collapse; }
        .movimientos th { background: #34495e; color: white; padding: 10px; text-align: left; }
        .movimientos td { padding: 8px; border-bottom: 1px solid #ddd; }
        .movimientos tr:hover { background: #f5f5f5; }
        .ingreso { color: #27ae60; font-weight: bold; }
        .egreso { color: #e74c3c; font-weight: bold; }
        .footer { margin-top: 30px; text-align: center; font-size: 10px; color: #95a5a6; }
        .total-row { font-weight: bold; background: #ecf0f1; }
    </style>
</head>
<body>
    <div class="header">
        <h1>Historial de Movimientos</h1>
        <p><strong>' . htmlspecialchars($usuario->nombre) . '</strong></p>
        <p>Generado: ' . date('d/m/Y H:i:s') . '</p>
    </div>
    
    <div class="resumen">
        <h2>Resumen Financiero</h2>
        <table>
            <tr>
                <td class="label">Total Ingresos:</td>
                <td class="ingreso">$' . number_format($resumen['ingresos_total'], 0, ',', '.') . '</td>
            </tr>
            <tr>
                <td class="label">Total Egresos:</td>
                <td class="egreso">$' . number_format($resumen['egresos_total'], 0, ',', '.') . '</td>
            </tr>
            <tr class="total-row">
                <td class="label">Saldo Actual:</td>
                <td>$' . number_format($resumen['saldo_actual'], 0, ',', '.') . '</td>
            </tr>
        </table>
    </div>
    
    <div class="movimientos">
        <h2>Detalle de Movimientos (' . count($movimientos) . ' registros)</h2>
        <table>
            <thead>
                <tr>
                    <th>Fecha</th>
                    <th>Concepto</th>
                    <th>Descripción</th>
                    <th>ScaWatt</th>
                    <th style="text-align: right;">Monto</th>
                </tr>
            </thead>
            <tbody>';
        
        foreach ($movimientos as $mov) {
            $clase = $mov['tipo'] === 'ingreso' ? 'ingreso' : 'egreso';
            $signo = $mov['tipo'] === 'ingreso' ? '+' : '-';
            $fecha = date('d/m/Y H:i', strtotime($mov['fecha']));
            $scawatt = $mov['scawatt_id'] ?? '-';
            
            $html .= '
                <tr>
                    <td>' . $fecha . '</td>
                    <td>' . htmlspecialchars($mov['concepto']) . '</td>
                    <td>' . htmlspecialchars($mov['descripcion']) . '</td>
                    <td>' . htmlspecialchars($scawatt) . '</td>
                    <td style="text-align: right;" class="' . $clase . '">' . $signo . '$' . number_format($mov['monto'], 0, ',', '.') . '</td>
                </tr>';
        }
        
        $html .= '
            </tbody>
        </table>
    </div>
    
    <div class="footer">
        <p>ScaWatts - Sistema de Gestión de Inversiones en Energía Solar</p>
        <p>Este documento es un registro oficial de movimientos financieros</p>
    </div>
</body>
</html>';
        
        return $html;
    }

    /**
     * Estadísticas generales de usuarios
     * GET /api/admin/usuarios/estadisticas
     */
    public function estadisticas()
    {
        $stats = [
            'total_usuarios' => Usuario::count(),
            'usuarios_activos' => Usuario::where('estado', 'activo')->count(),
            'usuarios_inactivos' => Usuario::where('estado', 'inactivo')->count(),
            'usuarios_bloqueados' => Usuario::where('estado', 'bloqueado')->count(),
            'usuarios_con_scawatts' => Usuario::has('scawatts')->count(),
            'total_saldo_sistema' => Usuario::sum('saldo_disponible')
        ];

        return response()->json($stats);
    }

    /**
     * Generar contraseña aleatoria segura
     */
    private function generarPasswordAleatoria(): string
    {
        // Generar password de 10 caracteres: letras mayúsculas, minúsculas y números
        $caracteres = 'ABCDEFGHJKLMNPQRSTUVWXYZabcdefghijkmnpqrstuvwxyz23456789';
        $password = '';
        $longitudCaracteres = strlen($caracteres);
        
        for ($i = 0; $i < 10; $i++) {
            $password .= $caracteres[random_int(0, $longitudCaracteres - 1)];
        }
        
        return $password;
    }

    /**
     * Enviar contraseña por correo electrónico
     */
    private function enviarPasswordPorEmail($usuario, $password)
    {
        // TODO: Implementar envío de email real
        // Por ahora, registrar en los logs
        \Log::info('Password generada para usuario', [
            'usuario_id' => $usuario->id,
            'nombre' => $usuario->nombre,
            'email' => $usuario->email,
            'password' => $password
        ]);

        // Crear una notificación en el sistema
        try {
            \App\Models\Notificacion::create([
                'usuario_id' => $usuario->id,
                'tipo' => 'sistema',
                'titulo' => 'Bienvenido a ScaWatts',
                'contenido' => "Tu cuenta ha sido creada exitosamente. Tu contraseña es: {$password}. Por seguridad, te recomendamos cambiarla después de tu primer inicio de sesión.",
                'leido' => false,
                'icono' => 'key'
            ]);
        } catch (\Exception $e) {
            \Log::warning('No se pudo crear notificación de bienvenida: ' . $e->getMessage());
        }
    }

    private function formatDecimalString(float $value): string
    {
        return number_format($value, 2, '.', '');
    }

    /**
     * Estructurar la respuesta del perfil del usuario para consumo del frontend
     */
    private function formatearPerfilUsuario(Usuario $usuario): array
    {
        $numeroCuenta = $usuario->numero_cuenta ? (string) $usuario->numero_cuenta : null;
        $cuenta = null;

        if ($numeroCuenta) {
            $mascara = strlen($numeroCuenta) >= 4
                ? '****' . substr($numeroCuenta, -4)
                : $numeroCuenta;

            $cuenta = [
                'banco' => $usuario->banco,
                'tipo_cuenta' => $usuario->tipo_cuenta,
                'numero_cuenta' => $numeroCuenta,
                'numero_cuenta_parcial' => $mascara,
                'estado' => $usuario->datos_bancarios_verificados_at ? 'verificado' : 'pendiente',
                'verificado' => (bool) $usuario->datos_bancarios_verificados_at,
                'verificado_fecha' => $usuario->datos_bancarios_verificados_at,
            ];
        }

        return [
            'id' => $usuario->id,
            'nombre' => $usuario->nombre,
            'email' => $usuario->email,
            'telefono' => $usuario->telefono,
            'picture' => $usuario->picture,
            'estado' => $usuario->estado,
            'fecha_registro' => $usuario->created_at,
            'actualizado_at' => $usuario->updated_at,
            'saldo' => [
                'disponible' => $usuario->saldo_disponible,
                'pendiente' => $usuario->saldo_pendiente,
                'total' => $usuario->saldo_total,
            ],
            'verificaciones' => [
                'cedula' => (bool) $usuario->cedula_verificada_at,
                'datos_bancarios' => (bool) $usuario->datos_bancarios_verificados_at,
            ],
            'cuentas_bancarias' => $cuenta ? [$cuenta] : [],
        ];
    }

    /**
     * Obtener valorización de ScaWatts del usuario
     * GET /api/usuarios/{id}/valorizacion
     */
    public function valorizacion($id)
    {
        try {
            $usuario = Usuario::findOrFail($id);

            $portafolio = $this->valuationService->buildRealtimePortfolio($usuario);

            if (($portafolio['scawatts'] ?? null) instanceof \Illuminate\Support\Collection) {
                $portafolio['scawatts'] = $portafolio['scawatts']->values()->toArray();
            }

            return response()->json($portafolio);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al obtener valorización',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}

