<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class AuditoriaAdmin extends Model
{
    use HasFactory;

    protected $table = 'auditoria_admin';

    protected $fillable = [
        'admin_id',
        'accion',
        'modulo',
        'entidad_tipo',
        'entidad_id',
        'datos_anteriores',
        'datos_nuevos',
        'ip_address',
        'user_agent',
        'descripcion'
    ];

    protected $casts = [
        'datos_anteriores' => 'array',
        'datos_nuevos' => 'array',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    /**
     * Relación con Usuario administrador
     */
    public function admin()
    {
        return $this->belongsTo(Usuario::class, 'admin_id');
    }

    /**
     * Registrar acción de auditoría
     */
    public static function registrar($adminId, $accion, $modulo, $descripcion = null, $entidadTipo = null, $entidadId = null, $datosAnteriores = null, $datosNuevos = null)
    {
        return self::create([
            'admin_id' => $adminId,
            'accion' => $accion,
            'modulo' => $modulo ?? 'general',
            'entidad_tipo' => $entidadTipo,
            'entidad_id' => $entidadId,
            'datos_anteriores' => $datosAnteriores,
            'datos_nuevos' => $datosNuevos,
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'descripcion' => $descripcion
        ]);
    }

    /**
     * Scope por módulo
     */
    public function scopePorModulo($query, $modulo)
    {
        return $query->where('modulo', $modulo);
    }

    /**
     * Scope por acción
     */
    public function scopePorAccion($query, $accion)
    {
        return $query->where('accion', $accion);
    }

    /**
     * Scope por admin
     */
    public function scopePorAdmin($query, $adminId)
    {
        return $query->where('admin_id', $adminId);
    }

    /**
     * Scope acciones críticas
     */
    public function scopeAccionesCriticas($query)
    {
        return $query->whereIn('accion', ['eliminar', 'aprobar_retiro', 'cambiar_estado_usuario']);
    }
}
