<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Retiro extends Model
{
    use HasFactory;

    protected $table = 'retiros';

    protected $fillable = [
        'usuario_id',
        'monto',
        'metodo_retiro',
        'datos_bancarios',
        'estado',
        'fecha_solicitud',
    'fecha_procesado',
        'procesado_por',
        'referencia_transaccion',
        'comprobante_ruta',
        'notas_admin',
        'notas_usuario'
    ];

    protected $casts = [
        'monto' => 'decimal:2',
        'datos_bancarios' => 'array',
        'fecha_solicitud' => 'datetime',
        'fecha_procesado' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    /**
     * Relación con Usuario
     */
    public function usuario()
    {
        return $this->belongsTo(Usuario::class, 'usuario_id');
    }

    /**
     * Relación con admin que procesó
     */
    public function procesador()
    {
        return $this->belongsTo(Usuario::class, 'procesado_por');
    }

    /**
     * Scope por estado
     */
    public function scopePorEstado($query, $estado)
    {
        return $query->where('estado', $estado);
    }

    /**
     * Scope pendientes
     */
    public function scopePendientes($query)
    {
        return $query->where('estado', 'pendiente');
    }

    /**
     * Scope procesados
     */
    public function scopeProcesados($query)
    {
    return $query->where('estado', 'completado');
    }

    /**
     * Scope rechazados
     */
    public function scopeRechazados($query)
    {
        return $query->where('estado', 'rechazado');
    }

    /**
     * Aprobar y procesar retiro
     */
    public function aprobar($adminId, $referenciaTransaccion = null, $notas = null)
    {
        // Verificar que el usuario tenga saldo pendiente suficiente
        if ($this->usuario->saldo_pendiente < $this->monto) {
            throw new \Exception('Saldo pendiente insuficiente para procesar el retiro');
        }

        $this->update([
            'estado' => 'completado',
            'fecha_procesado' => now(),
            'procesado_por' => $adminId,
            'referencia_transaccion' => $referenciaTransaccion,
            'notas_admin' => $notas
        ]);

        // Liberar del saldo pendiente (el saldo ya fue congelado cuando se creó la solicitud)
        $this->usuario->decrement('saldo_pendiente', $this->monto);
    }

    /**
     * Rechazar retiro
     */
    public function rechazar($adminId, $motivo)
    {
        $this->update([
            'estado' => 'rechazado',
            'fecha_procesado' => now(),
            'procesado_por' => $adminId,
            'notas_admin' => $motivo
        ]);

        // Devolver el saldo de pendiente a disponible
        $this->usuario->decrement('saldo_pendiente', $this->monto);
        $this->usuario->increment('saldo_disponible', $this->monto);
    }
}
