<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class SolicitudContactoScawatt extends Model
{
    use HasFactory;

    protected $table = 'solicitudes_contacto_scawatt';

    protected $fillable = [
        'usuario_id',
        'nombre_completo',
        'cedula',
        'email',
        'telefono',
        'cantidad_scawatts',
        'presupuesto_aproximado',
        'mensaje',
        'estado',
        'fecha_contacto',
        'atendido_por',
        'estado_actualizado_por',
        'notas_admin',
        'link_pago',
        'monto_pago',
        'fecha_pago',
        'mail_thread_token',
        'ultima_notificacion_enviada_at',
        'historial_eventos'
    ];

    protected $casts = [
        'cantidad_scawatts' => 'integer',
        'presupuesto_aproximado' => 'decimal:2',
        'fecha_contacto' => 'datetime',
        'fecha_pago' => 'datetime',
        'ultima_notificacion_enviada_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'historial_eventos' => 'array',
        'monto_pago' => 'decimal:2'
    ];

    /**
     * Relación con Usuario (solicitante)
     */
    public function usuario()
    {
        return $this->belongsTo(Usuario::class, 'usuario_id');
    }

    /**
     * Relación con Usuario (admin que atendió)
     */
    public function adminAtendio()
    {
        return $this->belongsTo(Usuario::class, 'atendido_por');
    }

    public function estadoActualizadoPor()
    {
        return $this->belongsTo(Usuario::class, 'estado_actualizado_por');
    }

    public function registrarEventoHistorial(array $evento): void
    {
        $historial = $this->historial_eventos ?? [];
        $historial[] = array_merge([
            'estado' => $this->estado,
            'registrado_por' => $this->estado_actualizado_por,
            'created_at' => now()->toIso8601String(),
        ], $evento);

        $this->historial_eventos = $historial;
        $this->save();
    }

    /**
     * Scope para solicitudes pendientes
     */
    public function scopePendientes($query)
    {
        return $query->where('estado', 'pendiente');
    }

    /**
     * Scope para solicitudes contactadas
     */
    public function scopeContactadas($query)
    {
        return $query->where('estado', 'contactado');
    }

    /**
     * Scope para solicitudes completadas
     */
    public function scopeCompletadas($query)
    {
        return $query->where('estado', 'completado');
    }

    /**
     * Marcar como contactado
     */
    public function marcarContactado($adminId, $notas = null)
    {
        $this->update([
            'estado' => 'contactado',
            'fecha_contacto' => now(),
            'atendido_por' => $adminId,
            'notas_admin' => $notas
        ]);
    }

    /**
     * Marcar como completado
     */
    public function marcarCompletado($adminId, $notas = null)
    {
        $this->update([
            'estado' => 'completado',
            'atendido_por' => $adminId,
            'notas_admin' => $notas
        ]);
    }

    /**
     * Marcar como cancelado
     */
    public function marcarCancelado($adminId, $motivo = null)
    {
        $this->update([
            'estado' => 'cancelado',
            'atendido_por' => $adminId,
            'notas_admin' => $motivo
        ]);
    }
}
