<?php

namespace App\Services;

use Illuminate\Support\Facades\Log;

class FirebaseService
{
    private $serverKey;
    private $fcmUrl = 'https://fcm.googleapis.com/fcm/send';

    public function __construct()
    {
        // Usar config() en lugar de env() para mejor performance y caching
        $this->serverKey = config('services.firebase.server_key', '');
    }

    /**
     * Enviar notificación push a un dispositivo
     * 
     * @param string $token Token FCM del dispositivo
     * @param string $title Título de la notificación
     * @param string $body Cuerpo del mensaje
     * @param array $data Datos adicionales (opcional)
     * @param string $sound Sonido (default, custom.mp3)
     * @param int|null $badge Badge count (si no se pasa, se calcula automáticamente)
     * @return array Resultado del envío
     */
    public function enviarNotificacion($token, $title, $body, $data = [], $sound = 'default', $badge = null)
    {
        if (empty($this->serverKey)) {
            Log::warning('FCM_SERVER_KEY no configurado en .env');
            return [
                'success' => false,
                'message' => 'FCM no configurado'
            ];
        }

        if (empty($token)) {
            return [
                'success' => false,
                'message' => 'Token FCM vacío'
            ];
        }

        // Calcular badge automáticamente si no se proporciona
        if ($badge === null && isset($data['usuario_id'])) {
            $badge = \App\Models\Notificacion::where('usuario_id', $data['usuario_id'])
                ->where('leido', false)
                ->count() + 1; // +1 porque esta notificación aún no está en BD
        } else {
            $badge = $badge ?? 1;
        }

        $notification = [
            'to' => $token,
            'notification' => [
                'title' => $title,
                'body' => $body,
                'sound' => $sound,
                'badge' => $badge, // Número de notificaciones sin leer
                'click_action' => 'FLUTTER_NOTIFICATION_CLICK',
                'priority' => 'high'
            ],
            'data' => array_merge($data, [
                'titulo' => $title,
                'mensaje' => $body,
                'timestamp' => now()->toIso8601String(),
                'badge' => $badge // También en data para que la app lo tenga
            ]),
            'priority' => 'high',
            'content_available' => true
        ];

        $headers = [
            'Authorization: key=' . $this->serverKey,
            'Content-Type: application/json'
        ];

        try {
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $this->fcmUrl);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($notification));
            curl_setopt($ch, CURLOPT_TIMEOUT, 10);

            $result = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);

            $response = json_decode($result, true);

            if ($httpCode == 200 && isset($response['success']) && $response['success'] == 1) {
                Log::info('FCM enviado exitosamente', [
                    'token' => substr($token, 0, 20) . '...',
                    'title' => $title
                ]);

                return [
                    'success' => true,
                    'message' => 'Notificación enviada',
                    'response' => $response
                ];
            } else {
                Log::error('Error al enviar FCM', [
                    'http_code' => $httpCode,
                    'response' => $response
                ]);

                return [
                    'success' => false,
                    'message' => 'Error al enviar notificación',
                    'error' => $response
                ];
            }
        } catch (\Exception $e) {
            Log::error('Excepción al enviar FCM: ' . $e->getMessage());

            return [
                'success' => false,
                'message' => 'Excepción: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Enviar notificación a múltiples dispositivos
     * 
     * @param array $tokens Array de tokens FCM
     * @param string $title Título
     * @param string $body Mensaje
     * @param array $data Datos adicionales
     * @return array Resultados
     */
    public function enviarNotificacionMultiple($tokens, $title, $body, $data = [])
    {
        if (empty($this->serverKey)) {
            return [
                'success' => false,
                'message' => 'FCM no configurado'
            ];
        }

        if (empty($tokens) || !is_array($tokens)) {
            return [
                'success' => false,
                'message' => 'Tokens inválidos'
            ];
        }

        $notification = [
            'registration_ids' => $tokens,
            'notification' => [
                'title' => $title,
                'body' => $body,
                'sound' => 'default',
                'priority' => 'high'
            ],
            'data' => array_merge($data, [
                'titulo' => $title,
                'mensaje' => $body,
                'timestamp' => now()->toIso8601String()
            ]),
            'priority' => 'high'
        ];

        $headers = [
            'Authorization: key=' . $this->serverKey,
            'Content-Type: application/json'
        ];

        try {
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $this->fcmUrl);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($notification));

            $result = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);

            $response = json_decode($result, true);

            Log::info('FCM múltiple enviado', [
                'total_tokens' => count($tokens),
                'success' => $response['success'] ?? 0,
                'failure' => $response['failure'] ?? 0
            ]);

            return [
                'success' => true,
                'message' => 'Notificaciones enviadas',
                'response' => $response
            ];
        } catch (\Exception $e) {
            Log::error('Error FCM múltiple: ' . $e->getMessage());

            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Validar si el token FCM es válido (formato básico)
     * 
     * @param string $token
     * @return bool
     */
    public function validarToken($token)
    {
        return !empty($token) && strlen($token) > 50;
    }
}
