<?php

namespace App\Services;

use App\Models\Usuario;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class ScawattDocumentManager
{
    public function copyDocuments(Usuario $usuario, string $scawattSerial): bool
    {
        try {
            $scawattDbId = DB::table('scawatts')->where('scawatt_id', $scawattSerial)->value('id');
            if (!$scawattDbId) {
                Log::warning('No se encontro registro del ScaWatt en la base de datos', [
                    'scawatt_serial' => $scawattSerial,
                ]);
                return false;
            }

            $userReference = $usuario->id_wordpress ?? $usuario->id;
            $repositoryBase = "/home/scalarapp/public_html/scadmin/documentos_scawatts/{$scawattSerial}";

            $legacyPaths = [
                "/home/scalarapp/public_html/scalar-api/api-scawatts/produccion/scausers/usuarios-app/{$userReference}/documentos-scawatt",
            ];

            if (is_numeric($userReference)) {
                $legacyPaths[] = "/home/scalarapp/public_html/scalar-api/api-scawatts/produccion/scausers/usuarios-app/laravel-{$userReference}/documentos-scawatt";
            }

            $laravelDocsPath = "documentos/{$usuario->id}";
            Storage::makeDirectory($laravelDocsPath);
            Storage::disk('public')->makeDirectory($laravelDocsPath);

            $candidates = [
                'certificado' => [
                    "{$repositoryBase}/certificado_{$scawattSerial}.pdf",
                ],
                'contrato' => [
                    "{$repositoryBase}/contrato_{$scawattSerial}.pdf",
                ],
            ];

            foreach ($legacyPaths as $path) {
                $candidates['certificado'][] = "{$path}/certificado_{$scawattSerial}.pdf";
                $candidates['contrato'][] = "{$path}/contrato_{$scawattSerial}.pdf";
            }

            foreach ($candidates as $tipo => $paths) {
                $source = null;
                foreach ($paths as $ruta) {
                    if (file_exists($ruta)) {
                        $source = $ruta;
                        break;
                    }
                }

                if (!$source) {
                    Log::warning('Documento no encontrado en repositorio', [
                        'tipo' => $tipo,
                        'scawatt_serial' => $scawattSerial,
                    ]);
                    continue;
                }

                $content = file_get_contents($source);

                $privateDest = "{$laravelDocsPath}/{$tipo}_{$scawattSerial}.pdf";
                Storage::put($privateDest, $content);

                $publicDest = "{$laravelDocsPath}/{$tipo}_{$scawattSerial}.pdf";
                Storage::disk('public')->put($publicDest, $content);

                $data = [
                    'usuario_id' => $usuario->id,
                    'scawatt_id' => $scawattDbId,
                    'tipo' => $tipo,
                    'nombre_archivo' => "{$tipo}_{$scawattSerial}.pdf",
                    'ruta_archivo' => $privateDest,
                    'url_publica' => Storage::disk('public')->url($publicDest),
                    'tamaño_bytes' => filesize($source),
                    'fecha_generacion' => now(),
                    'updated_at' => now(),
                ];

                $existing = DB::table('documentos')
                    ->where('usuario_id', $usuario->id)
                    ->where('scawatt_id', $scawattDbId)
                    ->where('tipo', $tipo)
                    ->first();

                if ($existing) {
                    DB::table('documentos')->where('id', $existing->id)->update($data);
                } else {
                    $data['created_at'] = now();
                    DB::table('documentos')->insert($data);
                }
            }

            return true;
        } catch (\Throwable $exception) {
            Log::error('Error copiando documentos de ScaWatt', [
                'error' => $exception->getMessage(),
                'scawatt_serial' => $scawattSerial,
            ]);
            return false;
        }
    }
}
