<?php

namespace App\Services;

use App\Models\Scawatt;
use App\Models\Usuario;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use RuntimeException;

class ScawattValuationService
{
    private ?array $latestSnapshotMeta = null;
    private string $tokensPath;
    private ?string $forecastPath;
    private int $tokensCacheTtl;
    private int $baselineTtl;

    public function __construct()
    {
        $this->tokensPath = (string) config('scawatt.paths.tokens');
        $this->forecastPath = config('scawatt.paths.forecast');
        $this->tokensCacheTtl = (int) config('scawatt.cache.tokens_snapshot_ttl', 5);
        $this->baselineTtl = (int) config('scawatt.cache.baseline_ttl', 86400);
    }

    public function syncDatabaseFromTokens(bool $persist = true): array
    {
        $snapshot = $this->readTokensSnapshot();
        $serials = array_keys($snapshot);

        if (empty($serials)) {
            return [
                'persisted' => $persist,
                'processed' => 0,
                'updated' => 0,
                'unchanged' => 0,
                'missing_in_db' => [],
                'snapshot' => $this->getSnapshotMeta(),
            ];
        }

        $scawatts = Scawatt::whereIn('scawatt_id', $serials)->get();

        $updated = 0;
        $unchanged = 0;

        foreach ($scawatts as $scawatt) {
            $serial = $scawatt->scawatt_id;
            $tokenPayload = $snapshot[$serial] ?? null;

            if (!$tokenPayload) {
                continue;
            }

            $valorInicial = (float) ($tokenPayload['valor'] ?? $scawatt->valor_inicial ?? 0);
            $valorActual = $valorInicial + (float) ($tokenPayload['valorizacion'] ?? 0);
            $valorizacion = $valorActual - $valorInicial;

            $needsUpdate = !$this->valuesAlmostEqual((float) $scawatt->valor_inicial, $valorInicial)
                || !$this->valuesAlmostEqual((float) $scawatt->valor_actual, $valorActual)
                || empty($scawatt->valorizaciones);

            if ($persist && $needsUpdate) {
                $scawatt->forceFill([
                    'valor_inicial' => $valorInicial,
                    'valor_actual' => $valorActual,
                    'valorizaciones' => [$valorizacion],
                ])->save();

                $updated++;
            } else {
                $unchanged++;
            }

            $this->rememberBaseline($serial, $valorActual, $valorInicial);
        }

        $missing = array_values(array_diff($serials, $scawatts->pluck('scawatt_id')->all()));

        return [
            'persisted' => $persist,
            'processed' => $scawatts->count(),
            'updated' => $updated,
            'unchanged' => $unchanged,
            'missing_in_db' => $missing,
            'snapshot' => $this->getSnapshotMeta(),
        ];
    }

    public function buildRealtimePortfolio(Usuario $usuario): array
    {
        $scawatts = $usuario->scawatts()->where('estado', 'activo')->get();

        try {
            $snapshot = $this->readTokensSnapshot();
        } catch (\Throwable $e) {
            Log::warning('No se pudo leer scawatts_tokens.json para valorizaciones en tiempo real', [
                'error' => $e->getMessage(),
            ]);
            $snapshot = [];
        }

        $resultScawatts = [];
        $totalInicial = 0.0;
        $totalActual = 0.0;

        foreach ($scawatts as $scawatt) {
            $serial = $scawatt->scawatt_id;
            $tokenPayload = $snapshot[$serial] ?? null;
            $baseline = $this->getBaseline($serial);

            if ($tokenPayload) {
                $valorInicial = (float) ($tokenPayload['valor'] ?? $scawatt->valor_inicial ?? 0);
                $valorActual = $valorInicial + (float) ($tokenPayload['valorizacion'] ?? 0);
            } elseif ($baseline) {
                $valorInicial = (float) ($baseline['valor_inicial'] ?? $scawatt->valor_inicial ?? 0);
                $valorActual = (float) ($baseline['valor_actual'] ?? $scawatt->valor_actual ?? 0);
            } else {
                $valorInicial = (float) $scawatt->valor_inicial;
                $valorActual = (float) $scawatt->valor_actual;
            }

            $valorizacion = $valorActual - $valorInicial;

            $resultScawatts[] = [
                'id' => $scawatt->id,
                'scawatt_id' => $serial,
                'valor_inicial' => round($valorInicial, 2),
                'valor_actual' => round($valorActual, 2),
                'valorizacion' => round($valorizacion, 2),
                'kwh_asignados' => (float) $scawatt->kwh_asignados,
                'fecha_inicio' => $scawatt->fecha_inicio,
                'fecha_final' => $scawatt->fecha_final,
            ];

            $totalInicial += $valorInicial;
            $totalActual += $valorActual;
        }

        $valorizacionTotal = $totalActual - $totalInicial;

        return [
            'total_scawatts' => (int) $scawatts->count(),
            'valor_inicial_total' => round($totalInicial, 2),
            'valor_actual_total' => round($totalActual, 2),
            'valorizacion_total' => round($valorizacionTotal, 2),
            'valorizacion_porcentaje' => $totalInicial > 0 ? round(($valorizacionTotal / $totalInicial) * 100, 2) : 0.0,
            'scawatts' => collect($resultScawatts),
            'forecast_window' => $this->buildForecastWindow(),
        ];
    }

    public function getSnapshotMeta(): array
    {
        return $this->latestSnapshotMeta ?? [];
    }

    private function readTokensSnapshot(): array
    {
        if (empty($this->tokensPath)) {
            throw new RuntimeException('Ruta del archivo scawatts_tokens.json no configurada');
        }

        $cacheKey = 'scawatt:tokens:snapshot';

        $snapshot = Cache::remember($cacheKey, now()->addSeconds(max(1, $this->tokensCacheTtl)), function () {
            $data = $this->decodeJsonFile($this->tokensPath);

            return [
                'data' => $data,
                'meta' => [
                    'path' => $this->tokensPath,
                    'file_mtime' => $this->fileModifiedAt($this->tokensPath),
                    'cached_at' => Carbon::now()->toIso8601String(),
                ],
            ];
        });

        $this->latestSnapshotMeta = $snapshot['meta'] ?? null;

        return $snapshot['data'] ?? [];
    }

    private function buildForecastWindow(): ?array
    {
        if (empty($this->forecastPath) || !is_file($this->forecastPath)) {
            return null;
        }

        $windowSeconds = (int) config('scawatt.forecast.window_seconds', 600);
        $windowSeconds = max(60, $windowSeconds);

        try {
            $forecastData = $this->decodeJsonFile($this->forecastPath, true);
        } catch (\Throwable $e) {
            Log::warning('No se pudo leer ScaWatt.json para pronóstico inmediato', [
                'error' => $e->getMessage(),
            ]);
            return null;
        }

        if (empty($forecastData)) {
            return null;
        }

        $now = Carbon::now('America/Bogota');
        $limit = $now->copy()->addSeconds($windowSeconds);
        $window = [];

        foreach ($forecastData as $entry) {
            if (!isset($entry['fecha'], $entry['valor'])) {
                continue;
            }

            try {
                $timestamp = Carbon::createFromFormat('d/m/Y H:i:s', $entry['fecha'], 'America/Bogota');
            } catch (\Throwable $e) {
                continue;
            }

            if ($timestamp->lt($now) || $timestamp->gt($limit)) {
                continue;
            }

            $window[] = [
                'fecha' => $timestamp->toIso8601String(),
                'valor' => (float) $entry['valor'],
            ];
        }

        if (empty($window)) {
            return null;
        }

        return [
            'desde' => $now->toIso8601String(),
            'hasta' => $limit->toIso8601String(),
            'registros' => $window,
        ];
    }

    private function decodeJsonFile(string $path, bool $allowTrailingComma = false): array
    {
        if (!is_file($path) || !is_readable($path)) {
            throw new RuntimeException("Archivo no accesible: {$path}");
        }

        $contents = file_get_contents($path);

        if ($contents === false) {
            throw new RuntimeException("No se pudo leer el archivo: {$path}");
        }

        if ($allowTrailingComma) {
            $cleaned = preg_replace('/,\s*([\]}])/m', '$1', $contents ?? '');
            if (is_string($cleaned)) {
                $contents = $cleaned;
            }
        }

        $decoded = json_decode($contents, true);

        if (json_last_error() !== JSON_ERROR_NONE || !is_array($decoded)) {
            throw new RuntimeException('JSON inválido en ' . $path . ': ' . json_last_error_msg());
        }

        return $decoded;
    }

    private function fileModifiedAt(string $path): ?string
    {
        $mtime = @filemtime($path);

        if ($mtime === false) {
            return null;
        }

        return Carbon::createFromTimestamp($mtime)->toIso8601String();
    }

    private function rememberBaseline(string $serial, float $valorActual, float $valorInicial): void
    {
        if ($this->baselineTtl <= 0) {
            return;
        }

        Cache::put(
            $this->baselineCacheKey($serial),
            [
                'valor_actual' => $valorActual,
                'valor_inicial' => $valorInicial,
                'captured_at' => Carbon::now()->toIso8601String(),
            ],
            now()->addSeconds($this->baselineTtl)
        );
    }

    private function getBaseline(string $serial): ?array
    {
        return Cache::get($this->baselineCacheKey($serial));
    }

    private function baselineCacheKey(string $serial): string
    {
        return 'scawatt:baseline:' . $serial;
    }

    private function valuesAlmostEqual(float $left, float $right, float $epsilon = 0.01): bool
    {
        return abs($left - $right) <= $epsilon;
    }
}
