<?php

namespace App\Services;

use App\Mail\NuevaSolicitudScawattAdminMail;
use App\Mail\NuevaSolicitudScawattUsuarioMail;
use App\Mail\SolicitudScawattDatosPendientesMail;
use App\Mail\SolicitudScawattPagoConfirmadoAdminMail;
use App\Mail\SolicitudScawattPagoConfirmadoMail;
use App\Mail\SolicitudScawattPagoPendienteMail;
use App\Mail\SolicitudScawattPagoRevisionMail;
use App\Models\Scawatt;
use App\Models\SolicitudContactoScawatt;
use App\Models\Usuario;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;

class SolicitudScawattWorkflowService
{
    public const ESTADO_PENDIENTE = 'pendiente';
    public const ESTADO_DATOS_PENDIENTES = 'datos_pendientes';
    public const ESTADO_EN_REVISION = 'en_revision';
    public const ESTADO_PAGO_PENDIENTE = 'pago_pendiente';
    public const ESTADO_PAGO_EN_REVISION = 'pago_en_revision';
    public const ESTADO_COMPLETADO = 'completado';
    public const ESTADO_CANCELADO = 'cancelado';

    public function __construct(private readonly ScawattDocumentManager $documentManager)
    {
    }

    public function handleNuevaSolicitud(SolicitudContactoScawatt $solicitud): void
    {
        $solicitud->loadMissing('usuario');
    $this->ensureThreadToken($solicitud);
        $this->ensureThreadToken($solicitud);
        $this->registrarEvento($solicitud, self::ESTADO_PENDIENTE, 'Solicitud creada');

        $adminRecipients = $this->adminRecipients();

        if ($adminRecipients->isNotEmpty()) {
            Mail::to($adminRecipients->all())->send(new NuevaSolicitudScawattAdminMail($solicitud, true));
        }

        $correoUsuario = $solicitud->email ?: $solicitud->usuario?->email;
        if (!empty($correoUsuario)) {
            Mail::to($correoUsuario)->send(new NuevaSolicitudScawattUsuarioMail($solicitud, true));
        }

        $solicitud->ultima_notificacion_enviada_at = now();
        $solicitud->save();
    }

    public function marcarDatosPendientes(SolicitudContactoScawatt $solicitud, Usuario $admin, ?string $nota = null): void
    {
        $solicitud->loadMissing('usuario');
    $this->ensureThreadToken($solicitud);
        $solicitud->estado = self::ESTADO_DATOS_PENDIENTES;
        $solicitud->estado_actualizado_por = $admin->id;
        if (!empty($nota)) {
            $solicitud->notas_admin = $nota;
        }

        $pendientes = $this->construirChecklistPendientes($solicitud);
        $correoUsuario = $solicitud->email ?: $solicitud->usuario?->email;

        if (!empty($correoUsuario)) {
            Mail::to($correoUsuario)->send(new SolicitudScawattDatosPendientesMail($solicitud, $pendientes));
        }

        $this->registrarEvento($solicitud, self::ESTADO_DATOS_PENDIENTES, 'Se solicitaron datos faltantes', $admin->id, $nota);
        $solicitud->ultima_notificacion_enviada_at = now();
        $solicitud->save();
    }

    public function liberarPago(SolicitudContactoScawatt $solicitud, Usuario $admin, string $linkPago, float $monto, ?string $nota = null): void
    {
        $solicitud->loadMissing('usuario');
    $this->ensureThreadToken($solicitud);
        $this->validarDocumentacionCompleta($solicitud);

        $solicitud->estado = self::ESTADO_PAGO_PENDIENTE;
        $solicitud->estado_actualizado_por = $admin->id;
        $solicitud->link_pago = $linkPago;
        $solicitud->monto_pago = $monto;
        if (!empty($nota)) {
            $solicitud->notas_admin = $nota;
        }

        $correoUsuario = $solicitud->email ?: $solicitud->usuario?->email;
        if (!empty($correoUsuario)) {
            Mail::to($correoUsuario)->send(new SolicitudScawattPagoPendienteMail($solicitud, $linkPago, $monto));
        }

        $this->registrarEvento($solicitud, self::ESTADO_PAGO_PENDIENTE, 'Se envio link de pago', $admin->id, $nota, [
            'link_pago' => $linkPago,
            'monto_pago' => $monto,
        ]);

        $solicitud->ultima_notificacion_enviada_at = now();
        $solicitud->save();
    }

    public function marcarPagoEnRevision(SolicitudContactoScawatt $solicitud, Usuario $admin, ?string $nota = null): void
    {
        $solicitud->loadMissing('usuario');
    $this->ensureThreadToken($solicitud);
        $solicitud->estado = self::ESTADO_PAGO_EN_REVISION;
        $solicitud->estado_actualizado_por = $admin->id;
        if (!empty($nota)) {
            $solicitud->notas_admin = $nota;
        }

        $correoUsuario = $solicitud->email ?: $solicitud->usuario?->email;
        if (!empty($correoUsuario)) {
            Mail::to($correoUsuario)->send(new SolicitudScawattPagoRevisionMail($solicitud));
        }

        $this->registrarEvento($solicitud, self::ESTADO_PAGO_EN_REVISION, 'Pago en revision', $admin->id, $nota);

        $solicitud->ultima_notificacion_enviada_at = now();
        $solicitud->save();
    }

    public function confirmarPago(SolicitudContactoScawatt $solicitud, Usuario $admin, ?float $montoConfirmado = null, ?string $nota = null): void
    {
        $solicitud->loadMissing('usuario');
        if (empty($solicitud->link_pago)) {
            throw ValidationException::withMessages([
                'accion' => 'La solicitud no tiene un enlace de pago registrado.',
            ]);
        }
        $this->ensureThreadToken($solicitud);

        if ($solicitud->estado === self::ESTADO_COMPLETADO) {
            throw ValidationException::withMessages([
                'accion' => 'Esta solicitud ya fue completada anteriormente.',
            ]);
        }

        $usuario = $solicitud->usuario;
        if (!$usuario) {
            throw ValidationException::withMessages([
                'usuario' => 'La solicitud no tiene un usuario asociado.',
            ]);
        }

        $cantidad = max(1, (int) ($solicitud->cantidad_scawatts ?? 1));

        DB::beginTransaction();
        try {
            $resumen = [];
            for ($i = 0; $i < $cantidad; $i++) {
                $resumen[] = $this->generarScawattParaUsuario($usuario);
            }

            $solicitud->estado = self::ESTADO_COMPLETADO;
            $solicitud->estado_actualizado_por = $admin->id;
            $solicitud->fecha_pago = Carbon::now();
            if ($montoConfirmado !== null) {
                $solicitud->monto_pago = $montoConfirmado;
            }
            if (!empty($nota)) {
                $solicitud->notas_admin = $nota;
            }

            $solicitud->save();
            DB::commit();
        } catch (\Throwable $exception) {
            DB::rollBack();
            Log::error('No se pudo confirmar el pago de la solicitud ScaWatt', [
                'solicitud_id' => $solicitud->id,
                'error' => $exception->getMessage(),
            ]);
            throw $exception;
        }

        $documentos = $this->obtenerDocumentosPrincipales($usuario, $resumen);

        $correoUsuario = $solicitud->email ?: $usuario->email;
        if (!empty($correoUsuario)) {
            Mail::to($correoUsuario)->send(new SolicitudScawattPagoConfirmadoMail(
                $solicitud,
                $resumen,
                $documentos['contrato'] ?? null,
                $documentos['certificado'] ?? null
            ));
        }

        $adminRecipients = $this->adminRecipients();
        if ($adminRecipients->isNotEmpty()) {
            Mail::to($adminRecipients->all())->send(new SolicitudScawattPagoConfirmadoAdminMail($solicitud, $resumen));
        }

        $this->registrarEvento($solicitud, self::ESTADO_COMPLETADO, 'Pago confirmado y ScaWatts generados', $admin->id, $nota, [
            'scawatts' => $resumen,
        ]);

        $solicitud->ultima_notificacion_enviada_at = now();
        $solicitud->save();
    }

    public function cancelar(SolicitudContactoScawatt $solicitud, Usuario $admin, ?string $motivo = null): void
    {
        $solicitud->loadMissing('usuario');
        $this->ensureThreadToken($solicitud);
        $solicitud->estado = self::ESTADO_CANCELADO;
        $solicitud->estado_actualizado_por = $admin->id;
        if (!empty($motivo)) {
            $solicitud->notas_admin = $motivo;
        }

        $this->registrarEvento($solicitud, self::ESTADO_CANCELADO, 'Solicitud cancelada', $admin->id, $motivo);
        $solicitud->save();
    }

    private function registrarEvento(
        SolicitudContactoScawatt $solicitud,
        string $estado,
        string $descripcion,
        ?int $adminId = null,
        ?string $nota = null,
        array $extra = []
    ): void {
        $historial = $solicitud->historial_eventos ?? [];
        $historial[] = array_merge([
            'estado' => $estado,
            'descripcion' => $descripcion,
            'admin_id' => $adminId,
            'nota' => $nota,
            'created_at' => now()->toIso8601String(),
        ], $extra);

        $solicitud->historial_eventos = $historial;
    }

    private function construirChecklistPendientes(SolicitudContactoScawatt $solicitud): array
    {
        $usuario = $solicitud->usuario;
        if (!$usuario) {
            return [];
        }

        $pendientes = [];

        if (!$usuario->cedula_verificada_at) {
            $pendientes[] = 'Subir fotos de la cedula y marcarlas como verificadas';
        }

        if (!$usuario->datos_bancarios_verificados_at) {
            $pendientes[] = 'Completar y verificar los datos bancarios';
        }

        if (empty($solicitud->mensaje)) {
            $pendientes[] = 'Agregar notas de contexto en la solicitud';
        }

        return $pendientes;
    }

    private function validarDocumentacionCompleta(SolicitudContactoScawatt $solicitud): void
    {
        $usuario = $solicitud->usuario;
        if (!$usuario) {
            throw ValidationException::withMessages([
                'usuario' => 'La solicitud no tiene un usuario asociado.',
            ]);
        }

        $pendientes = [];
        if (!$usuario->cedula_verificada_at) {
            $pendientes[] = 'Verificar cedula del usuario antes de enviar el pago.';
        }

        if (!$usuario->datos_bancarios_verificados_at) {
            $pendientes[] = 'Verificar datos bancarios del usuario antes de enviar el pago.';
        }

        if (!empty($pendientes)) {
            throw ValidationException::withMessages([
                'accion' => implode(' ', $pendientes),
            ]);
        }
    }

    private function generarScawattParaUsuario(Usuario $usuario): array
    {
        $payload = [
            'id_laravel' => $usuario->id,
            'id_wordpress' => $usuario->id_wordpress,
        ];

        $pythonUrl = config('services.scawatt_python.url', 'http://localhost:5004');
        $response = Http::timeout(45)->post("{$pythonUrl}/api/scawatts/crear", $payload);

        if (!$response->successful()) {
            throw new \RuntimeException('Error al crear ScaWatt en Python: ' . $response->body());
        }

        $data = $response->json();
        if (empty($data['success'])) {
            throw new \RuntimeException('Python retorno error: ' . ($data['error'] ?? 'Desconocido'));
        }

        $scawattSerial = $data['scawatt_serial'];
        $scawattData = $data['data'] ?? [];

        $scawatt = Scawatt::create([
            'scawatt_id' => $scawattSerial,
            'usuario_id' => $usuario->id,
            'kwh_asignados' => $scawattData['kwh_asignados'] ?? 45000,
            'valor_actual' => ($scawattData['valor'] ?? 0) + ($scawattData['valorizacion'] ?? 0),
            'valor_inicial' => $scawattData['valor'] ?? 0,
            'valorizaciones' => [$scawattData['valorizacion'] ?? 0],
            'fecha_inicio' => $scawattData['fecha_inicio'] ?? now()->toDateString(),
            'fecha_final' => $scawattData['fecha_final'] ?? now()->toDateString(),
            'estado' => 'activo',
            'tipo' => $scawattData['tipo'] ?? 'normal',
        ]);

        if (!empty($data['documentos_generados'])) {
            $this->documentManager->copyDocuments($usuario, $scawattSerial);
        }

        DB::table('trazabilidad')->insert([
            'scawatt_id' => $scawatt->id,
            'usuario_id' => $usuario->id,
            'evento' => 'creacion',
            'descripcion' => 'ScaWatt creado desde flujo de solicitud',
            'datos_antes' => null,
            'datos_despues' => json_encode([
                'valor_inicial' => $scawattData['valor'] ?? null,
                'valorizacion' => $scawattData['valorizacion'] ?? null,
                'valor_actual' => ($scawattData['valor'] ?? 0) + ($scawattData['valorizacion'] ?? 0),
            ]),
            'monto' => $scawattData['valor'] ?? null,
            'fecha_evento' => now(),
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        DB::table('propietarios')->insert([
            'scawatt_id' => $scawatt->id,
            'usuario_id' => $usuario->id,
            'fecha_inicio' => $scawattData['fecha_inicio'] ?? now()->toDateString(),
            'tipo_adquisicion' => 'compra_inicial',
            'precio_adquisicion' => $scawattData['valor'] ?? 0,
            'es_propietario_actual' => true,
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        return [
            'serial' => $scawattSerial,
            'valor_inicial' => (float) ($scawattData['valor'] ?? 0),
            'kwh' => (float) ($scawattData['kwh_asignados'] ?? 0),
        ];
    }

    private function obtenerDocumentosPrincipales(Usuario $usuario, array $resumen): array
    {
        if (empty($resumen)) {
            return [];
        }

        $ultimo = $resumen[count($resumen) - 1]['serial'] ?? null;
        if (!$ultimo) {
            return [];
        }

        $basePath = "documentos/{$usuario->id}";

        $resultado = [];
        $contratoPath = "{$basePath}/contrato_{$ultimo}.pdf";
        $certificadoPath = "{$basePath}/certificado_{$ultimo}.pdf";

        if (Storage::disk('public')->exists($contratoPath)) {
            $resultado['contrato'] = Storage::disk('public')->url($contratoPath);
        }

        if (Storage::disk('public')->exists($certificadoPath)) {
            $resultado['certificado'] = Storage::disk('public')->url($certificadoPath);
        }

        return $resultado;
    }

    private function ensureThreadToken(SolicitudContactoScawatt $solicitud): void
    {
        if (empty($solicitud->mail_thread_token)) {
            $solicitud->mail_thread_token = Str::uuid()->toString();
            $solicitud->save();
        }
    }

    private function adminRecipients()
    {
        $managers = Usuario::whereIn('estado', ['admin', 'master'])->pluck('email');
        $masters = collect(Usuario::correosMaster());

        return $managers->merge($masters)->filter()->unique();
    }
}
