<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Usuario;
use App\Models\Scawatt;
use App\Models\Trazabilidad;
use Illuminate\Support\Facades\DB;

class SyncScawattsFromJson extends Command
{
    protected $signature = 'scawatts:sync {--force : Force sync without confirmation}';
    protected $description = 'Sincroniza datos de ScaWatts desde JSONs a MySQL';

    // Rutas a los archivos JSON
    private $jsonPaths = [
        'tokens' => '/home/scalarapp/scalar-data/scawatt-server/produccion-scawatts/data_scawatts/raiz-scawatt/scawatts_tokens.json',
        'trazabilidad' => '/home/scalarapp/scalar-data/scawatt-server/produccion-scawatts/data_scawatts/raiz-scawatt/trazabilidad_scawatts.json',
        'usuarios_base' => '/home/scalarapp/public_html/scalar-api/api-scawatts/produccion/scausers/usuarios-app'
    ];

    public function handle()
    {
        $this->info('🚀 Iniciando sincronización ScaWatts → MySQL...');
        
        if (!$this->option('force')) {
            if (!$this->confirm('¿Deseas sincronizar los datos de JSON a MySQL?')) {
                $this->warn('❌ Sincronización cancelada.');
                return;
            }
        }

        DB::beginTransaction();
        
        try {
            // 1. Sincronizar usuarios
            $this->info('📥 Sincronizando usuarios...');
            $usuariosSync = $this->syncUsuarios();
            $this->info("✅ {$usuariosSync} usuarios sincronizados");

            // 2. Sincronizar ScaWatts
            $this->info('📥 Sincronizando tokens ScaWatts...');
            $scawattsSync = $this->syncScawatts();
            $this->info("✅ {$scawattsSync} ScaWatts sincronizados");

            // 3. Sincronizar trazabilidad
            $this->info('📥 Sincronizando trazabilidad...');
            $trazaSync = $this->syncTrazabilidad();
            $this->info("✅ {$trazaSync} eventos de trazabilidad sincronizados");

            DB::commit();
            
            $this->info('');
            $this->info('🎉 ¡Sincronización completada exitosamente!');
            $this->table(
                ['Concepto', 'Cantidad'],
                [
                    ['Usuarios', $usuariosSync],
                    ['ScaWatts', $scawattsSync],
                    ['Trazabilidad', $trazaSync]
                ]
            );

        } catch (\Exception $e) {
            DB::rollBack();
            $this->error('❌ Error en sincronización: ' . $e->getMessage());
            $this->error('Línea: ' . $e->getLine());
            return 1;
        }

        return 0;
    }

    private function syncUsuarios(): int
    {
        $count = 0;
        $usuariosDir = $this->jsonPaths['usuarios_base'];

        if (!is_dir($usuariosDir)) {
            $this->warn("⚠️  Directorio de usuarios no encontrado: {$usuariosDir}");
            return 0;
        }

        $carpetas = array_diff(scandir($usuariosDir), ['.', '..']);

        foreach ($carpetas as $carpeta) {
            $miScawattPath = "{$usuariosDir}/{$carpeta}/mi-scawatt.json";
            
            if (!file_exists($miScawattPath)) {
                continue;
            }

            $data = json_decode(file_get_contents($miScawattPath), true);
            
            if (!isset($data['id_user'])) {
                continue;
            }

            Usuario::updateOrCreate(
                ['id_wordpress' => $data['id_user']],
                [
                    'nombre' => $data['nombre'] ?? 'Usuario ' . $data['id_user'],
                    'email' => $data['email'] ?? "usuario{$data['id_user']}@scawatts.com",
                    'telefono' => $data['telefono'] ?? null,
                    'saldo_disponible' => $data['saldo_disponible'] ?? 0,
                    'saldo_pendiente' => $data['saldo_pendiente'] ?? 0,
                    'saldo_total' => $data['saldo_total'] ?? 0,
                    'estado' => 'activo',
                    'fecha_registro' => now()
                ]
            );

            $count++;
        }

        return $count;
    }

    private function syncScawatts(): int
    {
        $tokensPath = $this->jsonPaths['tokens'];
        
        if (!file_exists($tokensPath)) {
            $this->warn("⚠️  Archivo tokens no encontrado: {$tokensPath}");
            return 0;
        }

        $tokens = json_decode(file_get_contents($tokensPath), true);
        $count = 0;

        foreach ($tokens as $scawattId => $data) {
            // Buscar el usuario por id_wordpress
            $usuario = Usuario::where('id_wordpress', $data['id_user'])->first();
            
            if (!$usuario) {
                $this->warn("⚠️  Usuario {$data['id_user']} no encontrado para ScaWatt {$scawattId}");
                continue;
            }

            // Determinar el estado
            $estado = 'activo';
            if (isset($data['estado'])) {
                $estado = $data['estado'];
            } elseif (isset($data['fecha_final']) && strtotime($data['fecha_final']) < time()) {
                $estado = 'finalizado';
            }

            Scawatt::updateOrCreate(
                ['scawatt_id' => $scawattId],
                [
                    'usuario_id' => $usuario->id,
                    'kwh_asignados' => $data['kw_asociados'] ?? 0,
                    'valor_actual' => $data['valorizacion'] ?? $data['valor'] ?? 0,
                    'valor_inicial' => $data['valor'] ?? 0,
                    'valorizaciones' => $data['valorizaciones'] ?? null,
                    'fecha_inicio' => $data['fecha_inicio'] ?? now(),
                    'fecha_final' => $data['fecha_final'] ?? now()->addYear(),
                    'estado' => $estado,
                    'tipo' => 'normal'
                ]
            );

            $count++;
        }

        return $count;
    }

    private function syncTrazabilidad(): int
    {
        $trazaPath = $this->jsonPaths['trazabilidad'];
        
        if (!file_exists($trazaPath)) {
            $this->warn("⚠️  Archivo trazabilidad no encontrado: {$trazaPath}");
            return 0;
        }

        $eventos = json_decode(file_get_contents($trazaPath), true);
        $count = 0;

        foreach ($eventos as $evento) {
            if (!isset($evento['scawatt_id'])) {
                continue;
            }

            $scawatt = Scawatt::where('scawatt_id', $evento['scawatt_id'])->first();
            
            if (!$scawatt) {
                continue;
            }

            Trazabilidad::create([
                'scawatt_id' => $scawatt->id,
                'usuario_id' => $scawatt->usuario_id,
                'evento' => $evento['evento'] ?? 'valorizacion',
                'descripcion' => $evento['descripcion'] ?? null,
                'datos_antes' => isset($evento['valor_anterior']) ? ['valor' => $evento['valor_anterior']] : null,
                'datos_despues' => isset($evento['valor_nuevo']) ? ['valor' => $evento['valor_nuevo']] : null,
                'monto' => $evento['monto'] ?? null,
                'fecha_evento' => $evento['fecha'] ?? now()
            ]);

            $count++;
        }

        return $count;
    }
}
