<?php

use App\Models\Scawatt;
use App\Models\Usuario;
use App\Services\ScawattValuationService;
use Carbon\Carbon;
use Illuminate\Foundation\Inspiring;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

Artisan::command('inspire', function () {
    $this->comment(Inspiring::quote());
})->purpose('Display an inspiring quote');

Artisan::command('scawatt:assign {userId} {cantidad=1} {--valor=40000000} {--kwh=1000} {--granja=1}', function (int $userId, int $cantidad) {
    if ($cantidad <= 0) {
        $this->error('La cantidad debe ser mayor a 0');
        return 1;
    }

    $usuario = Usuario::find($userId);
    if (!$usuario) {
        $this->error("Usuario ID {$userId} no encontrado");
        return 1;
    }

    $valorInicial = (int) $this->option('valor');
    $kwhAsignados = (float) $this->option('kwh');
    $granjaId = (int) $this->option('granja');

    $baseDataPath = realpath(base_path('../../../scalar-data/scawatt-server/produccion-scawatts/data_scawatts'));
    if ($baseDataPath === false) {
        $this->error('No se pudo resolver la ruta base de datos de ScaWatt');
        return 1;
    }

    $tokensPath = $baseDataPath . '/raiz-scawatt/scawatts_tokens.json';
    $activeUsersPath = $baseDataPath . '/scawatts_active_user.json';

    if (!is_file($tokensPath) || !is_readable($tokensPath)) {
        $this->error("No se puede leer el archivo de tokens: {$tokensPath}");
        return 1;
    }

    if (!is_file($activeUsersPath) || !is_readable($activeUsersPath)) {
        $this->error("No se puede leer el archivo de usuarios activos: {$activeUsersPath}");
        return 1;
    }

    $tokensData = json_decode(file_get_contents($tokensPath), true) ?? [];
    $activeUsersData = json_decode(file_get_contents($activeUsersPath), true) ?? [];

    $existingSerials = Scawatt::pluck('scawatt_id')->all();
    $serialRegistry = array_fill_keys($existingSerials, true);
    foreach (array_keys($tokensData) as $serial) {
        $serialRegistry[$serial] = true;
    }

    $this->info("Usuario: {$usuario->nombre} ({$usuario->email})");
    $this->info("Creando {$cantidad} ScaWatts manualmente...\n");

    $newTokens = [];
    $newActiveEntries = [];

    try {
        DB::beginTransaction();

        for ($i = 1; $i <= $cantidad; $i++) {
            $serial = null;
            do {
                $serial = sprintf(
                    'SCA-%s-%s-%s',
                    strtoupper(Str::random(4)),
                    strtoupper(Str::random(4)),
                    strtoupper(Str::random(4))
                );
            } while (isset($serialRegistry[$serial]));
            $serialRegistry[$serial] = true;

            $scawattUuid = (string) Str::uuid();
            $fechaInicio = Carbon::now('UTC')->addSeconds($i);
            $fechaFinal = (clone $fechaInicio)->addYears(25);
            $valorizacionInicial = 0.0;
            $valorActual = $valorInicial + $valorizacionInicial;

            /** @var Scawatt $scawatt */
            $scawatt = Scawatt::create([
                'scawatt_id' => $serial,
                'usuario_id' => $usuario->id,
                'granja_id' => $granjaId,
                'kwh_asignados' => $kwhAsignados,
                'valor_actual' => $valorActual,
                'valor_inicial' => $valorInicial,
                'valorizaciones' => [$valorizacionInicial],
                'fecha_inicio' => $fechaInicio->toDateString(),
                'fecha_final' => $fechaFinal->toDateString(),
                'estado' => 'activo',
                'tipo' => 'normal',
            ]);

            DB::table('propietarios')->insert([
                'scawatt_id' => $scawatt->id,
                'usuario_id' => $usuario->id,
                'fecha_inicio' => $fechaInicio->toDateString(),
                'fecha_fin' => null,
                'tipo_adquisicion' => 'compra_inicial',
                'precio_adquisicion' => $valorInicial,
                'es_propietario_actual' => true,
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            DB::table('trazabilidad')->insert([
                'scawatt_id' => $scawatt->id,
                'usuario_id' => $usuario->id,
                'evento' => 'creacion_manual',
                'descripcion' => 'ScaWatt creado manualmente desde consola',
                'datos_antes' => null,
                'datos_despues' => json_encode([
                    'scawatt_serial' => $serial,
                    'valor_inicial' => $valorInicial,
                    'valorizacion' => $valorizacionInicial,
                ]),
                'monto' => $valorInicial,
                'fecha_evento' => now(),
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            $newTokens[] = [
                'serial' => $serial,
                'uuid' => $scawattUuid,
                'fecha_inicio' => $fechaInicio,
                'fecha_final' => $fechaFinal,
                'valor' => $valorInicial,
                'valorizacion' => $valorizacionInicial,
            ];

            $newActiveEntries[] = [
                'scawatt_id' => $serial,
                'fecha_adquisicion' => $fechaInicio->toIso8601String(),
                'valor_inicial' => $valorInicial,
                'valorizacion_actual' => $valorizacionInicial,
                'fecha_final' => $fechaFinal->toIso8601String(),
                'estado' => 'activo',
            ];

            $this->info("  ✔️  {$serial} creado");
        }

        DB::commit();
    } catch (\Throwable $e) {
        DB::rollBack();
        $this->error('Error durante la creación: ' . $e->getMessage());
        return 1;
    }

    foreach ($newTokens as $token) {
        $tokensData[$token['serial']] = [
            'scawatt_serial' => $token['serial'],
            'scawatt_id' => $token['uuid'],
            'fecha_inicio' => $token['fecha_inicio']->toIso8601String(),
            'valor' => $token['valor'],
            'fecha_final' => $token['fecha_final']->toIso8601String(),
            'valorizacion' => $token['valorizacion'],
            'estado' => 'Activo',
            'id_laravel' => $usuario->id,
            'id_wordpress' => $usuario->id_wordpress ?? null,
        ];
    }

    $userKey = (string) ($usuario->id_wordpress ?? $usuario->id);
    $activeEntry = $activeUsersData[$userKey] ?? [
        'user_id' => $usuario->id,
        'id_laravel' => $usuario->id,
        'id_wordpress' => $usuario->id_wordpress ?? null,
        'user_name' => $usuario->nombre,
        'user_email' => $usuario->email,
        'cantidad_scawatts' => 0,
        'scawatts' => [],
        'total_valor_inicial' => 0,
        'total_valorizacion_actual' => 0,
        'ultima_actualizacion' => Carbon::now('UTC')->toIso8601String(),
    ];

    foreach ($newActiveEntries as $entry) {
        $activeEntry['scawatts'][] = $entry;
        $activeEntry['cantidad_scawatts'] += 1;
        $activeEntry['total_valor_inicial'] += $entry['valor_inicial'];
        $activeEntry['total_valorizacion_actual'] += $entry['valorizacion_actual'];
    }

    $activeEntry['ultima_actualizacion'] = Carbon::now('UTC')->toIso8601String();
    $activeUsersData[$userKey] = $activeEntry;

    $encodedTokens = json_encode($tokensData, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    $encodedActive = json_encode($activeUsersData, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);

    if ($encodedTokens === false || $encodedActive === false) {
        $this->error('Error al codificar los archivos JSON actualizados');
        return 1;
    }

    file_put_contents($tokensPath, $encodedTokens);
    file_put_contents($activeUsersPath, $encodedActive);

    $this->info("\n✅ Proceso completado. {$cantidad} ScaWatts asignados a {$usuario->nombre}.");
    $this->info("Archivos actualizados:\n - {$tokensPath}\n - {$activeUsersPath}");

    return 0;
})->purpose('Asignar ScaWatts manualmente a un usuario y actualizar los archivos JSON del servidor Python');

Artisan::command('scawatt:sync-valuations {--dry-run}', function (ScawattValuationService $service) {
    $dryRun = (bool) $this->option('dry-run');
    $report = $service->syncDatabaseFromTokens(! $dryRun);

    $this->info(($dryRun ? '[DRY-RUN] ' : '') . 'Total procesados: ' . $report['processed']);
    $this->info('Actualizados: ' . $report['updated']);
    $this->info('Sin cambios: ' . $report['unchanged']);

    if (!empty($report['missing_in_db'])) {
        $mostrar = implode(', ', array_slice($report['missing_in_db'], 0, 10));
        $this->warn('Seriales sin fila en BD: ' . $mostrar . (count($report['missing_in_db']) > 10 ? '…' : ''));
    }

    if ($dryRun) {
        $this->comment('No se realizaron escrituras (dry-run activo).');
    } else {
        $this->comment('Valores sincronizados desde scawatts_tokens.json.');
    }
})->purpose('Replica scawatts_tokens.json dentro de la tabla scawatts y reinicia la base para el caché en tiempo real');
