<?php
/**
 * Script para sincronizar certificados y contratos de ScaWatts con la tabla documentos
 * 
 * Este script:
 * 1. Escanea las carpetas documentos-scawatt de cada usuario
 * 2. Registra los PDFs en la tabla documentos si no existen
 * 3. Asocia cada documento con su usuario y ScaWatt correspondiente
 * 
 * Uso: php scripts/sincronizar_documentos_scawatt.php
 */

require __DIR__ . '/../vendor/autoload.php';

$app = require_once __DIR__ . '/../bootstrap/app.php';
$app->make(Illuminate\Contracts\Console\Kernel::class)->bootstrap();

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

// Configuración
$basePathUsuarios = '/home/scalarapp/public_html/scalar-api/api-scawatts/produccion/scausers/usuarios-app';
$urlBase = 'https://escalar.app/scalar-api/api-scawatts/produccion/scausers/usuarios-app';

echo "\n";
echo "╔════════════════════════════════════════════════════════════════╗\n";
echo "║  SINCRONIZACIÓN DE DOCUMENTOS SCAWATT                          ║\n";
echo "║  " . date('Y-m-d H:i:s') . "                                        ║\n";
echo "╚════════════════════════════════════════════════════════════════╝\n\n";

$totalProcesados = 0;
$totalRegistrados = 0;
$totalExistentes = 0;
$errores = [];

try {
    // Obtener todos los usuarios con ScaWatts
    $usuarios = DB::table('usuarios')
        ->select('id', 'nombre', 'id_wordpress')
        ->whereExists(function ($query) {
            $query->select(DB::raw(1))
                ->from('scawatts')
                ->whereColumn('scawatts.usuario_id', 'usuarios.id');
        })
        ->get();

    echo "📊 Usuarios con ScaWatts encontrados: " . $usuarios->count() . "\n\n";

    foreach ($usuarios as $usuario) {
        echo "👤 Usuario #{$usuario->id}: {$usuario->nombre}\n";
        
        // Determinar carpeta del usuario (probar múltiples opciones)
        $carpetaUsuario = null;
        $posiblesCarpetas = [
            $basePathUsuarios . '/' . $usuario->id . '/documentos-scawatt', // Por ID directo
            $basePathUsuarios . '/' . $usuario->id_wordpress . '/documentos-scawatt', // Por WP ID
            $basePathUsuarios . '/laravel-' . $usuario->id . '/documentos-scawatt', // Por Laravel ID
        ];
        
        foreach ($posiblesCarpetas as $carpeta) {
            if (is_dir($carpeta)) {
                $carpetaUsuario = $carpeta;
                $folderId = basename(dirname($carpeta));
                break;
            }
        }
        
        if (!$carpetaUsuario) {
            echo "   ⚠️  Carpeta no existe en ninguna ubicación\n\n";
            continue;
        }

        // Obtener ScaWatts del usuario
        $scawatts = DB::table('scawatts')
            ->select('id', 'scawatt_id')
            ->where('usuario_id', $usuario->id)
            ->get()
            ->keyBy('scawatt_id');

        if ($scawatts->isEmpty()) {
            echo "   ℹ️  Sin ScaWatts asignados\n\n";
            continue;
        }

        // Escanear archivos PDF en la carpeta
        $archivos = glob($carpetaUsuario . '/*.pdf');
        
        foreach ($archivos as $rutaArchivo) {
            $nombreArchivo = basename($rutaArchivo);
            $totalProcesados++;

            // Extraer tipo y serial del ScaWatt del nombre del archivo
            // Formato: certificado_SCA-XXXX-XXXX-XXXX.pdf o contrato_SCA-XXXX-XXXX-XXXX.pdf
            if (preg_match('/^(certificado|contrato)_(SCA-[A-F0-9]{4}-[A-F0-9]{4}-[A-F0-9]{4})\.pdf$/i', $nombreArchivo, $matches)) {
                $tipo = $matches[1]; // certificado o contrato
                $scawattSerial = $matches[2]; // SCA-XXXX-XXXX-XXXX

                // Buscar el ScaWatt correspondiente
                $scawatt = $scawatts->get($scawattSerial);
                
                if (!$scawatt) {
                    echo "   ⚠️  ScaWatt no encontrado: {$scawattSerial} (puede pertenecer a otro usuario)\n";
                    continue;
                }

                // Verificar si ya existe en la BD
                $existente = DB::table('documentos')
                    ->where('usuario_id', $usuario->id)
                    ->where('scawatt_id', $scawatt->id)
                    ->where('tipo', $tipo . '_scawatt')
                    ->where('nombre_archivo', $nombreArchivo)
                    ->exists();

                if ($existente) {
                    $totalExistentes++;
                    echo "   ✓ Ya registrado: {$nombreArchivo}\n";
                    continue;
                }

                // Obtener tamaño del archivo
                $tamañoBytes = filesize($rutaArchivo);

                // Construir URL pública
                $urlPublica = $urlBase . '/' . $folderId . '/documentos-scawatt/' . rawurlencode($nombreArchivo);

                // Registrar en la BD
                try {
                    DB::table('documentos')->insert([
                        'usuario_id' => $usuario->id,
                        'scawatt_id' => $scawatt->id,
                        'tipo' => $tipo . '_scawatt',
                        'nombre_archivo' => $nombreArchivo,
                        'ruta_archivo' => $rutaArchivo,
                        'url_publica' => $urlPublica,
                        'tamaño_bytes' => $tamañoBytes,
                        'fecha_generacion' => date('Y-m-d H:i:s', filemtime($rutaArchivo)),
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]);

                    $totalRegistrados++;
                    echo "   ✅ Registrado: {$nombreArchivo} ({$tipo} para {$scawattSerial})\n";

                    Log::info('Documento ScaWatt sincronizado', [
                        'usuario_id' => $usuario->id,
                        'scawatt_id' => $scawatt->id,
                        'tipo' => $tipo,
                        'archivo' => $nombreArchivo
                    ]);

                } catch (\Exception $e) {
                    $errores[] = "Error registrando {$nombreArchivo}: " . $e->getMessage();
                    echo "   ❌ Error: {$e->getMessage()}\n";
                }
            } else {
                echo "   ⚠️  Formato de archivo no reconocido: {$nombreArchivo}\n";
            }
        }

        echo "\n";
    }

    // Resumen final
    echo "╔════════════════════════════════════════════════════════════════╗\n";
    echo "║  RESUMEN DE SINCRONIZACIÓN                                     ║\n";
    echo "╚════════════════════════════════════════════════════════════════╝\n";
    echo "📊 Archivos procesados:  {$totalProcesados}\n";
    echo "✅ Nuevos registrados:   {$totalRegistrados}\n";
    echo "ℹ️  Ya existentes:       {$totalExistentes}\n";
    echo "❌ Errores:              " . count($errores) . "\n\n";

    if (!empty($errores)) {
        echo "⚠️  ERRORES ENCONTRADOS:\n";
        foreach ($errores as $error) {
            echo "   • {$error}\n";
        }
        echo "\n";
    }

    if ($totalRegistrados > 0) {
        echo "✨ Sincronización completada exitosamente\n";
        echo "💡 Los documentos ahora aparecerán en el portal admin\n\n";
    } else {
        echo "ℹ️  No se encontraron documentos nuevos para sincronizar\n\n";
    }

} catch (\Exception $e) {
    echo "\n❌ ERROR FATAL: " . $e->getMessage() . "\n";
    echo "Stack trace:\n" . $e->getTraceAsString() . "\n\n";
    exit(1);
}

exit(0);
